Program nonorthoSA
!
! This program calculates the accessible surface area by using a simple Monte
! Carlo integration for both orthorhombic and non-orthorhombic unit cells. 
! To run the program the following files are needed:
! 1) an input file containing the name of the file containing the diameters of 
!    the atoms, the name of the file containing the coordinates of the 
!    framework atoms, the diameter of the probe, the number of random trials
!    around each framework atom, the lenghts of the unit cells (a, b, c), the
!    cell angles (alpha, beta and gamma) and the crystal density.
! 2) A file containing the diameters of the framework atoms. The program is
!    written to read in an xyz file. Please note that the coordinates of all
!    framework atoms need to be specified (P1) as the program does not handle
!    symmetry operations.
!
! Things that you might have to change:
! - The maximum number of framework atoms 5000 at the moment. Change the value
!   of max_no if your unit cells contains more atoms.
! - Atom types/names must be defined both in your atom input file and here. Atom
!   name definitions start at line 184. Add new atom names using the same format.
!
! Changes to the code
! 
! 9/3/2009   Tina Duren
!            fixed a bug that wrote the error message to a file rather than
!            the screen if an atom type was not specified in the diameter list
!            Thanks to Kenji Sumida from the University of Berkeley for 
!            pointing this out.
! 11/8/2010  Introduced allocatable arrays to avoid problems with array size
! 25/6/2014  Matt Lennox
!            Updated variable declarations in the function ran0 from F77 
!            to be compatible with F95 NAG compilers
! 28/7/2016  Matt Lennox
!            Completely replanced function with random number generator as
!            previous one was given weird results for some unknown reason
! 28/7/2016  Tina Duren
!            Updated array allocation so that all arrays are now allocatable
!            Changed the way how atom types are read in so that all types are 
!            now defined in atomfile and no more recompilation is required if 
!            new atom types are added.
!
!  	
Use matrix
Use defaults
        
Implicit None

!
! Maximum number of atoms in your unit cell. Increase if your unit cell 
! contains more atoms.
!

Character(len = 3)   :: atom
Character(len = 2)   :: symbol
Character(len = 10), Allocatable  :: atomtype(:)
Character(len = 10), Allocatable :: atomname(:)
Character(len = 100) :: atom_file, coord_file, output_file

Type(MatrixType)     :: lij, lijinv

Real(kind=RDbl), Allocatable      :: sigmatype(:), atomsigma(:)
Real(kind=RDbl), Allocatable      :: x(:), y(:), z(:)
Real(kind=RDbl), Allocatable      :: xnew(:), ynew(:), znew(:)
Real(kind=RDbl)      :: alpha, beta, gamma, alpha_r, beta_r, gamma_r
Real(kind=RDbl)      :: a, b, c, ai, bi, ci, aj, bj, cj, ak, bk, ck
Real(kind=RDbl)      :: xpoint, ypoint, zpoint, xpointnew, ypointnew, zpointnew
Real(kind=RDbl)      :: phi, costheta, theta
Real(kind=RDbl)      :: rho_crys, dprobe
Real(kind=RDbl)      :: dx, dy, dz, dx_trans, dy_trans, dz_trans, dist2 
Real(kind=RDbl)      :: sjreal, stotal, sfrac, uc_volume, stotalreduced
Real(kind=RDbl)      :: ran_num1, ran_num2, trash
Real(kind=RDBL)      :: ran0

Integer              :: seed, N, ntypes, i, j, k, ncount, Nsample
Integer              :: ierror

Logical              :: match, deny, in_list
!
! Seed for the random number generator, change if you want to start from a 
! different random number
!  	
seed = -52907
!
! to properly initialise the random number generator, a negative seed is
! needed
!
If (seed> 0) seed = -1 * seed
!
! Read the data from the input file
!
Read(*,('(A)')) atom_file      ! file containing the diameters of atoms
Read(*,('(A)')) coord_file     ! file containing the cartesian coordinates
Read(*,*) dprobe                 ! diameter of probe in A
Read(*,*) Nsample                ! Number of trials per framework atom 
Read(*,*) a, b, c                ! Cell parameters in A
Read(*,*) alpha, beta, gamma     ! cell angles
Read(*,*) rho_crys               ! density of crystal in g / cm3

               
alpha_r = alpha*degtorad
beta_r = beta*degtorad
gamma_r = gamma*degtorad

ai = a
aj = 0
ak = 0
bi = b*cos(gamma_r)
bj = b*sin(gamma_r)
bk = 0
ci = c*cos(beta_r)
cj = (b*c*cos(alpha_r)-bi*ci)/bj
ck = sqrt(c**2-ci**2-cj**2)
    
lij%comp = 0
lij%comp(1,1) = ai
lij%comp(1,2) = bi
lij%comp(1,3) = ci
lij%comp(2,1) = aj
lij%comp(2,2) = bj
lij%comp(2,3) = cj
lij%comp(3,1) = ak
lij%comp(3,2) = bk
lij%comp(3,3) = ck

lijinv=matrix_inverse(lij)

!
! Open the files and read the data
!
Open(10, file=atom_file, status='old', IOSTAT = ierror)
If (ierror /= 0) Then
    Write(*,*) 'Error opening file: ',atom_file
    Write(*,*) 'Make sure it exists or check the spelling of the filename'
END IF

    
ntypes = 0

Do
   Read(10,*) atom
   IF(Trim(atom) == 'EOF') EXIT
   ntypes = ntypes + 1
END DO

Rewind(10)
Allocate(atomtype(1:ntypes), sigmatype(1:ntypes))
Do i = 1, ntypes
   Read(10,*) atomtype(i), sigmatype(i)
End do
!
! Openeing and reading the coord file
!
Open(20, file=coord_file, status='old', IOSTAT = ierror)
If (ierror /= 0) Then
    Write(*,*) 'Error opening file: ',coord_file
    Write(*,*) 'Make sure it exists or check the spelling of the filename'
END IF
!
! The first line contains the number of framework atoms
!
Read(20,*) N
!
! Allocate all arrays for the framework atoms
!
Allocate(atomname(1:N), atomsigma(1:N))
Allocate(x(1:N), y(1:N), z(1:N))
Allocate(xnew(1:N), ynew(1:N), znew(1:N))
!
! Skip the blank line of the xyz file
!
Read(20,*)   
Do i=1, N
!
! The format of the coordinate file corresponds to an xyz file as exported 
! from Diamond.
! x(i), y(i), z(i): x,y,z coordinates in Angstrom
! atomtype: chemical symbol of framework atom.
!
! Change the following read statement if you want to use a different
! file format. But ensure that you end up with the coordinates in A and
! the name of the framework atoms!


   Read(20,*) atomname(i), x(i), y(i), z(i)

   atomname(i) = TRIM(atomname(i))
!
! Check if the atom is in the list with the diameters
!
   in_list = .FALSE.
   Do k = 1, ntypes
      IF(atomname(i) == atomtype(k)) then
         in_list = .TRUE.
         EXIT
      END IF
   END DO
   IF (.NOT. in_list) THEN
      Write(*,'(A,I5,1X,A)') ' Could not find match for atom: ', i, atomname(i)
      Write(*,*) 'The name is either read in incorrectly or does not exist'
      Write(*,*) 'Amend the list of available atoms in ',atom_file
      STOP
   END IF    

    
! Transform molecules coordinates
   xnew(i) = x(i)*lijinv%comp(1,1) + y(i)*lijinv%comp(1,2) + &
             z(i)*lijinv%comp(1,3)
   ynew(i) = x(i)*lijinv%comp(2,1) + y(i)*lijinv%comp(2,2) + &
             z(i)*lijinv%comp(2,3)
   znew(i) = x(i)*lijinv%comp(3,1) + y(i)*lijinv%comp(3,2) + &
             z(i)*lijinv%comp(3,3) 
   xnew(i) = a*xnew(i)
   ynew(i) = b*ynew(i)
   znew(i) = c*znew(i)
                
! Translate the transformed coordinates so they lie in the box a,b,c
      
    If(xnew(i)<0.0) xnew(i) = xnew(i) + a
    If(xnew(i)>=a) xnew(i) = xnew(i) - a
    If(ynew(i)<0.0) ynew(i) = ynew(i) + b
    If(ynew(i)>=b) ynew(i) = ynew(i) - b
    If(znew(i)<0.0) znew(i) = znew(i) + c
    If(znew(i)>=c) znew(i) = znew(i) - c
    
! Match sigmas with coordinates            
                
    atomname(i)=Trim(atomname(i))
    
    match=.False.
    
    Do j=1, ntypes
       If(atomname(i)==atomtype(j)) Then
          atomsigma(i)=sigmatype(j)+dprobe
          match=.True.
          Exit
       End If
    End Do
    
 End Do

Write(*,*)
Write(*,*) 'Calculating the accessible surface area for the following input parameters'
Write(*,*) '-------------------------------------------------------------------------'
Write(*,*) 'File with framework coordinates: ',TRIM(coord_file)
Write(*,*) 'File with atom diameters: ', TRIM(atom_file)
Write(*,'(A,F12.3)') ' Probe diameter in A: ',dprobe

! Main sampling cycle

stotal=0.0

Do i=1, N ! Loop over all framework atoms

    ncount=0

    Do j=1, Nsample ! Number of trial positions around each framework atom
!                 
! Generate random vector of length 1
! First generate phi 0:+2pi
!
       phi = ran0(seed)*twopi
!
!
! Generate cosTheta -1:1 to allow for even distribution of random vectors
! on the unit sphere.See http://mathworld.wolfram.com/SpherePointPicking.html
! for further explanations
!
       costheta = 1. - ran0(seed) * 2.0
       theta = Acos(costheta)
       xpoint = sin(theta)*cos(phi)
       ypoint = sin(theta)*sin(phi)
       zpoint = costheta

    
! Make this vector of (sigma+probe_diameter)/2.0 length

       xpoint=xpoint*atomsigma(i)/2.0
       ypoint=ypoint*atomsigma(i)/2.0
       zpoint=zpoint*atomsigma(i)/2.0

! Transform random vector
                
       xpointnew = xpoint*lijinv%comp(1,1) + ypoint*lijinv%comp(1,2) &
                   + zpoint*lijinv%comp(1,3)
       ypointnew = xpoint*lijinv%comp(2,1) + ypoint*lijinv%comp(2,2) &
                   + zpoint*lijinv%comp(2,3)
       zpointnew = xpoint*lijinv%comp(3,1) + ypoint*lijinv%comp(3,2) &
                   + zpoint*lijinv%comp(3,3) 
       xpointnew = a*xpointnew
       ypointnew = b*ypointnew
       zpointnew = c*zpointnew
                                
! Translate the center of coordinate to the particle i center and apply PBC

       xpointnew = xpointnew + xnew(i)
       ypointnew = ypointnew + ynew(i)
       zpointnew = zpointnew + znew(i)

       If(xpointnew < 0.0) xpointnew = xpointnew + a
       If(xpointnew >= a) xpointnew = xpointnew - a
       If(ypointnew < 0.0) ypointnew = ypointnew + b
       If(ypointnew >= b) ypointnew = ypointnew - b
       If(zpointnew < 0.0) zpointnew = zpointnew + c
       If(zpointnew >= c) zpointnew = zpointnew - c

! Now we check for overlap 

       deny=.False.

       Do k=1,N
          if(k==i) cycle
          dx = xpointnew - xnew(k)
          dx = dx - a*int(2.0*dx/a)
          dy = ypointnew - ynew(k)
          dy = dy - b*int(2.0*dy/b)
          dz = zpointnew-znew(k)
          dz = dz-c*int(2.0*dz/c)
                                
          dx = dx / a
          dy = dy / b
          dz = dz / c
        
          dx_trans = dx*lij%comp(1,1) + dy*lij%comp(1,2) + dz*lij%comp(1,3)
          dy_trans = dx*lij%comp(2,1) + dy*lij%comp(2,2) + dz*lij%comp(2,3)
          dz_trans = dx*lij%comp(3,1) + dy*lij%comp(3,2) + dz*lij%comp(3,3)
                                
          dist2=dx_trans*dx_trans+dy_trans*dy_trans+dz_trans*dz_trans

          If(sqrt(dist2)<0.999*atomsigma(k)/2.0) then
             deny=.True.
             Exit
          End If
       End Do

       If(deny) Cycle
   
       ncount=ncount+1

    End Do
                
                
! Fraction of the accessible surface area for sphere i

    sfrac=Real(ncount)/Real(Nsample)

! Surface area for sphere i in real units (A^2)
    
    sjreal=pi*atomsigma(i)*atomsigma(i)*sfrac
    stotal=stotal+sjreal
    
End Do

! Converting stotal on Surface per Volume
! Unit volume calculated from the absolute value of the determinate of the 3x3 
! matrix containing the vectors defining the unit cell. 
! See e.g. Marsden, et al. "Basic Multivariable Calculus" 1993 pg. 53

uc_volume=abs(ai*(bj*ck-bk*cj)-aj*(bi*ck-bk*ci)+ak*(bi*cj-bj*ci))
stotalreduced=stotal/uc_volume*1.E4

! Report results
     

Write(*,'(A,F12.2)') ' Total surface area in Angstroms^2: ', stotal
Write(*,'(A,F12.2)') ' Total surface area per volume in m^2/cm^3: ', &
                      stotalreduced
Write(*,'(A,F12.2)') ' Total surface area per volume in m^2/g: ', &
                            stotalreduced / rho_crys
	

End Program NonorthoSA

!----------------FUNCTIONS-------------------------------------

FUNCTION ran0(idum)

!The ran1 function fro WH PRess et al 'Numerical Recipes in F77'

Use defaults, Only: RDbl
INTEGER :: idum
INTEGER, Parameter :: IA=16807, IM=2147483647, IQ=127773, IR=2836
INTEGER, Parameter :: NTAB=32, NDIV=1+(IM-1)/NTAB
REAL(kind=RDbl), Parameter :: EPS=1.2e-7, RNMX=1.-EPS, AM=1./IM
REAL(kind=RDbl) :: ran0
INTEGER :: j,k
INTEGER, save :: iy=0
INTEGER, DIMENSION(NTAB), save :: iv=0

!Initialize things. First time if idum < 0 or iy=0

If (idum .lt. 0 .or. iy .eq. 0) Then
 idum=max(-idum,1)
  do j=NTAB+8, 1, -1
   k=idum/IQ
   idum=IA*(idum-k*IQ)-IR*k
   if (idum .lt. 0) idum=idum+IM
   if (j .le. NTAB) iv(j)=idum
 Enddo

 iy=iv(1)

Endif

!Not intializing

 k=idum/IQ
 idum=IA*(idum-k*IQ)-IR*k
 if (idum .lt. 0) idum=idum+IM
 j=1+iy/NDIV
 iy=iv(j)
 iv(j)=idum
 ran0=min(AM*iy,RNMX)
 return

END FUNCTION ran0





      

       


