class Node
{
  int Number;
  float XCoord, YCoord;
  boolean XFixed, YFixed;
  float XLoad, YLoad;
  float XDisplacement, YDisplacement;
  
  Node (int n, float x, float y, boolean xfix, boolean yfix, float xld, float yld)
  {
    Number = n;
    XCoord = x;
    YCoord = y;
    XFixed = xfix;
    YFixed = yfix;
    XLoad = xld;
    YLoad = yld;
    XDisplacement = 0.0;
    YDisplacement = 0.0;
  }
}

class Line
{
  int Number;
  Node NodeStart, NodeEnd;
  float Area, Force, XLength, YLength;
  
  Line (int n, Node start, Node end, float a)
  {
    Number = n;
    NodeStart = start;
    NodeEnd = end;
    Area = a;
    Force = 0.0;
    XLength = NodeEnd.XCoord-NodeStart.XCoord;
    YLength = NodeEnd.YCoord-NodeStart.YCoord;
  }
}

float Mouse_X, Mouse_Y, Mouse_OldX, Mouse_OldY;
float RotateX, RotateY, PanX, PanY, Zoom = 100.0;
float RotateX_Drag, RotateY_Drag, PanX_Drag, PanY_Drag, Zoom_Drag;
int NUM_NODES = 0, NUM_LINES = 0;
float YoungsModulus = 205000.0;
int Animation_Frame, Animation_Increment = -1, Animation_Limit = 10;

Node[] arrayNodes;
Line[] arrayLines;
float[][] stiffness_matrix;
float[][] stiffness_matrix_inverse;

Node FindNode(int num)
{
  Node aNode;
  
  for (int count=0; count<NUM_NODES; count++)
  {
    aNode = arrayNodes[count];
    if (aNode.Number==num) return aNode;
  }
  
  println("WARNING : Not Found Node#"+num);
  return null;
}


void setup() 
{ 
  size(800, 520, P3D);
  ortho(0, width, height, 0);
  colorMode(RGB, 1);
  stroke(0.9, 0.9, 0.9);
  frameRate(Animation_Limit);
  PanX = width/2 + 100;
  //PanY = height/2;
  
  FileOpenNodes();
  if (NUM_NODES>0)
  {
    OutputNodes();
    FileOpenLines();
  }
  if (NUM_LINES>0)
  {
    BuildStiffnessMatrix();
    println("\n\nRAW MATRIX :");
    OutputMatrix();
    ApplyBoundaryConditions();
    println("\n\nRAW MATRIX : PLUS BCs");
    OutputMatrix();
    InvertStiffnessMatrix();
    println("\n\nINVERSE MATRIX :");
    OutputMatrix();
    SolveDisplacements();
    SolveForces();
    OutputNodes();
    OutputLines();
  }

  loop();
} 

void FileOpenNodes()
{
  String buffer[], lines[] = loadStrings("Nodes.txt");
  float x, y, xload, yload;
  int xfix, yfix;
  int num;
  Node aNode;
  
  if (lines==null)
  {
    status("Nodes File Not Found");
    return;
  }
  
  NUM_NODES = lines.length;
  arrayNodes = new Node[NUM_NODES];
  
  for (int count=0; count<NUM_NODES; count++)
  {
    buffer = splitTokens(lines[count], " ");
    if (buffer.length>0)
    {
      num = int(buffer[0]);
      x = float(buffer[1]);
      y = float(buffer[2]);
      xfix = int(buffer[3]);
      yfix = int(buffer[4]);
      xload = float(buffer[5]);
      yload = float(buffer[6]);
    
      arrayNodes[count] = new Node(num, x, y, (xfix==1), (yfix==1), xload, yload);
    }
  }
}
 
void FileOpenLines()
{
  String buffer[], lines[] = loadStrings("Lines.txt");
  int num, start, end;
  float area;
  Node startNode, endNode;
  Line aLine;
  
  if (lines==null) return;
  
  NUM_LINES = lines.length;
  arrayLines = new Line[NUM_LINES];
  
  for (int count=0; count<NUM_LINES; count++)
  {
    buffer = splitTokens(lines[count], " ");
    if (buffer.length>0)
    {
      num = int(buffer[0]);
      start = int(buffer[1]);
      end = int(buffer[2]);
      area = int(buffer[3]);
      
      startNode = FindNode(start);
      endNode = FindNode(end);
    
      if ((startNode!=null)&&(endNode!=null))
      {
        arrayLines[count] = new Line(num, startNode, endNode, area);
      }
    }
  }
}
 
void BuildStiffnessMatrix()
{
  int start_dof, end_dof;
  float member_length, x_length, y_length, xx_stiffness, yy_stiffness, xy_stiffness, EA_over_Lcubed;
  Line aLine;
  
  stiffness_matrix = new float[2*NUM_NODES][2*NUM_NODES];
  stiffness_matrix_inverse = new float[2*NUM_NODES][2*NUM_NODES];

  //  Set inverse to Identity Martix;
  for (int row=0; row<2*NUM_NODES; row++)
  {
    for (int col=0; col<2*NUM_NODES; col++) stiffness_matrix_inverse[row][col] = 0.0;
    stiffness_matrix_inverse[row][row] = 1.0;
  }
  
  for (int count=0; count<NUM_LINES; count++)
  {
    aLine = arrayLines[count];
    member_length = sqrt( aLine.XLength*aLine.XLength + aLine.YLength*aLine.YLength );
    
    EA_over_Lcubed = YoungsModulus*aLine.Area/(member_length*member_length*member_length);
    xx_stiffness = EA_over_Lcubed*aLine.XLength*aLine.XLength;
    yy_stiffness = EA_over_Lcubed*aLine.YLength*aLine.YLength;
    xy_stiffness = EA_over_Lcubed*aLine.XLength*aLine.YLength;
    
    start_dof = 2*(aLine.NodeStart.Number-1);
    end_dof = 2*(aLine.NodeEnd.Number-1);
    
    stiffness_matrix[start_dof][start_dof] += xx_stiffness;
    stiffness_matrix[end_dof][end_dof] += xx_stiffness;
    stiffness_matrix[start_dof][end_dof] -= xx_stiffness;
    stiffness_matrix[end_dof][start_dof] -= xx_stiffness;
    
    stiffness_matrix[start_dof+1][start_dof+1] += yy_stiffness;
    stiffness_matrix[end_dof+1][end_dof+1] += yy_stiffness;
    stiffness_matrix[start_dof+1][end_dof+1] -= yy_stiffness;
    stiffness_matrix[end_dof+1][start_dof+1] -= yy_stiffness;
    
    stiffness_matrix[start_dof+1][start_dof] += xy_stiffness;
    stiffness_matrix[end_dof+1][end_dof] += xy_stiffness;
    stiffness_matrix[start_dof+1][end_dof] -= xy_stiffness;
    stiffness_matrix[end_dof+1][start_dof] -= xy_stiffness;
    
    stiffness_matrix[start_dof][start_dof+1] += xy_stiffness;
    stiffness_matrix[end_dof][end_dof+1] += xy_stiffness;
    stiffness_matrix[start_dof][end_dof+1] -= xy_stiffness;
    stiffness_matrix[end_dof][start_dof+1] -= xy_stiffness;
  }
}

void ApplyBoundaryConditions()
{
  int x_dof, y_dof;
  Node aNode;
  
  for (int count=0; count<NUM_NODES; count++)
  {
    aNode = arrayNodes[count];
    x_dof = 2*(aNode.Number-1);
    y_dof = x_dof+1;
    
    if (aNode.XFixed)
    {
      for (int dof_count=0; dof_count<NUM_NODES*2; dof_count++)
      {
        stiffness_matrix[x_dof][dof_count] = 0.0;
        stiffness_matrix[dof_count][x_dof] = 0.0;
      }
      stiffness_matrix[x_dof][x_dof] = 1.0;
      
      aNode.XLoad = 0.0;
    }
    
    if (aNode.YFixed)
    {
      for (int dof_count=0; dof_count<NUM_NODES*2; dof_count++)
      {
        stiffness_matrix[y_dof][dof_count] = 0.0;
        stiffness_matrix[dof_count][y_dof] = 0.0;
      }
      stiffness_matrix[y_dof][y_dof] = 1.0;
      
      aNode.YLoad = 0.0;
    }
  }
}


void InvertStiffnessMatrix()
{
  int current_row, row, col, count, biggest;
  float pivot_val;
    
  for (current_row=0; current_row<2*NUM_NODES; current_row++)
  {
    //  Find Biggest Pivot
    biggest = current_row;
    for (row=current_row+1; row<2*NUM_NODES; row++) if (stiffness_matrix[row][current_row]>stiffness_matrix[biggest][current_row]) biggest = row;
    
    if (biggest!=current_row)
    {
      //  Swap current_row for one with biggest pivot
      for (col=0; col<2*NUM_NODES; col++)
      {
        pivot_val = stiffness_matrix[current_row][col];
        stiffness_matrix[current_row][col] = stiffness_matrix[biggest][col];
        stiffness_matrix[biggest][col] = pivot_val;
        
        pivot_val = stiffness_matrix_inverse[current_row][col];
        stiffness_matrix_inverse[current_row][col] = stiffness_matrix_inverse[biggest][col];
        stiffness_matrix_inverse[biggest][col] = pivot_val;
      }
    }
    
    //  Divide row through by pivot
    pivot_val = stiffness_matrix[current_row][current_row];
    for (col=0; col<2*NUM_NODES; col++)
    {
      stiffness_matrix[current_row][col] /= pivot_val;
      stiffness_matrix_inverse[current_row][col] /= pivot_val;
    }
    
    //  Subtract multiple of current row from all others below
    for (row=current_row+1; row<2*NUM_NODES; row++)
    {
      pivot_val = stiffness_matrix[row][current_row];
      for (col=0; col<2*NUM_NODES; col++)
      {
        stiffness_matrix[row][col] -= pivot_val*stiffness_matrix[current_row][col];
        stiffness_matrix_inverse[row][col] -= pivot_val*stiffness_matrix_inverse[current_row][col];
      }
    }
    
  }
    
  //  Matrix is now Upper Triangular
  for (current_row=2*NUM_NODES-2; current_row>=0; current_row--)
  {
    for (row=current_row+1; row<2*NUM_NODES; row++)
    {
      pivot_val = stiffness_matrix[current_row][row];
      for (col=0; col<2*NUM_NODES; col++)
      {
        stiffness_matrix[current_row][col] -= pivot_val*stiffness_matrix[row][col];
        stiffness_matrix_inverse[current_row][col] -= pivot_val*stiffness_matrix_inverse[row][col];
      }      
    }
  }
  
}

void SolveDisplacements()
{
  int x_dof, y_dof, x_count, y_count;
  Node aNode;
  float[] load_vector = new float[2*NUM_NODES] ;
  
  for (x_count=0; x_count<NUM_NODES; x_count++)
  {
    aNode = arrayNodes[x_count];
    x_dof = 2*(aNode.Number-1);
    y_dof = x_dof+1;
    
    load_vector[x_dof] = aNode.XLoad;
    load_vector[y_dof] = aNode.YLoad;
  }
  
  for (x_count=0; x_count<NUM_NODES; x_count++)
  {
    aNode = arrayNodes[x_count];
    x_dof = 2*(aNode.Number-1);
    y_dof = x_dof+1;
    
    aNode.XDisplacement = 0.0;
    aNode.YDisplacement = 0.0;
    for (y_count=0; y_count<2*NUM_NODES; y_count++)
    {
      aNode.XDisplacement += stiffness_matrix_inverse[x_dof][y_count]*load_vector[y_count];
      aNode.YDisplacement += stiffness_matrix_inverse[y_dof][y_count]*load_vector[y_count];
    }
  }
  
}

void SolveForces()
{
  int x_dof, y_dof, x_count, y_count;
  Line aLine;
  
  for (int count=0; count<NUM_LINES; count++)
  {
    aLine = arrayLines[count];
    aLine.Force = YoungsModulus*aLine.Area/(aLine.XLength*aLine.XLength+aLine.YLength*aLine.YLength);
    aLine.Force *= aLine.XLength*(aLine.NodeEnd.XDisplacement-aLine.NodeStart.XDisplacement) + aLine.YLength*(aLine.NodeEnd.YDisplacement-aLine.NodeStart.YDisplacement);
  }
}


void OutputNodes()
{
  Node aNode;
  
  //  Output Nodes for Debug
  println("\n\nNODES :");
  for (int count=0; count<NUM_NODES; count++)
  {
    aNode = arrayNodes[count];
    
    println("\nNumber       = "+aNode.Number);
    println("Coord        = ("+aNode.XCoord+","+aNode.YCoord+")");
    println("Fixity       = ("+aNode.XFixed+","+aNode.YFixed+")");
    println("Load         = ("+aNode.XLoad+","+aNode.YLoad+")");
    println("Displacement = ("+aNode.XDisplacement+","+aNode.YDisplacement+")");
  }
}
  
void OutputLines()
{
  Line aLine;
  
  //  Output Nodes for Debug
  println("\n\nLINES :");
  for (int count=0; count<NUM_LINES; count++)
  {
    aLine = arrayLines[count];
    
    println("\nNumber       = "+aLine.Number);
    println("Nodes        = "+aLine.NodeStart.Number+" --> "+aLine.NodeEnd.Number);
    println("Length       = ("+aLine.XLength+","+aLine.YLength+")");
    println("Force        = "+aLine.Force);
  }
}
  
void OutputMatrix()
{
  int row, col;
  
  //  Output Matrix for Debug
  print("\n");
  for (row=0; row<2*NUM_NODES; row++)
  {
    for (col=0; col<2*NUM_NODES; col++) print("  "+nfs(stiffness_matrix[row][col], 5, 1));
    print("\n");
  }
  
  //  Output Matrix for Debug
  print("\n");
  for (row=0; row<2*NUM_NODES; row++)
  {
    for (col=0; col<2*NUM_NODES; col++) print("  "+nfs(stiffness_matrix_inverse[row][col], 5, 1));
    print("\n");
  }
}
  


    
void mousePressed()
{
  Mouse_OldX = mouseX;
  Mouse_OldY = mouseY;
}
void mouseDragged()
{
  Mouse_X = mouseX;
  Mouse_Y = mouseY;
  
  switch (mouseButton)
  {
    case LEFT :
      RotateX_Drag = PI*(Mouse_X-Mouse_OldX)/width;
      RotateY_Drag = -PI*(Mouse_Y-Mouse_OldY)/height;
      break;
    case RIGHT :
      PanX_Drag = 100*(Mouse_X-Mouse_OldX)/width;
      PanY_Drag = -100*(Mouse_Y-Mouse_OldY)/height;
      break;
    case CENTER :
      Zoom_Drag = -100*(Mouse_Y-Mouse_OldY)/height;
      break;
  }
}
void mouseReleased()
{
  switch (mouseButton)
  {
    case LEFT :
      RotateX += RotateX_Drag;
      RotateY += RotateY_Drag;
      RotateX_Drag = 0.0;
      RotateY_Drag = 0.0;
      break;
    case RIGHT :
      PanX += PanX_Drag;
      PanY += PanY_Drag;
      PanX_Drag = 0.0;
      PanY_Drag = 0.0;
      break;
    case CENTER :
      Zoom += Zoom_Drag;
      Zoom_Drag = 0.0;
      break;
  }
}


void draw() 
{ 
  background(0.0, 0.0, 0.0);
  lights();
  
  pushMatrix(); 
 
  translate(PanX+PanX_Drag, PanY+PanY_Drag, 0); 
  rotateY(-RotateX-RotateX_Drag); 
  rotateX(RotateY+RotateY_Drag); 
  if (Zoom+Zoom_Drag>1.0) scale(Zoom+Zoom_Drag);
  
  DrawLines();
  DrawDisplacements();
  DrawNodes();
  
  popMatrix(); 
  
  //if (Animation_Frame==Animation_Limit) save("c:\\Temp\\image.bmp");
} 

void DrawNodes()
{
  Node aNode;
  
  strokeWeight(10);
  beginShape(POINTS);
    for (int count=0; count<NUM_NODES; count++)
    {
      aNode = arrayNodes[count];
      stroke(255*int(aNode.XFixed), 255*int(aNode.YFixed), 255);  //  White
      vertex(aNode.XCoord, aNode.YCoord, 0.0);
    }
  endShape();
}

void DrawLines()
{
  Line aLine;
  
  stroke(255, 255, 255);  //  White
  strokeWeight(2);
  beginShape(LINES);
    for (int count=0; count<NUM_LINES; count++)
    {
      aLine = arrayLines[count];
      vertex(aLine.NodeStart.XCoord, aLine.NodeStart.YCoord, 0.0);
      vertex(aLine.NodeEnd.XCoord, aLine.NodeEnd.YCoord, 0.0);
    }
  endShape();
}

void DrawDisplacements()
{
  Line aLine;
  float anim_scale;
  
  if ((Animation_Frame==Animation_Limit)||(Animation_Frame==0)) Animation_Increment = -Animation_Increment;
  Animation_Frame += Animation_Increment;
  anim_scale = 10.0*float(Animation_Frame)/Animation_Limit;
  
  stroke(255, 0, 0);  //  Red
  strokeWeight(1);
  beginShape(LINES);
    for (int count=0; count<NUM_LINES; count++)
    {
      aLine = arrayLines[count];
      vertex(aLine.NodeStart.XCoord+aLine.NodeStart.XDisplacement*anim_scale, aLine.NodeStart.YCoord+aLine.NodeStart.YDisplacement*anim_scale, 0.0);
      vertex(aLine.NodeEnd.XCoord+aLine.NodeEnd.XDisplacement*anim_scale, aLine.NodeEnd.YCoord+aLine.NodeEnd.YDisplacement*anim_scale, 0.0);
    }
  endShape();

}

