import processing.core.*; 
import processing.data.*; 
import processing.event.*; 
import processing.opengl.*; 

import java.util.HashMap; 
import java.util.ArrayList; 
import java.io.File; 
import java.io.BufferedReader; 
import java.io.PrintWriter; 
import java.io.InputStream; 
import java.io.OutputStream; 
import java.io.IOException; 

public class HangingChain extends PApplet {

int NUM_NODES = 11;
float SlackFactor = 0.5f;
float Gravity = 0.0098f;
float DampingFactor = 0.95f;
float MassFactor = 1.0f;
float EA = 200.0f;
boolean ParabolaInstead = false;
boolean GravityOn = false;

class Node
{
    float XCoord, YCoord;
    boolean XFixed, YFixed;
    float XForce, YForce, XVelocity, YVelocity;
    float Mass, Radius;

    Node (float x, float y, boolean xfix, boolean yfix)
    {
        XCoord = x;
        YCoord = y;
        XFixed = xfix;
        YFixed = yfix;
        XForce = 0.0f;
        YForce = 0.0f;
        XVelocity = 0.0f;
        YVelocity = 0.0f;
        Mass = 0.0f;
        Radius = 10.0f;
    }
}

class Line
{
    Node NodeStart, NodeEnd;
    float SlackLength;

    Line (Node start, Node end)
    {
        NodeStart = start;
        NodeEnd = end;
        SlackLength = GetLength()*SlackFactor;
    }
    
    public float GetLength()
    {
        return sqrt((NodeEnd.XCoord-NodeStart.XCoord)*(NodeEnd.XCoord-NodeStart.XCoord) + (NodeEnd.YCoord-NodeStart.YCoord)*(NodeEnd.YCoord-NodeStart.YCoord));
    }
}

Node[] arrayNodes;
Line[] arrayLines;

public void setup() 
{ 
    size(800, 800);
    int y_pos = height/4;
    
    arrayNodes = new Node[NUM_NODES];
    arrayNodes[0] = new Node(width/8, y_pos, true, true);
    for (int count=1; count<NUM_NODES-1; count++)
    {
        arrayNodes[count] = new Node(width/8+0.75f*count*width/(NUM_NODES-1), y_pos, false, false);
    }
    arrayNodes[NUM_NODES-1] = new Node(7*width/8, y_pos, true, true);
    
    Node startNode, endNode;
    arrayLines = new Line[NUM_NODES-1];
    endNode = arrayNodes[0];
    for (int count=1; count<NUM_NODES; count++)
    {
        startNode = endNode;
        endNode = arrayNodes[count];
        arrayLines[count-1] = new Line(startNode, endNode);
    }
} 

public void draw() 
{ 
    background(0.0f, 0.0f, 0.0f);
    //lights();
    
    if (mousePressed)
    {
        Node aNode;
        if (mouseButton==LEFT)
        {
            aNode = arrayNodes[0];
        }
        else 
        {
            aNode = arrayNodes[NUM_NODES-1];
        }
        aNode.XCoord = mouseX;
        aNode.YCoord = mouseY;
    }
    
    UpdateCoords(1.0f);

    DrawLines();
    DrawNodes();
} 

public void UpdateCoords(float time_step)
{
    Node aNode;
    Line aLine;
    float force, linelength, xlength, ylength;

/*
    for (int count=0; count<NUM_NODES; count++)
    {
        aNode = arrayNodes[count];
        aNode.XForce = 0.0;
        aNode.YForce = 0.0;  //  ready for next time
        aNode.XVelocity *= DampingFactor;
        aNode.YVelocity *= DampingFactor;  //  ready for next time
        aNode.Mass = 0.0;  //  ready for next time
    }
*/    
    for (int count=0; count<NUM_NODES-1; count++)
    {
        aLine = arrayLines[count];
        linelength = aLine.GetLength();
        force = EA*(linelength-aLine.SlackLength)/aLine.SlackLength;
        //force = 1.0*(linelength-aLine.SlackLength);
        xlength = aLine.NodeEnd.XCoord-aLine.NodeStart.XCoord;
        ylength = aLine.NodeEnd.YCoord-aLine.NodeStart.YCoord;
        
        aLine.NodeStart.XForce += 0.5f * force * xlength/linelength;
        aLine.NodeStart.YForce += 0.5f * force * ylength/linelength;
        aLine.NodeEnd.XForce -= 0.5f * force * xlength/linelength;
        aLine.NodeEnd.YForce -= 0.5f * force * ylength/linelength;
        
        if (ParabolaInstead)
        {
          aLine.NodeStart.Mass += 0.5f*abs(xlength)*MassFactor;
          aLine.NodeEnd.Mass += 0.5f*abs(xlength)*MassFactor;
        }
        else
        {
          aLine.NodeStart.Mass += 0.5f*linelength*MassFactor;
          aLine.NodeEnd.Mass += 0.5f*linelength*MassFactor;
        }
    }

    for (int count=0; count<NUM_NODES; count++)
    {
        aNode = arrayNodes[count];
        aNode.XVelocity += aNode.XForce * time_step / aNode.Mass;
        aNode.YVelocity += aNode.YForce * time_step / aNode.Mass;
        if (GravityOn) aNode.YVelocity += Gravity * aNode.Mass * time_step;  //  Gravity
        if (aNode.XFixed) aNode.XVelocity = 0.0f;
        if (aNode.YFixed) aNode.YVelocity = 0.0f;
        aNode.XCoord += aNode.XVelocity * time_step;
        aNode.YCoord += aNode.YVelocity * time_step;
        
        aNode.XForce = 0.0f;
        aNode.YForce = 0.0f;  //  ready for next time
        aNode.XVelocity *= DampingFactor;
        aNode.YVelocity *= DampingFactor;  //  ready for next time
        aNode.Radius = aNode.Mass*MassFactor/2;
        aNode.Mass = 0.0f;  //  ready for next time
    }

}

public void DrawNodes()
{
    Node aNode;

    //smooth();
    beginShape(POINTS);
    for (int count=0; count<NUM_NODES; count++)
    {
        aNode = arrayNodes[count];
        if (aNode.XFixed||aNode.YFixed)
        {
          strokeWeight(aNode.Radius*2);
          stroke(196*PApplet.parseInt(aNode.XFixed), 196*PApplet.parseInt(aNode.YFixed), 196);  //  White
        }
        else
        {
          strokeWeight(aNode.Radius);
          stroke(64, 64, 255);  //  Blueish
        }
        vertex(aNode.XCoord, aNode.YCoord);
    }
    endShape();
    //noSmooth();
}

public void DrawLines()
{
    Line aLine;

    stroke(255, 0, 0);  //  Red
    strokeWeight(4);
    beginShape(LINES);
    for (int count=0; count<NUM_NODES-1; count++)
    {
        aLine = arrayLines[count];
        vertex(aLine.NodeStart.XCoord, aLine.NodeStart.YCoord);
        vertex(aLine.NodeEnd.XCoord, aLine.NodeEnd.YCoord);
    }
    endShape();
}

public void keyPressed()
{
  //println((int)key);
  switch (key)
  {
  case ' ':
    GravityOn = !GravityOn;
    break;
    
  case 10:
    ParabolaInstead = !ParabolaInstead;
    break;
    
  case CODED:
    //println(keyCode);
    switch (keyCode)
    {
    case 34:
      GravityOn = !GravityOn;
      break;
      
    case 33:
      ParabolaInstead = !ParabolaInstead;
      break;
    }
    break;
  }
  
} 


  static public void main(String[] passedArgs) {
    String[] appletArgs = new String[] { "HangingChain" };
    if (passedArgs != null) {
      PApplet.main(concat(appletArgs, passedArgs));
    } else {
      PApplet.main(appletArgs);
    }
  }
}
