// SevenSegmentDecoder
// 
// C.T.Clarke
//
// October 2017
// A full hexadecimal decoder for a seven 
// segment display. If the blank input is
// high, the output is switched off. 
// Setting the ActiveValue parameter
// to 0 will make the output active
// low.

module SevenSegmentDecoder 
#(
	parameter				ActiveValue = 0	// The output value that switches an LED on.
)
(
	input		[3:0] Value,						// The input value in the HEX range 0-F
	input 		 	Blank,						// An input which causes all LED outputs
														// to be off if this input is active (1)
	output 	[6:0] SegmentsOut					// The 7 bit output for the 7 segment display.
);

	logic  	[6:0] Segments;

	always @*
	begin
	   // Create the right bit sequence to drive the 7 segment 
		// display based upon the input value and whether or not
		// blanking is applied.
		case ({Blank,Value})
			0: 		Segments = 7'b0111111;
			1: 		Segments = 7'b0000110;
			2: 		Segments = 7'b1011011;
			3: 		Segments = 7'b1001111;
			4: 		Segments = 7'b1100110;
			5: 		Segments = 7'b1101101;
			6:			Segments = 7'b1111101;
			7: 		Segments = 7'b0000111;
			8: 		Segments = 7'b1111111;
			9: 		Segments = 7'b1101111;
			10: 		Segments = 7'b1110111;
			11: 		Segments = 7'b1111100;
			12: 		Segments = 7'b0111001;
			13: 		Segments = 7'b1011110;
			14: 		Segments = 7'b1111001;
			15: 		Segments = 7'b1110001;
			default:	Segments = 7'b0000000;
		endcase
	end

	// Invert the result if the output is active low.
	assign SegmentsOut = (ActiveValue == 1) ? Segments : ~Segments;
endmodule
	