#!/usr/bin/env python
"""
A python module with logging to animate a reversible
M/M/1 queue.

Metadata:

  - Author:    W.S.Kendall
  - Copyright: opensource (c) W.S.Kendall 2008
  - Date:      2008-07-04
  - URL:       http://www.wilfridkendall.co.uk
  - Generator: '/home/wilfrid/lib/python/generate.py' version 1.18.


Command-line options::

  usage: %prog [options] arguments
    --logging=LOGGING:	   logging verbosity level.

Notes:
 - values for 'logging' option:
 'DEBUG', 'INFO', 'WARNING', 'ERROR', 'CRITICAL'
 (decreasing level of verbosity), with 'WARNING' as default.

"""
__RCSID__ = '$Revision: 1.1 $ $Date: 2008/07/05 15:26:33 $'


# \section{Definitions}
# We need the logging module regardless of whether the #logging# option
# is deployed, since #logging.info# calls are scattered through the text.
import logging

from Tkinter import BOTH, Checkbutton, DoubleVar, FIRST, Frame, GROOVE, \
    IntVar, LAST, LEFT, Radiobutton, Scale, Tk, VERTICAL, W, X, Y

from mm1 import MM1

from display2 import ITALICS, BOLD, Display2

#\section{Classes}
class AnimateMM1(Display2):
	"""
	Presents animation of reversible M/M/1 queue.
	"""
	def __init__(self, root, x0=0, x1=20, y0=0, y1=15):
		Display2.__init__(self, root, xlo=x0, xhi=x1, ylo=y0, yhi=y1)#, \
					#  image='image/queue.jpg')

		# Buttons for actions.
		self.button(text='SIMULATE', font=BOLD, foreground='darkgreen',\
				    command=self.simulate)
		self.button(text='ANALYZE', font=BOLD, foreground='darkgreen', \
				    command=self.analyze)
		self.button(text='ITERATE', font=BOLD, foreground='darkgreen', \
				    command=self.iterate)

		# Scales for adjustments.
		self.alpha = DoubleVar()
		self.alpha.set(1)

		self.lambd = DoubleVar()
		self.lambd.set(1)
		Scale(self.buttonframe, from_=10.0, to=0.0, \
				      resolution=0.1, \
				      orient=VERTICAL, width=20, \
				      label='lambda', font=BOLD, \
				      variable=self.lambd, \
				      command=self.wipe).pack(fill=BOTH)

		self.mu = DoubleVar()
		self.mu.set(2)
		Scale(self.buttonframe, from_=10.0, to=0.0, \
				      resolution=0.1, \
				      orient=VERTICAL, width=20, \
				      label='mu', font=BOLD, \
				      variable=self.mu, \
				      command=self.wipe).pack(fill=BOTH)

		self.batch = IntVar()
		self.batch.set(200)

		# Buttons for adjustments.
		radioframe = Frame(self.buttonframe, \
					   borderwidth=1, relief=GROOVE)
		radioframe.pack()
		self.highlightvar = IntVar()
		self.highlightvar.set(1)
		Radiobutton(radioframe, value=1, text='None', width=11, \
				    anchor=W, font=ITALICS, justify=LEFT, \
				    command=self.highlight, \
				    variable=self.highlightvar).pack(fill=X)
		Radiobutton(radioframe, value=2, text='Forwards', \
				    anchor=W, width=11, font=ITALICS, \
				    justify=LEFT, \
				    command=self.highlight, \
				    variable=self.highlightvar).pack(fill=X)
		Radiobutton(radioframe, value=3, text='Backwards', \
				    anchor=W, width=11, font=ITALICS, \
				    justify=LEFT, \
				    command=self.highlight, \
				    variable=self.highlightvar).pack(fill=X)
		self.toggle = IntVar()
		self.toggle.set(True)
		self.togglecb = Checkbutton(self.buttonframe, \
						    foreground='black', \
						    text='EQUILIBRIUM', \
						    width = 11,
						    font=BOLD, anchor=W, \
						    variable=self.toggle, \
						    command=self.togglewipe)
		self.togglecb.pack(fill=X)

		self.button(text='WIPE', font=BOLD, foreground='firebrick', \
				    command=self.wipe)

		for yinc in range(y0+1, y1+1):
			self.axis(x0, yinc, x1, fill='lightgray')
		
		self.x0, self.x1, self.y0, self.y1 = x0, x1, y0, y1
		
		self.chain, self.rev_trans, self.rev_delay = None, [], []

		for i in range(self.y0, self.y1+1):
			self.rev_trans.append([0, 0])
			self.rev_delay.append([0, 0.0])

		self.iterations = 1

	def highlight(self):
		c = self.highlightvar.get()
		if c == 1:
			self.canvas.itemconfig('arrival', fill='yellow')
			self.canvas.itemconfig('depart', fill='yellow')
			self.canvas.itemconfig('arrivaldot', fill='white')
			self.canvas.itemconfig('departdot', fill='white')
		elif c == 2:
			self.canvas.itemconfig('arrival', fill='red')
			self.canvas.itemconfig('depart', fill='yellow')
			self.canvas.lift('arrivaldot')
			self.canvas.itemconfig('arrivaldot', fill='red')
			self.canvas.itemconfig('departdot', fill='white')
		elif c == 3:
			self.canvas.itemconfig('arrival', fill='yellow')
			self.canvas.itemconfig('depart', fill='red')
			self.canvas.lift('departdot')
			self.canvas.itemconfig('arrivaldot', fill='white')
			self.canvas.itemconfig('departdot', fill='red')
		
	def simulate(self):
		"""
		Run a single animation.
		"""
		if self.chain is not None:
			return
		
		self.canvas.delete('active')
		self.chain = MM1(lambd=self.lambd.get(), mu=self.mu.get())
		if self.toggle.get():
			self.chain.equilibrium_draw()
		else:
			self.chain[:] = [(0, self.chain.rng.choice([0, 10]))]
		loc = self.chain[-1]
		self.dot(loc, fill='red', width=4, tags='active')

		data = {}

		while self.chain[-1][0] < self.x1:
			self.chain.update()

			if self.chain[-1][0] > self.x1:
				self.chain[-1] = (self.x1, self.chain[-1][1])
			loc = self.chain[-1]
			self.dot(loc, fill='midnightblue', \
					 tags='active')
		self.canvas.update_idletasks()
		self.iterations += 1
	
	def analyze(self):
		"""
		Analyze last animation.
		"""
		if self.chain is None:
			return

		self.canvas.delete('transient')
		T = self.x1
		for transition in self.chain.reversed_statistics():
			t, start, finis = transition
			# Note we record under #finis# not #start#!
			if finis > self.y1:
				pass
			elif start == finis:
				raise Error, start
			else:
				self.rev_delay[finis][0] += 1
				self.rev_delay[finis][1] += t

			# Backwards version
			self.dot((T, finis), outline='', fill='lightblue', \
					 tags='transient')
			if start < finis:
				self.line(T, start, T, finis, \
						  fill='yellow', arrow=LAST, \
						  tag=('transient', 'arrival'))
				self.dot((T, -0.3), outline='', fill='', width=3, tag=('transient', 'arrivaldot'))
			else:
				self.line(T, start, T, finis, \
						  fill='yellow', arrow=LAST, \
						  tag=('transient', 'depart'))
				self.dot((T, -0.3), outline='', fill='', width=3, tag=('transient', 'departdot'))
			self.line(T, finis, T - t, finis, \
					  fill='red', \
					  tag='transient')

			T -= t
		
		def delay_plus(x):
			if x[0] == 0:
				return '???'
			else:
				return '%3.1f' % (float(x[0]) / x[1])
		self.canvas.delete('prob')
		for i, txt in enumerate(delay_plus(x) for x in self.rev_delay):
			self.write(txt, -0.6, i, font=BOLD, tags='prob')
				
		self.chain = None

	def iterate(self):
		m, dm = 10, self.batch.get()
		for i in range(self.iterations, self.iterations + dm):
			self.simulate()
			self.analyze()
			if not i % m:
				self.canvas.delete('record')
				self.write('[%4d]' % i, -0.5, -0.5, \
						   font=ITALICS, tags='record')
				self.canvas.update_idletasks()
		self.iterations += dm
	
	def togglewipe(self):
		"""
		Wipe all objects and change checkbutton appearance.
		"""
		if self.toggle.get():
			self.togglecb['text'] = 'EQUILIBRIUM'
			self.togglecb['foreground'] = 'black'
		else:
			self.togglecb['text'] = '2 PT START '
			self.togglecb['foreground'] = 'red'
		self.wipe()

	def wipe(self, *args):
		"""
		Wipe all objects.
		"""
		if self.lambd.get() >= self.mu.get():
			self.lambd.set(self.mu.get() * 0.5)

		self.chain, self.iterations = None, 1
		self.rev_trans, self.rev_delay = [], []
		for i in range(self.y0, self.y1 + 1):
			self.rev_trans.append([0, 0])
			self.rev_delay.append([0, 0.0])
		
		self.canvas.delete('objects', 'active', 'transient', 'prob', 'record')


#\section{Exceptions}
class Error(Exception):
	"""
	Generic exception to be raised by this module.
	"""

# \section{Functions}
def main(opt, arguments):
	"""
	The action of the module as script should be concentrated here!
	"""
	root = Tk()
	root.title('Reversible M/M/1 queue')
	animation = AnimateMM1(root)
	
	root.mainloop()




# \section{Main script}
if __name__ == '__main__':
	import optionparse
	option, args = optionparse.parse(__doc__, version=__RCSID__)

	if option.logging is not None:
		logging.basicConfig(level=logging.__dict__[option.logging])
		logging.info('Logging level %s' % option.logging)


	# The action of the module as script should be concentrated
	# in the following function, to facilitate profiling.
	main(option, args)


	logging.info('Finishing')
	raise SystemExit


# \newpage \scriptsize
# \begin{multicols}{2}
# \section{History}
# \text{ }
#\\ $Log: animatemm1.pyw,v $
#\\ Revision 1.1  2008/07/05 15:26:33  wskendall
#\\ Initial revision
#\\
#
# \end{multicols}

'$Id: animatemm1.pyw,v 1.1 2008/07/05 15:26:33 wskendall Exp wskendall $'
