// Generates some distribution on a discrete lattice, and then runs a Metropolis-Hastings sampler starting from the uniform distribution,
// drawing the heat map of the sampler on the right. If you press a key it will start another sampler from the invariant distribution,
// which moves independently of the original sampler until the two meet, at which point they stick together.
int latticesize;
float spacing;
float[][] distn;
float totalmass;
int[][] heat;
int maxheat;
int[] peaks_x;
int[] peaks_y;
float[] peaks_height;
int nopeaks;
int rw_x, old_x, sw_x, sold_x;
int rw_y, old_y, sw_y, sold_y;
int randno;
float acceptanceprob;
int secondwalker;
int stopper = 1;

void setup() {
  
  size(1600, 800); // Size of the window
  background(255); // Start with everything white
  
  latticesize=15;  // Width and height of the lattice we are going to work on
  nopeaks = 3;     // The number of peaks in the landscape used to define the distribution we sample from
  distn = new float[latticesize+5][latticesize+5]; // Initialising arrays
  heat = new int[latticesize+5][latticesize+5];
  peaks_x = new int[nopeaks+1];
  peaks_y = new int[nopeaks+1];
  peaks_height = new float[nopeaks+1];
  
  spacing = (height-100)/(float)latticesize; // size of squares in the lattice
  
  for( int i=1 ; i<=nopeaks ; i++ ) {        // we are going to create a landscape with a few peaks, which will be used to define the distribution we sample from
    peaks_x[i] = floor(random(latticesize)); // the x and y co-ords of the peaks are uniform
    peaks_y[i] = floor(random(latticesize));
    peaks_height[i] = random(50);            // the heights of the peaks are Unif(50) (no reason for this, I made a more or less arbitrary choice)
  }
  
  totalmass = 0;                             // keep track of the total mass of the measure (the total height of all points in the lattice added together)
  for( int i=1 ; i<=latticesize ; i++ ) {    
    for( int j=1 ; j<=latticesize ; j++ ) {
      distn[i][j]=1;                         // distn[i][j] calculates the height of the landscape at the site (i,j). When we rescale by totalmass it is a distribution
      for( int k=1 ; k<=nopeaks ; k++ ) {
        distn[i][j] += peaks_height[k]/(3*dist(i,j,peaks_x[k],peaks_y[k])+1); // We now decide the height of each point in the lattice depending on its distance from the peaks. Again I made some more or less arbitrary choice of function here.
      }
      fill(min(255,75*log(distn[i][j])));
      rect(50+(i-0.5)*spacing, 50+(j-0.5)*spacing, spacing, spacing); // draw the landscape, with peaks lighter and valleys darker
      totalmass += distn[i][j];
    }
  }
  
  rw_x = floor(random(latticesize))+1; // start a sampler from a uniform location
  rw_y = floor(random(latticesize))+1;
  
  fill(255,0,0);
  rect(50+(rw_x-0.5)*spacing, 50+(rw_y-0.5)*spacing, spacing, spacing); // draw the sampler in red
  heat[rw_x][rw_y] += 1;                                                // heat[i][j] tells us how much time the sampler has spent at site (i,j)
  
  noLoop();
}

void draw()
{
  old_x = rw_x;
  old_y = rw_y;
  fill(min(255,75*log(distn[old_x][old_y]))); // we are going to move the sampler, so colour its old location in its landscape colour
  rect(50+(old_x-0.5)*spacing, 50+(old_y-0.5)*spacing, spacing, spacing);
  
  randno = floor(random(4)); // Generate a random integer between 0 and 3, to decide the direction the sampler proposes to move
  if( randno == 0 ) { rw_x += 1; } else if( randno == 1 ) { rw_x += -1; } else if( randno == 2 ) { rw_y += 1; } else { rw_y += -1; } // if the integer above is zero, propose to move right, etc.
  while( rw_x <= 0 ) { rw_x += latticesize; } // torus boundary conditions for the sampler (somewhat incosistent with the landscape, but this shouldn't cause any problems)
  while( rw_x > latticesize ) { rw_x -= latticesize; }
  while( rw_y <= 0 ) { rw_y += latticesize; }
  while( rw_y > latticesize ) { rw_y -= latticesize; }
  
  acceptanceprob = min( 1, distn[rw_x][rw_y]/distn[old_x][old_y] ); // calculate the probability that the sampler moves to its proposed location
  
  if( random(1) > acceptanceprob ) { rw_x = old_x; rw_y = old_y; } // if the proposed location is too unlikely, the sampler stays where it was
  
  fill(255,0,0);
  rect(50+(rw_x-0.5)*spacing, 50+(rw_y-0.5)*spacing, spacing, spacing); // draw the sampler in red
  heat[rw_x][rw_y] += 1;                                                // update the heatmap
  if( heat[rw_x][rw_y] > maxheat ) { maxheat = heat[rw_x][rw_y]; }      // update the maximum heat tracker
  
  for( int i=1 ; i<=latticesize ; i++ ) {    
    for( int j=1 ; j<=latticesize ; j++ ) {
      fill(255*heat[i][j]/(float)maxheat,0,0);                         // colour the heatmap more red the more times a site has been visited
      rect(810+(i-0.5)*spacing, 50+(j-0.5)*spacing, spacing, spacing); // draw the heatmap
    }
  }
  
  if( secondwalker == 1 ) {   // Below there is a function that checks whether a key has been pressed. If it has, it starts a second sampler
    sold_x = sw_x;            // If we have a second sampler then we do basically the same thing as for the first sampler, independently
    sold_y = sw_y;
    
    fill(min(255,75*log(distn[sold_x][sold_y]))); // we are going to move the sampler, so colour its old location in its landscape colour
    rect(50+(sold_x-0.5)*spacing, 50+(sold_y-0.5)*spacing, spacing, spacing);
    
    randno = floor(random(4)); // proposing a move for the second sampler
    if( randno == 0 ) { sw_x += 1; } else if( randno == 1 ) { sw_x += -1; } else if( randno == 2 ) { sw_y += 1; } else { sw_y += -1; }
    while( sw_x <= 0 ) { sw_x += latticesize; }
    while( sw_x > latticesize ) { sw_x -= latticesize; }
    while( sw_y <= 0 ) { sw_y += latticesize; }
    while( sw_y > latticesize ) { sw_y -= latticesize; }
    
    acceptanceprob = min( 1, distn[sw_x][sw_y]/distn[sold_x][sold_y] ); // calculate the probability that the second sampler moves to its proposed location
    
    if( random(1) > acceptanceprob ) { sw_x = sold_x; sw_y = sold_y; } // if the proposed location is too unlikely, the second sampler stays where it was
    
    fill(0,255,0);
    rect(50+(sw_x-0.5)*spacing, 50+(sw_y-0.5)*spacing, spacing, spacing); // draw the second sampler in green
    
    if( rw_x == sw_x && rw_y == sw_y ) { secondwalker = 0; }              // if the two samplers meet, they stick together (i.e. we stop tracking the second one)
  }
  
}

void keyPressed() {                  // if a key is pressed...
  if (key == 'p' || key == 'P') { if (stopper == 1) { loop(); stopper = 0; } else { noLoop(); stopper = 1; } } // if it's a P, pause / unpause
  else if(secondwalker == 0 ) {           // ...and we don't already have a second sampler on the screen... 
    secondwalker = 1;                // ...then start a second sampler.
    
    for( int i=1 ; i<=latticesize ; i++ ) {
      for( int j=1 ; j<=latticesize ; j++ ) {
        if( random(1) < distn[i][j]/totalmass ) {
          sw_x = i; // start the second sampler from a location chosen according to the equilibrium measure
          sw_y = j;
        }
      }
    }
  }
}
