// Generates some distribution on a (continuous) region, and then runs a Metropolis-Hastings sampler starting from the uniform distribution,
// drawing the heat map of the sampler on the right. If you press a key it will start another sampler from the invariant distribution,
// which moves independently of the original sampler until the two meet, at which point they stick together. Except they will basically never meet
// because the space is continuous! (Note some aspects are done somewhat discretely but with a relatively fine mesh. It would be nicer to do things
// more continuously but for simplicity I went with the grid.)
float[][] distn;
int gridwidth;
float totalmass;
int maxheat;
int[][] heat;
float[] peaks_x;
float[] peaks_y;
float[] peaks_height;
int nopeaks;
float rw_x, old_x, sw_x, sold_x;
float rw_y, old_y, sw_y, sold_y;
int randno;
float randno1, randno2;
float acceptanceprob;
int secondwalker;
int counter;

void setup() {
  
  size(1600, 800);  // Size of the window
  background(255);  // Start with everything white
  
  nopeaks = 20;     // The number of peaks in the landscape used to define the distribution we sample from
  gridwidth = 700;  // We will approximate a continuous distribution with a discrete grid of this width and height
  distn = new float[gridwidth+5][gridwidth+5]; // Initialising arrays
  peaks_x = new float[nopeaks+1];
  peaks_y = new float[nopeaks+1];
  peaks_height = new float[nopeaks+1];
  heat = new int[gridwidth+5][gridwidth+5];
  maxheat = 0;
  counter = 0;
  
  for( int i=1 ; i<=nopeaks ; i++ ) {        // we are going to create a landscape with a few peaks, which will be used to define the distribution we sample from
    peaks_x[i] = random(gridwidth);          // the x and y co-ords of the peaks are uniform
    peaks_y[i] = random(gridwidth);
    peaks_height[i] = random(100);           // the heights of the peaks are Unif(100) (no reason for this, I made a more or less arbitrary choice)
  }
  
  totalmass = 0;                             // keep track of the total mass of the measure (the total height of all points in the grid added together)
  for( int i=1 ; i<=gridwidth ; i++ ) {    
    for( int j=1 ; j<=gridwidth ; j++ ) {
      distn[i][j]=1;                         // distn[i][j] calculates the height of the landscape at the site (i,j). When we rescale by totalmass it is a distribution
      for( int k=1 ; k<=nopeaks ; k++ ) {
        distn[i][j] += peaks_height[k]/(0.1*dist(i,j,peaks_x[k],peaks_y[k])*sqrt(dist(i,j,peaks_x[k],peaks_y[k]))+1); // We now decide the height of each point in the grid depending on its distance from the peaks. Again I made some more or less arbitrary choice of function here.
      }
      stroke(min(255,75*log(distn[i][j])));
      point(50+i, 50+j); // draw the landscape, with peaks lighter and valleys darker
      totalmass += distn[i][j];
    }
  }
  
  rw_x = random(gridwidth)+1; // start a sampler from a uniform location
  rw_y = random(gridwidth)+1;
  
  strokeWeight(7);
  stroke(255,0,0);
  point(50+rw_x, 50+rw_y); // draw the sampler in red
  heat[floor(rw_x)][floor(rw_y)] += 1; // heat[i][j] tells us how much time the sampler has spent near point (i,j) in the grid
}

void draw()
{
  counter++;
  
  old_x = rw_x;
  old_y = rw_y;
                         // we are going to move the sampler, so colour its old location in its landscape colour
  strokeWeight(1);       // the thickness of the "point" function - here we want it to draw one pixel at a time
  strokeCap(PROJECT);    // when we draw with the "point" function, make sure it draws squares
  for( int i=max(floor(old_x) - 5,1) ; i<= min(ceil(old_x)+5,gridwidth) ; i++ ) {   // we want to make sure we cover the area where the sampler was
    for( int j=max(floor(old_y) - 5,1) ; j<= min(ceil(old_y)+5,gridwidth) ; j++ ) {
      stroke(min(255,75*log(distn[i][j])));                                         // choose the appropriate colour to draw with (for some reason this draws a very slightly different colour from initially, which I don't understand, but it has the benefit of letting us see where the sampler has been
      point(50+i, 50+j); // draw the landscape, with peaks lighter and valleys darker
      
      heat[i][j] += 1;   // update the heatmap
      if( heat[i][j] > maxheat ) { maxheat = heat[i][j]; } // update the maximum heat tracker
    }
  }
  
  randno1 = random(50);
  randno2 = random(2*PI);
  rw_x += randno1*cos(randno2); // generate a random point uniform on a circle of radius 50 (the choice of 50 is arbitrary), to decide where the sampler proposes to move
  rw_y += randno1*sin(randno2);
  
  while( rw_x < 1 ) { rw_x += gridwidth; } // torus boundary conditions for the sampler (somewhat incosistent with the landscape, but this shouldn't cause any problems)
  while( rw_x > gridwidth+1 ) { rw_x -= gridwidth; }
  while( rw_y < 1 ) { rw_y += gridwidth; }
  while( rw_y > gridwidth+1 ) { rw_y -= gridwidth; }
  
  acceptanceprob = min( 1, distn[floor(rw_x)][floor(rw_y)]/distn[floor(old_x)][floor(old_y)] ); // calculate the probability that the sampler moves to its proposed location
    
  if( random(1) > acceptanceprob ) { rw_x = old_x; rw_y = old_y; } // if the proposed location is too unlikely, the sampler stays where it was
  
  if( counter > 100 ) { // updating the whole heatmap is a bit slow, so only do it every 100 steps
    for( int i=1 ; i<=gridwidth ; i++ ) {    
      for( int j=1 ; j<=gridwidth ; j++ ) {
        stroke(255*heat[i][j]/(float)maxheat,0,0); // draw in the right colour
        strokeCap(PROJECT);           // draw squares
        strokeWeight(1);              // squares of size one
        point(gridwidth+150+i, 50+j); // draw the heatmap
      }
    }
    counter = 0;
  }
  
  
  if( secondwalker == 1 ) { // Below there is a function that checks whether a key has been pressed. If it has, it starts a second sampler
    sold_x = sw_x;          // If we have a second sampler then we do basically the same thing as for the first sampler, independently
    sold_y = sw_y;
    
    for( int i=max(floor(sold_x) - 5,1) ; i<= min(ceil(sold_x)+5,gridwidth) ; i++ ) { // we are going to move the sampler, so colour its old location in its landscape colour
      for( int j=max(floor(sold_y) - 5,1) ; j<= min(ceil(sold_y)+5,gridwidth) ; j++ ) {
        stroke(min(255,75*log(distn[i][j])));
        point(50+i, 50+j);
      }
    }
    
    randno1 = random(50); // proposing a move for the second sampler
    randno2 = random(2*PI);
    sw_x += randno1*cos(randno2);
    sw_y += randno1*sin(randno2);
    
    while( sw_x < 1 ) { sw_x += gridwidth; }
    while( sw_x > gridwidth+1 ) { sw_x -= gridwidth; }
    while( sw_y < 1 ) { sw_y += gridwidth; }
    while( sw_y > gridwidth+1 ) { sw_y -= gridwidth; }
    
    acceptanceprob = min( 1, distn[floor(sw_x)][floor(sw_y)]/distn[floor(sold_x)][floor(sold_y)] ); // calculate the probability that the second sampler moves to its proposed location
      
    if( random(1) > acceptanceprob ) { sw_x = sold_x; sw_y = sold_y; } // if the proposed location is too unlikely, the second sampler stays where it was
    
    if( sw_x > 6 && sw_x < gridwidth - 5 && sw_y > 6 && sw_y < gridwidth - 5 ) {
      strokeWeight(10);
      strokeCap(ROUND);
      stroke(0,255,0);        // draw the second sampler in green
      point(50+sw_x, 50+sw_y);
    } 
    
    if( rw_x == sw_x && rw_y == sw_y ) { secondwalker = 0; } // if the two samplers meet, they stick together (i.e. we stop tracking the second one) - essentially this will never happen as they are making continuous moves
    
  }
  
  if( rw_x > 6 && rw_x < gridwidth - 5 && rw_y > 6 && rw_y < gridwidth - 5 ) {
    strokeWeight(10);
    strokeCap(ROUND);
    stroke(255,0,0);
    point(50+rw_x, 50+rw_y); // draw the first sampler in red
  }
}


void keyPressed() {          // if a key is pressed...
  if(secondwalker == 0 ) {   // ...and we don't already have a second sampler on the screen...
    secondwalker = 1;        // ...then start a second sampler.
    
    for( int i=1 ; i<=gridwidth ; i++ ) {
      for( int j=1 ; j<=gridwidth ; j++ ) {
        if( random(1) < distn[i][j]/totalmass ) { // start the second sampler from a location chosen according to the equilibrium measure
          sw_x = i; 
          sw_y = j;
        }
      }
    }
  }
}
