function [meanMLQMC,stdMLQMC,costMLQMC] = mlqmc(l0,L,R,skl,kinit,eps)
%
% Multilevel Quasi-Monte Carlo method
%
%   inputs:  l0    ... coarsest level
%            L     ... finest level
%            R     ... number of random shifts
%            skl   ... number of KL modes (choose smaller than 500)
%            eps   ... tolerance for sampling error
%            kinit ... log_2 of initial number of lattice points
%
%   output:  costMLQMC  ... cost for run in FLOPs
%
%   Note: To balance the FE, truncation and sampling errors, the following 
%         values for skl and eps should be chosen
%
%    sig2=1, nu=1, lam=1:   skl = ceil(0.38*2^(l)), eps = 0.159*2^(-l+1)
%    sig2=1, nu=1, lam=0.1: skl = ceil(1.78*2^(l)), eps = 0.507*2^(-l+1)
%    sig2=1, nu=2, lam=1:   skl = ceil(0.76*2^(l/2)), eps = 0.154*2^(-l+1)
%    sig2=1, nu=1, lam=0.1: skl = ceil(2.38*2^(l/2)), eps = 0.400*2^(-l+1)
%
% ++++++++++++++++++++++++++++++++++
% Written March 2015 by R. Scheichl
% ++++++++++++++++++++++++++++++++++

nL = L-l0+1;

% Open file with generating vector for lattice points
%fid = fopen('lattice-39101-1024-1048576.3600.txt');  % weights 1/j^1
fid = fopen('lattice-39102-1024-1048576.3600.txt');  % weights 1/j^2
%fid = fopen('lattice-33002-1024-1048576.9125.txt');  % weights 1/j^3

data = textscan(fid,'%f%f');
fclose(fid);

z = data{2};          % generating vector (3600 x 1)

% Choose the parameters for the covariance function

nu   = 3;    % smoothness parameter (in [1/2, \infty))
lam  = 1;    % correlation length (> 0) 
sig2 = 1;    % variance (> 0)

% Cost (in FLOPs) to produce one sample

cost_l = (2*skl+13).*2.^[l0:L];
cost_l(1) = cost_l(1) - 4*2^l0;

% Evaluate solution at x = 1/3
xs = 1/3;

% Set up the different FE problems
h = 2.^(-[l0:L]);
m = 1./h;

% Generate cell array for the shifts
b = cell(nL,1);
sh = cell(nL,1);  
MM = cell(nL,1);

% Evaluate the QMC estimates on all levels for an initial number of estimates 

N=2^kinit;  % initial number of lattice points
v = ([0:N-1] / N )' * z(1:skl)';

sumQMC = zeros(R,nL);
meanQMC = zeros(1,nL);
varQMC = zeros(1,nL);

for l=l0:L
    
    il = l-l0+1;
    b{il} = ones(m(il)-1,1)*h(il)^2;
    sh{il} = rand(R,skl);  % generate random shifts on level l
    
    [ev,ef] = klsetup1D_matern(lam,nu,sig2,skl,h(il),1000);
    MM{il} = ef*diag(sqrt(ev));

    for r=1:R     % loop over shifts
        
        vs = v + repmat(sh{il}(r,:),N,1);  % add random shift
        x = vs-floor(vs);                      % fractional part
        
        for n=1:N
        
            % Coordinate transformation -> mkl dimensional normal distribution !
            xiu = x(n,1:skl)';  
            xi = icdf('normal',xiu,0,1);
        
            kf = exp(MM{il}*xi);
            Uf = solverTh(m(il),kf,b{il});
            Pf = eval_pde1D(m(il),Uf,xs);
               
            if l > l0
                kc = exp(MM{il-1}*xi(1:skl));
                Uc = solverTh(m(il-1),kc,b{il-1});
                Pc = eval_pde1D(m(il-1),Uc,xs);
            else 
                Pc = 0;
            end
                    
            sumQMC(r,il) = sumQMC(r,il) + Pf-Pc;
            
        end
        
        % Evaluate the required sums.
        % Note: To limit the impact of round-off errors, we compute the running 
        % arithmetic mean and corrected sum of squares and use an update formula 
        % from 
        %
        % B.P. Welford: Note on a method for calculating corrected sums of
        % squares and products. Technometrics 4(3), 1962, pp. 419-420.
            
        varQMC(il) = varQMC(il) + (r-1)/r * (sumQMC(r,il)/N-meanQMC(il))^2;
        meanQMC(il) = ((r-1)* meanQMC(il) + sumQMC(r,il)/N)/r;
        
    end
    
end

varQMC = varQMC/((R-1)*R);
Nl = N*ones(1,nL);

maxit=100;
for it=1:maxit
      
    varMLQMC = sum(varQMC(1:nL));
    
    if varMLQMC < eps^2
        costMLQMC = R*sum(Nl.*cost_l);
        break;
    end
    
    [profit,il] = max(varQMC./cost_l./Nl);
           
    Nl(il) = 2*Nl(il);
    
    meanQMC(il) = 0;
    varQMC(il) = 0;
    
    v = ([1:2:Nl(il)-1] / Nl(il) )' * z(1:skl)';
    
    for r=1:R     % loop over shifts
        
        vs = v + repmat(sh{il}(r,1:skl),Nl(il)/2,1);  % add random shift
        x  = vs-floor(vs);                        % fractional part
        
        for n=1:Nl(il)/2
        
            % Coordinate transformation -> mkl dimensional normal distribution !
            xiu = x(n,1:skl)';  
            xi = icdf('normal',xiu,0,1);
        
            kf = exp(MM{il}*xi);
            Uf = solverTh(m(il),kf,b{il});
            Pf = eval_pde1D(m(il),Uf,xs);
               
            if il > 1
                kc = exp(MM{il-1}*xi(1:skl));
                Uc = solverTh(m(il-1),kc,b{il-1});
                Pc = eval_pde1D(m(il-1),Uc,xs);
            else 
                Pc = 0;
            end
                    
            sumQMC(r,il) = sumQMC(r,il) + Pf-Pc;
            
        end
        
        % Evaluate the required sums.
        % Note: To limit the impact of round-off errors, we compute the running 
        % arithmetic mean and corrected sum of squares and use an update formula 
        % from 
        %
        % B.P. Welford: Note on a method for calculating corrected sums of
        % squares and products. Technometrics 4(3), 1962, pp. 419-420.
            
        varQMC(il) = varQMC(il) + (r-1)/r * (sumQMC(r,il)/Nl(il)-meanQMC(il))^2;
        meanQMC(il) = ((r-1)* meanQMC(il) + sumQMC(r,il)/Nl(il))/r;
        
    end
    
    varQMC(il) = varQMC(il)/((R-1)*R);
    
end

R*Nl
meanMLQMC = sum(meanQMC);
stdMLQMC = sqrt(varMLQMC);

end
