function [meanMC,stdMC,costMC] = mc(l,skl,Ninit,eps)
%
% Standard Monte Carlo method
%
%   inputs:  l     ... grid level
%            skl   ... number of KL modes (choose smaller than 500)
%            Ninit ... initial number of samples
%            eps   ... tolerance for sampling error
%
%   output:  meanMC  ... the expected value of the quantity of interest
%            stdMC   ... the standard error of the estimator
%            costMC  ... cost for run in FLOPs
%
%   Note: To balance the FE, truncation and sampling errors, the following 
%         values for skl and eps should be chosen
%
%    sig2=1, nu=1, lam=1:   skl = ceil(0.38*2^(l)), eps = 0.159*2^(-l+1)
%    sig2=1, nu=1, lam=0.1: skl = ceil(1.78*2^(l)), eps = 0.507*2^(-l+1)
%    sig2=1, nu=2, lam=1:   skl = ceil(0.76*2^(l/2)), eps = 0.154*2^(-l+1)
%    sig2=1, nu=1, lam=0.1: skl = ceil(2.38*2^(l/2)), eps = 0.400*2^(-l+1)
%
% ++++++++++++++++++++++++++++++++++
% Written May 2015 by R. Scheichl
% ++++++++++++++++++++++++++++++++++
          
format long

% Choose the parameters for the covariance function

nu   = 1;    % smoothness parameter (in [1/2, \infty))
lam  = 1;    % correlation length (> 0) 
sig2 = 1;    % variance (> 0)

% Cost (in FLOPs) to produce one sample

cost = (2*skl+9)*2^l;

% Evaluate solution at x = 1/3

xs = 1/3;

hf = 2^(-l);     % fine mesh at level L
mf = 1/hf;

% RHs f=1

bf=hf^2*ones(mf-1,1); 
    
% Set up the KL expansions (using 1000 Gauss points in Nystrom method)

[ev,eff] = klsetup1D_matern(lam,nu,sig2,skl,hf,1000);

Mf = eff*diag(sqrt(ev));
    
%%%%%%%%%%%%%%%
% MC sampling

Ndone=0; N=Ninit; sum1=0; sum2=0;

while N > Ndone

    Nnew = N-Ndone;
    
    xMC = randn(skl,Nnew);

    for n= 1:Nnew     % Loop over samples

        kf = exp(Mf*xMC(:,n));

        % finite element solves and evaluation of QoI on both levels
        Uf = solverTh(mf,kf,bf);
        Pf = eval_pde1D(mf,Uf,xs);

        sum1 = sum1 + Pf;
        sum2 = sum2 + Pf^2;
    
    end

    varMC = (sum2-sum1^2/N)/(N-1);   
    Ndone = N;
    N = ceil(varMC/eps^2);
    
    N = max(N,Ndone);
    
end

meanMC = sum1/Ndone;
stdMC = sqrt(varMC/Ndone);
Ndone

costMC = Ndone*cost;

end
      