function [meanMLMC,stdMLMC,costMLMC] = mlmc(l0,L,skl,Ninit,eps)
%
%   inputs:  l0    ... coarsest level
%            L     ... finest level
%            skl   ... number of KL modes (choose smaller than 500)
%            Ninit ... initial number of samples
%            eps   ... tolerance for sample variance error
%
%   output:  costMLMC  ... cost for run in FLOPs
%
%   Note: To balance the FE, truncation and sampling errors, the following 
%         values for skl and eps should be chosen
%
%    sig2=1, nu=1, lam=1:   skl = ceil(0.38*2^(l)), eps = 0.159*2^(-l+1)
%    sig2=1, nu=1, lam=0.1: skl = ceil(1.78*2^(l)), eps = 0.507*2^(-l+1)
%    sig2=1, nu=2, lam=1:   skl = ceil(0.76*2^(l/2)), eps = 0.154*2^(-l+1)
%    sig2=1, nu=1, lam=0.1: skl = ceil(2.38*2^(l/2)), eps = 0.400*2^(-l+1)
%
% ++++++++++++++++++++++++++++++++++
% Written March 2015 by R. Scheichl
% ++++++++++++++++++++++++++++++++++
      
format long

nL = L-l0+1;

% Choose the parameters for the covariance function

nu   = 1;    % smoothness parameter (in [1/2, \infty))
lam  = 1;    % correlation length (> 0) 
sig2 = 1;    % variance (> 0)

% Cost (in FLOPs) to produce one sample

cost_l = (2*skl+13).*2.^[l0:L];
cost_l(1) = cost_l(1) - 4*2^l0;

% Evaluate solution at x = 1/3

xs = 1/3;

% Set up the different FE problems
h = 2.^(-[l0:L]);
m = 1./h;

% Generate cell array for the RHS and the scaled eigenfunctions
b = cell(nL,1);
MM = cell(nL,1);

for l=l0:L
    
    il = l-l0+1;
    b{il} = ones(m(il)-1,1)*h(il)^2;

    [ev,ef] = klsetup1D_matern(lam,nu,sig2,skl,h(il),1000);
    MM{il} = ef*diag(sqrt(ev));

end

Ndone=zeros(nL,1); N=ones(nL,1)*Ninit; 
sum1=zeros(nL,1); sum2=zeros(nL,1); varMC = zeros(nL,1);

while max(N-Ndone) > 0

    Nnew = N-Ndone;
    
    for l=l0:L
    
        if Nnew(il) > 0
            
            il = l-l0+1;
            
            for block=1:ceil(Nnew(il)/1000)
                
                nsamp = min(1000,Nnew(il)-(block-1)*1000);
                xMC = randn(skl,nsamp);

                for n= 1:nsamp     % Loop over samples

                    kf = exp(MM{il}*xMC(:,n));
                    Uf = solverTh(m(il),kf,b{il});
                    Pf = eval_pde1D(m(il),Uf,xs);
               
                    if l > l0
                        kc = exp(MM{il-1}*xMC(:,n));
                        Uc = solverTh(m(il-1),kc,b{il-1});
                        Pc = eval_pde1D(m(il-1),Uc,xs);
                    else 
                        Pc = 0;
                    end
                    
                    sum1(il) = sum1(il) + Pf-Pc;
                    sum2(il) = sum2(il) + (Pf-Pc)^2;
            
                end
            
            end
            
        end
        
    end
    
    varMC = (sum2-sum1.^2./N)./(N-1);   
    Ndone = N;
    
    cvar = sum(sqrt(varMC.*cost_l'));
    N = ceil(cvar/eps^2*sqrt(varMC./cost_l'));

    N=max(Ndone,N);
    
end

meanMLMC = sum(sum1./Ndone);
stdMLMC = sqrt(sum(varMC./Ndone));
Ndone
costMLMC = sum(Ndone.*cost_l');
    
end
      