function [ev,ef1] = klsetup1D_matern(lam,nu,sigma2,mkl,h,ngl)
%
% [ev,ef1] = klsetup1D_matern(lam,nu,sigma2,mkl,h,ngl)
%
% Compute KL eigenpairs for Matern covariance function
%
% c(x,y) = 2^(1-nu)/Gamma(nu) * (z)^nu * K_nu(z), z=2*sqrt(nu)*r/lam, r = |x-y|_2
%
% on the spatial domain D = [0,1]
%
% Inputs:
%
% lam:    correlation length (scalar)
%
% nu:     smoothness parameter (scalar)
%
% sigma2: variance (scalar)
%
% mkl:    number of KL eigenpairs
%
% h:      mesh size 
%
% ngl:    number of collocation points
%
% Outputs:
%
% ev:     KL eigenvalues (decreasing magnitude)
%         mkl x 1 vector
%
% ef1:    normalized KL eigenfunctions evaluated at midpoints of mesh 
%         with meshwidth h
%         (1/h) x mkl matrix
%
% ++++++++++++++++++++++++++++++++++
% MLMC project function
% Written 2011-14 by E. Ullmann
% ++++++++++++++++++++++++++++++++++

% (a) Set up collocation nodes
% on reference interval
[x,w] = g_data(ngl);  % in [-1,1] 

% transform nodes and weights to [0,1]
xt = 0.5 * x + 0.5;   % Nodes: 0; 0.5; 1
wt = 0.5 * w;         

% on domain D
X = xt;
W = wt;

% (b) Compute KL eigenpairs
D = diag(sqrt(W));

n = length(X);
R = abs(repmat(X,1,n)-repmat(X,1,n)');
Z = 2*sqrt(nu)*R/lam;

C = sigma2*(2^(1-nu)/gamma(nu))*(Z.^nu).*besselk(nu,Z);
for k=1:ngl, C(k,k) = sigma2; end

S = repmat(sqrt(W)',n,1) .* repmat(sqrt(W),1,n);
A = C .* S;     %A = D * C * D;

[V,EV] = eig(A);

% (c) Order eigenpairs (decreasing magnitude of eigenvalues) and normalize
% eigenfunctions
[ev,ind] = sort(real(diag(EV)),'descend');
V = V(:,ind);
V = D \ V; % back-transformation of eigenvectors

ef = V(:,1:mkl); ev = ev(1:mkl);

norm_ef = sqrt(sum(ef.*ef.*repmat(W,1,mkl),1));

ef = ef./repmat(norm_ef,size(ef,1),1);

% (d) Extract value of eigenfunctions on midpoints of requested mesh

% Use f(y) = {\int c(y,x) * f(x) dx} / lambda

Y = [h/2:h:1-h/2]';  % midpoints of mesh with size h
 
nrows = length(Y); ncols = length(X);

R1 = abs(repmat(Y,1,ncols)-repmat(X,1,nrows)');
Z1 = 2*sqrt(nu)*R1/lam;

C1 = sigma2*(2^(1-nu)/gamma(nu))*(Z1.^nu).*besselk(nu,Z1);
C1 = C1 .* repmat(W,1,nrows)';

ef1 = (C1 * ef) * diag(1./ev); 

return


