/*
 * one_level_amc.hh
 *
 *  Created on: 2013. 11. 29.
 *      Author: parkmh
 */

#ifndef ONE_LEVEL_AMC_HH_
#define ONE_LEVEL_AMC_HH_

#include "../mesh/point.hh"
#include "mc_return_value.hh"
#include <iostream>
#include <functional>
#include <string>

template <class Category,typename T>
class OneLevelMC{
public:
	typedef mc_traits<Category, T> traits;
	typedef T		datatype;
	typedef typename traits::rfgenerator 	rfgenerator;
	typedef typename traits::discretization 	discretization;
	typedef typename discretization::pos		pos;
	typedef typename discretization::N			N;
	typedef typename traits::preconditioner 	preconditioner;
	typedef typename traits::lsolver			lsolver;
	typedef typename traits::qoifun				qoifun;
	typedef typename traits::secondop			secondop;
	OneLevelMC(){ rfg_ = NULL; disc_ = NULL; is_first_amg = true;}
	OneLevelMC(const discretization&, const rfgenerator&);
	~OneLevelMC(){}
	int getID() const {return disc_->nx(); }
	void init(const discretization&, const rfgenerator&);
	mc_return<T> compute() const;

	std::string getDescription() const {
		std::string s = "Monte Carlo Method with Antithetic Variates";
		return s;
	}
	friend std::ostream&
	operator<<(std::ostream& os, const OneLevelMC& olmc){
		os << "[ Discretisation ] " << std::endl << olmc.disc_ << std::endl;
		os << "[ Solver ] " << std::endl << olmc.solver_ << std::endl;
		return os;
	}
private:
	const rfgenerator    	*rfg_;
	const 	discretization 	*disc_;
	mutable preconditioner 	precond_;
	mutable lsolver		 	solver_;
	qoifun				qoifun_;
	mutable CRSMatrix<datatype> 	A_;
	mutable NumVec<datatype> 		b_;
	mutable NumVec<datatype>     v_;
	mutable NumVec<datatype>     a_;
	mutable NumVec<datatype> 	 anti_a_;
	mutable bool    is_first_amg;
	secondop		secondop_;
};

template <class Category, typename T>
OneLevelMC<Category,T>::OneLevelMC(const discretization& disc, const rfgenerator& rfg)
{
	init(disc, rfg);

}

template <class Category, typename T>
void OneLevelMC<Category,T>::init(const discretization& disc, const rfgenerator& rfg)
{
	rfg_ = &rfg;
	disc_ = &disc;
	is_first_amg = true;
}

template <class Category, typename T>
mc_return<T> OneLevelMC<Category,T>::compute() const {
	rfg_->generate(a_,anti_a_);
#ifdef MC_SECOND_OP
	a_ = secondop_(a_);
#endif
	disc_->discretize(A_,b_,a_);
#ifdef AMG_RECYCLE
	if (!is_first_amg){
		precond_.second_init(A_,10,1);
	} else{
		precond_.init(A_,10,1);
		is_first_amg = false;
	}
#else
	precond_.init(A_,10,1);
#endif

	solver_.init(A_,precond_,MAX_SOLVE_CYCLE, SOLVE_TOL);
	v_.rand(b_.size());
	solver_.solve(b_,v_);
	mc_return<T> rval;
	rval.qh = qoifun_(*disc_,v_,a_);
	rval.q2h = 0;

#ifdef MC_SECOND_OP
	anti_a_ = secondop_(anti_a_);
#endif
	disc_->discretize(A_,b_,anti_a_);
#ifdef AMG_RECYCLE
	if (!is_first_amg){
		precond_.second_init(A_,10,1);

	} else{
		precond_.init(A_,10,1);
		is_first_amg = false;
	}
#else
	precond_.init(A_,10,1);
#endif

	solver_.init(A_,precond_,MAX_SOLVE_CYCLE, SOLVE_TOL);
	v_.rand(b_.size());
	solver_.solve(b_,v_);
	rval.qh += qoifun_(*disc_,v_,anti_a_);
	rval.qh /= 2.0;

	return rval;

}


#endif /* ONE_LEVEL_AMC_HH_ */
