/*
 * montecarlo.hh
 *
 *  Created on: 2013. 8. 18.
 *      Author: parkmh
 */

#ifndef MONTECARLO_HH_
#define MONTECARLO_HH_

#include "../util/config.hh"
#include "mc_return_value.hh"
#include <cmath>

template <class MCBASE>
class MC{
public:
	typedef typename MCBASE::datatype datatype;
	MC();
	MC(const MCBASE &, const int);
	~MC();
	void init(const MCBASE &, const int);
	size_t more_samp();
	size_t nsamp() const {return nsamp_;}
	void apply(const size_t &);
	double meanQ() const;
	double varQ() const;
	float stdvQ() const;
	double meanQc() const;
	double varQc() const;
	float stdvQc() const;
	double meanY() const;
	double varY() const;
	float stdvY() const;
private:
	mutable const MCBASE* qoi_;
#ifdef SAVE_QOI
	std::ofstream qoi_file_;
#endif
	size_t nsamp_;
	datatype sum_qoih_;
	datatype sum_qoih2_;
	datatype sum_qoi2h_;
	datatype sum_qoi2h2_;
	datatype sum_y_;
	datatype sum_y2_;
};



template <class MCBASE>
MC<MCBASE>::MC(){
	qoi_ = NULL; nsamp_ = 0;
}
template <class MCBASE>
MC<MCBASE>::MC(const MCBASE & qoi, const int firstnsamp){
	init(qoi,firstnsamp);
}

template <class MCBASE>
MC<MCBASE>::~MC() {
#ifdef SAVE_QOI
	qoi_file_.close();
#endif
}
template <class MCBASE>
void MC<MCBASE>::init(const MCBASE & qoi, const int firstnsamp = 10){
	qoi_ = &qoi;
#ifdef SAVE_QOI
	std::stringstream filename;
	filename << "NX" << qoi_->getID() << ".qoi";
	qoi_file_.open(filename.str().c_str(),std::ios::out);
#endif
	nsamp_ = 0;
	sum_qoih_ = 0;
	sum_qoih2_ = 0;
	sum_qoi2h_ = 0;
	sum_qoi2h2_ = 0;
	sum_y_ = 0;
	sum_y2_ = 0;
	apply(firstnsamp);
}

template <class MCBASE>
void MC<MCBASE>::apply(const size_t & n){
	std::ofstream save;
	save.open("Sample.idx",std::ios::out);
	nsamp_ += n;
	datatype q;
	mc_return<datatype> rval;


	for (size_t i = 0; i < n; i	++){
		save.width(10);
		save << std::left << i;
		save.seekp(save.tellp()-(long)10);

		rval = qoi_->compute();
		while (rval.qh<0|rval.q2h<0){
			rval = qoi_->compute();
		}
#ifdef SAVE_QOI
		qoi_file_ << rval.qh << " " << rval.q2h << std::endl;
#endif
		sum_qoih_ += rval.qh;
		sum_qoih2_ += rval.qh*rval.qh;
		sum_qoi2h_ += rval.q2h;
		sum_qoi2h2_ += rval.q2h*rval.q2h;
		sum_y_ += (rval.qh-rval.q2h);
		sum_y2_ +=  (rval.qh-rval.q2h)*(rval.qh-rval.q2h);
	}
	save.close();
}


template <class MCBASE>
double MC<MCBASE>::meanQ() const {
	return (double)sum_qoih_/(double)nsamp_;
}

template <class MCBASE>
double MC<MCBASE>::varQ() const {
	return (double)(nsamp_*sum_qoih2_-sum_qoih_*sum_qoih_)/(double)(nsamp_*(nsamp_-1));
}

template <class MCBASE>
float MC<MCBASE>::stdvQ() const {
	return std::sqrt((float)varQ());
}

template <class MCBASE>
double MC<MCBASE>::meanQc() const {
	return (double)sum_qoi2h_/(double)nsamp_;
}

template <class MCBASE>
double MC<MCBASE>::varQc() const {
	return (double)(nsamp_*sum_qoi2h2_-sum_qoi2h_*sum_qoi2h_)/(double)(nsamp_*(nsamp_-1));
}

template <class MCBASE>
float MC<MCBASE>::stdvQc() const {
	return std::sqrt((float)varQc());
}

template <class MCBASE>
double MC<MCBASE>::meanY() const {
	return (double)sum_y_/(double)nsamp_;
}

template <class MCBASE>
double MC<MCBASE>::varY() const {
	return (double)(nsamp_*sum_y2_-sum_y_*sum_y_)/(double)(nsamp_*(nsamp_-1));
}

template <class MCBASE>
float MC<MCBASE>::stdvY() const {
	return std::sqrt((float)varY());
}
#endif /* MONTECARLO_HH_ */
