/*
 * temp.hh
 *
 *  Created on: 2013. 9. 2.
 *      Author: parkmh
 */

#ifndef TEMP_HH_
#define TEMP_HH_

void pmgmres_ilu_cr ( int n, int nz_num, int ia[], int ja[], double a[],
		double x[], double rhs[], int itr_max, int mr, double tol_abs,
		double tol_rel )

//****************************************************************************80
//
//  Purpose:
//
//    PMGMRES_ILU_CR applies the preconditioned restarted GMRES algorithm.
//
//  Discussion:
//
//    The matrix A is assumed to be stored in compressed row format.  Only
//    the nonzero entries of A are stored.  The vector JA stores the
//    column index of the nonzero value.  The nonzero values are sorted
//    by row, and the compressed row vector IA then has the property that
//    the entries in A and JA that correspond to row I occur in indices
//    IA[I] through IA[I+1]-1.
//
//    This routine uses the incomplete LU decomposition for the
//    preconditioning.  This preconditioner requires that the sparse
//    matrix data structure supplies a storage position for each diagonal
//    element of the matrix A, and that each diagonal element of the
//    matrix A is not zero.
//
//    Thanks to Jesus Pueblas Sanchez-Guerra for supplying two
//    corrections to the code on 31 May 2007.
//
//
//    This implementation of the code stores the doubly-dimensioned arrays
//    H and V as vectors.  However, it follows the C convention of storing
//    them by rows, rather than my own preference for storing them by
//    columns.   I may come back and change this some time.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license.
//
//  Modified:
//
//    26 July 2007
//
//  Author:
//
//    Original C version by Lili Ju.
//    C++ version by John Burkardt.
//
//  Reference:
//
//    Richard Barrett, Michael Berry, Tony Chan, James Demmel,
//    June Donato, Jack Dongarra, Victor Eijkhout, Roidan Pozo,
//    Charles Romine, Henk van der Vorst,
//    Templates for the Solution of Linear Systems:
//    Building Blocks for Iterative Methods,
//    SIAM, 1994.
//    ISBN: 0898714710,
//    LC: QA297.8.T45.
//
//    Tim Kelley,
//    Iterative Methods for Linear and Nonlinear Equations,
//    SIAM, 2004,
//    ISBN: 0898713528,
//    LC: QA297.8.K45.
//
//    Yousef Saad,
//    Iterative Methods for Sparse Linear Systems,
//    Second Edition,
//    SIAM, 2003,
//    ISBN: 0898715342,
//    LC: QA188.S17.
//
//  Parameters:
//
//    Input, int N, the order of the linear system.
//
//    Input, int NZ_NUM, the number of nonzero matrix values.
//
//    Input, int IA[N+1], JA[NZ_NUM], the row and column indices
//    of the matrix values.  The row vector has been compressed.
//
//    Input, double A[NZ_NUM], the matrix values.
//
//    Input/output, double X[N]; on input, an approximation to
//    the solution.  On output, an improved approximation.
//
//    Input, double RHS[N], the right hand side of the linear system.
//
//    Input, int ITR_MAX, the maximum number of (outer) iterations to take.
//
//    Input, int MR, the maximum number of (inner) iterations to take.
//    MR must be less than N.
//
//    Input, double TOL_ABS, an absolute tolerance applied to the
//    current residual.
//
//    Input, double TOL_REL, a relative tolerance comparing the
//    current residual to the initial residual.
//
{
	double av;
	double *c;
	double delta = 1.0e-03;
	double *g;
	double *h;
	double htmp;
	int i;
	int itr;
	int itr_used;
	int j;
	int k;
	int k_copy;
	double *l;
	double mu;
	double *r;
	double rho;
	double rho_tol;
	double *s;
	int *ua;
	double *v;
	int verbose = 1;
	double *y;

	itr_used = 0;

	c = new double[mr+1];
	g = new double[mr+1];
	h = new double[(mr+1)*mr];
	l = new double[ia[n]+1];
	r = new double[n];
	s = new double[mr+1];
	ua = new int[n];
	v = new double[(mr+1)*n];
	y = new double[mr+1];

	rearrange_cr ( n, nz_num, ia, ja, a );

	diagonal_pointer_cr ( n, nz_num, ia, ja, ua );

	ilu_cr ( n, nz_num, ia, ja, a, ua, l );

	if ( verbose )
	{
		cout << "\n";
		cout << "PMGMRES_ILU_CR\n";
		cout << "  Number of unknowns = " << n << "\n";
	}

	for ( itr = 0; itr < itr_max; itr++ )
	{
		ax_cr ( n, nz_num, ia, ja, a, x, r );

		for ( i = 0; i < n; i++ )
		{
			r[i] = rhs[i] - r[i];
		}

		lus_cr ( n, nz_num, ia, ja, l, ua, r, r );

		rho = sqrt ( r8vec_dot ( n, r, r ) );

		if ( verbose )
		{
			cout << "  ITR = " << itr << "  Residual = " << rho << "\n";
		}

		if ( itr == 0 )
		{
			rho_tol = rho * tol_rel;
		}

		for ( i = 0; i < n; i++ )
		{
			v[0*n+i] = r[i] / rho;
		}

		g[0] = rho;
		for ( i = 1; i < mr + 1; i++ )
		{
			g[i] = 0.0;
		}

		for ( i = 0; i < mr + 1; i++ )
		{
			for ( j = 0; j < mr; j++ )
			{
				h[i*(mr)+j] = 0.0;
			}
		}

		for ( k = 0; k < mr; k++ )
		{
			k_copy = k;

			ax_cr ( n, nz_num, ia, ja, a, v+k*n, v+(k+1)*n );

			lus_cr ( n, nz_num, ia, ja, l, ua, v+(k+1)*n, v+(k+1)*n );

			av = sqrt ( r8vec_dot ( n, v+(k+1)*n, v+(k+1)*n ) );

			for ( j = 0; j <= k; j++ )
			{
				h[j*mr+k] = r8vec_dot ( n, v+(k+1)*n, v+j*n );
				for ( i = 0; i < n; i++ )
				{
					v[(k+1)*n+i] = v[(k+1)*n+i] - h[j*mr+k] * v[j*n+i];
				}
			}
			h[(k+1)*mr+k] = sqrt ( r8vec_dot ( n, v+(k+1)*n, v+(k+1)*n ) );

			if ( ( av + delta * h[(k+1)*mr+k]) == av )
			{
				for ( j = 0; j < k + 1; j++ )
				{
					htmp = r8vec_dot ( n, v+(k+1)*n, v+j*n );
					h[j*mr+k] = h[j*mr+k] + htmp;
					for ( i = 0; i < n; i++ )
					{
						v[(k+1)*n+i] = v[(k+1)*n+i] - htmp * v[j*n+i];
					}
				}
				h[(k+1)*mr+k] = sqrt ( r8vec_dot ( n, v+(k+1)*n, v+(k+1)*n ) );
			}

			if ( h[(k+1)*mr+k] != 0.0 )
			{
				for ( i = 0; i < n; i++ )
				{
					v[(k+1)*n+i] = v[(k+1)*n+i] / h[(k+1)*mr+k];
				}
			}

			if ( 0 < k )
			{
				for ( i = 0; i < k + 2; i++ )
				{
					y[i] = h[i*mr+k];
				}
				for ( j = 0; j < k; j++ )
				{
					mult_givens ( c[j], s[j], j, y );
				}
				for ( i = 0; i < k + 2; i++ )
				{
					h[i*mr+k] = y[i];
				}
			}
			mu = sqrt ( h[k*mr+k] * h[k*mr+k] + h[(k+1)*mr+k] * h[(k+1)*mr+k] );
			c[k] = h[k*mr+k] / mu;
			s[k] = -h[(k+1)*mr+k] / mu;
			h[k*mr+k] = c[k] * h[k*mr+k] - s[k] * h[(k+1)*mr+k];
			h[(k+1)*mr+k] = 0.0;
			mult_givens ( c[k], s[k], k, g );

			rho = fabs ( g[k+1] );

			itr_used = itr_used + 1;

			if ( verbose )
			{
				cout << "  K   = " << k << "  Residual = " << rho << "\n";
			}

			if ( rho <= rho_tol && rho <= tol_abs )
			{
				break;
			}
		}

		k = k_copy;

		y[k] = g[k] / h[k*mr+k];
		for ( i = k - 1; 0 <= i; i-- )
		{
			y[i] = g[i];
			for ( j = i + 1; j < k + 1; j++ )
			{
				y[i] = y[i] - h[i*mr+j] * y[j];
			}
			y[i] = y[i] / h[i*mr+i];
		}
		for ( i = 0; i < n; i++ )
		{
			for ( j = 0; j < k + 1; j++ )
			{
				x[i] = x[i] + v[j*n+i] * y[j];
			}
		}

		if ( rho <= rho_tol && rho <= tol_abs )
		{
			break;
		}
	}

	if ( verbose )
	{
		cout << "\n";;
		cout << "PMGMRES_ILU_CR:\n";
		cout << "  Iterations = " << itr_used << "\n";
		cout << "  Final residual = " << rho << "\n";
	}

	delete [] c;
	delete [] g;
	delete [] h;
	delete [] l;
	delete [] r;
	delete [] s;
	delete [] ua;
	delete [] v;
	delete [] y;

	return;
}


#endif /* TEMP_HH_ */
