/*
 * fvm.hh
 *
 *  Created on: 2013. 8. 15.
 *      Author: parkmh
 */

#ifndef FVM_HH_
#define FVM_HH_
#include "../mesh/point.hh"
#include "../math/crs_matrix.hh"
#include "../math/numvec.hh"
#include "../util/config.hh"
#include <string>

template <class Category, typename T>
class FVM{
public:
	typedef T datatype;
	typedef fvm_traits<Category,T> 			traits;
	const static int dim = traits::dim;
	typedef Point<dim,T> pos;
	typedef Point<dim,size_t> N;


	FVM(){ dof_ = 0; nnz_ = 0;};
	FVM(const pos &, const pos &, const N&);
	void init(const pos &, const pos &, const N&);
	void discretize(CRSMatrix<datatype> &, NumVec<datatype> &, const NumVec<datatype> &) const;
	size_t dof() const{return dof_;}
	size_t nnz() const{return nnz_;}
	N n() const{return N_;}
	pos start() const {return start_;}
	pos end() const {return end_;}
	size_t nx() const{return N_[0];}
	size_t ny() const{return N_[1];}
	size_t nz() const{return N_[2];}
	T hx() const{return h_[0];}
	T hy() const{return h_[1];}
	T hz() const{return h_[2];}
	void save(const char *) const;
	void load(const char *);
	std::string getDescription() const { return std::string("Cell-centred Finite Volume");}

	friend std::ostream&
	operator<<(std::ostream& os, const FVM& fvm){
		os << dim << "D Cell-centred Finite Volume Method" << std::endl;

		os << "start : " << fvm.start_ << ", end : " << fvm.end_ << std::endl;
		os << "N : " << fvm.N_ << "h : " << fvm.h_ << std::endl;
		os << "dof : " << fvm.dof_ << ", nnz : " << fvm.nnz_ << std::endl;
		os << "[ FVM -> Boundary Condition ] " << std::endl;
		os << "[1] x = " << fvm.start_[0] << " : ";
		if (fvm_traits<Category,T>::x0 == DIRICHLET) os << "DIRICHLET";
		else if (fvm_traits<Category,T>::x0 == NEUMANN) os << "NEUMANN";
		os << std::endl << "[2] x = " << fvm.end_[0] << " : ";
		if (fvm_traits<Category,T>::x1 == DIRICHLET) os << "DIRICHLET";
		else if (fvm_traits<Category,T>::x1 == NEUMANN) os << "NEUMANN";
		os <<std::endl;
		if (dim > 1) {
			os <<  "[3] y = " << fvm.start_[1] << " : ";
			if (fvm_traits<Category,T>::y0 == DIRICHLET) os << "DIRICHLET";
			else if (fvm_traits<Category,T>::y1 == NEUMANN) os << "NEUMANN";
			os  << "[4] y = " << fvm.end_[1] << " : ";
			if (fvm_traits<Category,T>::y1 == DIRICHLET) os << "DIRICHLET";
			else if (fvm_traits<Category,T>::y1 == NEUMANN) os << "NEUMANN";
			os <<std::endl;
		}
		if (dim > 2) {
			os << "[5] z = " << fvm.start_[2] << " : ";
			if (fvm_traits<Category,T>::z0 == DIRICHLET) os << "DIRICHLET";
			else if (fvm_traits<Category,T>::z1 == NEUMANN) os << "NEUMANN";
			os << "[6] z = " << fvm.end_[2] << " : ";
			if (fvm_traits<Category,T>::z1 == DIRICHLET) os << "DIRICHLET";
			else if (fvm_traits<Category,T>::z1 == NEUMANN) os << "NEUMANN";
			os <<std::endl;
		}

		return os;
	}
private:
	pos start_;
	pos end_;
	pos h_;
	N N_;
	size_t dof_;
	size_t nnz_;

	void discretize1d(CRSMatrix<datatype> &, NumVec<datatype> &, const NumVec<datatype> &) const;
	void discretize2d(CRSMatrix<datatype> &, NumVec<datatype> &, const NumVec<datatype> &) const;
	void discretize3d(CRSMatrix<datatype> &, NumVec<datatype> &, const NumVec<datatype> &) const;
};

template <class Category, typename T>
FVM<Category, T>::FVM(const pos &start, const pos &end, const N& n){
	init(start,end,n);
}

template <class Category, typename T>
void FVM<Category, T>::init(const pos &start, const pos &end, const N& n){
	start_ = start;
	end_   = end;
	N_ = n;
	h_ = end-start;

	for (int i = 0; i < dim; i++){
		h_[i]/= (T)N_[i];
	}
	switch (dim){
	case 1:
		nnz_ = 3*N_[0]-2;
		dof_ = N_[0];
		break;
	case 2:
		dof_ = N_[0]*N_[1];
		nnz_ = 5*dof_ -2*(N_[0]+N_[1]);
		break;
	case 3:
		dof_ = N_[0]*N_[1]*N_[2];
		nnz_ = 7*dof_ - 2*(N_[0]*N_[1] + N_[1]*N_[2] + N_[2]*N_[0]);
		break;
	default:
		break;
	}
}

template <class Category, typename T>
void FVM<Category, T>::discretize(CRSMatrix<datatype> &A,
		NumVec<datatype>& b,
		const NumVec<datatype>& a) const{
	A.resizeia(dof_+1);
	A.resizeja(nnz_);
	A.resizeaa(nnz_);
	A.setSize(dof_,dof_,nnz_);
	switch (dim){
	case 1:
		discretize1d(A,b,a);
		break;
	case 2:
		discretize2d(A,b,a);
		break;
	case 3:
		discretize3d(A,b,a);
		break;
	default:
		break;
	}
}

template <class Category, typename T>
void FVM<Category, T>::discretize1d(CRSMatrix<datatype> &A,
		NumVec<datatype>& b, const NumVec<datatype>& a) const{
	b.resize(N_[0]);
	T value;
	T center;
	typename traits::boundary bndfun;
	typename traits::force force;

	T hx2 = h_[0]*h_[0];
	T hxh = h_[0]/2.0;

	A.ia(0) = 0;
	int index = 0;
	pos position;

	/*
	 * Boundary (X = 0)
	 */
	// Left
	center = 0;
	position[0] = hxh;
	b[0] = force(position);

	if (traits::x0 == DIRICHLET){	// Dirichlet boundary condition
		center += a[0]*2.0/(hx2);
		b[0] += bndfun(start_)*a[0]*2.0/(hx2);
	} else {
		b[0] -= bndfun(start_)/h_[0];
	}
	// Right
	value = 2.0/((1.0/a[0]+1.0/a[1])*hx2);
	center += value;
	A.ja(index) = 1;
	A.aa(index) = -value;
	index++;

	// Center
	A.ja(index) = 0;
	A.aa(index) = center;
	index++;
	A.ia(1) = index;

	for (size_t i = 1; i < N_[0] - 1; i++){
		position[0] += h_[0];
		b[i] = force(position);
		center = 0;

		// Left
		value = 2.0/((1.0/a[i]+1.0/a[i-1])*hx2);
		center += value;
		A.ja(index) = i-1;
		A.aa(index) = -value;
		index++;

		// Right
		value = 2.0/((1.0/a[i]+1.0/a[i+1])*hx2);
		center += value;
		A.ja(index) = i+1;
		A.aa(index) = -value;
		index++;

		// Center
		A.ja(index) = i;
		A.aa(index) = center;
		index++;
		A.ia(i+1) = index;
	}

	/*
	 * Boundary (X = 1)
	 */
	// Right
	center = 0;
	position[0] += h_[0];
	b[N_[0] - 1] = force(position);

	if (traits::x1 == DIRICHLET){
		center += a[N_[0] - 1]*2.0/hx2;
		b[N_[0] - 1] += bndfun(end_)*a[N_[0] - 1]*2.0/hx2;
	} else {
		b[N_[0] - 1] -= bndfun(end_)/h_[0];
	}

	// Left
	value = 2.0/((1.0/a[N_[0]-1]+1.0/a[N_[0] - 2 ])*hx2);
	center += value;
	A.ja(index) = N_[0] - 2;
	A.aa(index) = -value;
	index++;

	//Center
	A.ja(index) = N_[0] - 1;
	A.aa(index) = center;
	index++;
	A.ia(N_[0]) = index;

}

template <class Category, typename T>
void FVM<Category, T>::discretize2d(CRSMatrix<datatype> &A, NumVec<datatype>& b, const NumVec<datatype>& a) const{
	//
	typename traits::boundary bndfun;
	typename traits::force force;

	size_t i,ix,iy;
	size_t Nxy = N_[0] * N_[1];

	b.resize(Nxy);
	T value;
	T center;

	T hx2 = h_[0]*h_[0];
	T hxh = h_[0]/2.0;
	T hy2 = h_[1]*h_[1];
	T hyh = h_[1]/2.0;

	A.ia(0) = 0;
	int index = 0;
	T cx0 = start_[0] + hxh;
	T cy0 = start_[1] + hyh;
	pos position;
	pos bndpos;
	for ( i = 0; i < Nxy; i++)
	{
		ix = (size_t)fmod((double)i,(double)N_[0]);
		iy = (size_t)floor((double)i/(double)N_[0]);
		position[0] = cx0 + h_[0]*ix;
		position[1] = cy0 + h_[1]*iy;
		b[i] = force(position);

		center = 0.0;
		if ( ix == 0 )
		{
			bndpos[0] = start_[0];
			bndpos[1] = position[1];

			if (traits::x0 == DIRICHLET) 	// Dirichlet Bnd
			{
				center = center + a[i]*2.0/hx2;
				b[i] += bndfun(bndpos)*a[i]*2.0/(hx2);
			}
			else {
				b[i] -= bndfun(bndpos)/h_[0];
			}
		}
		else
		{
			value = 2.0/((1.0/a[i]+1.0/a[i-1])*hx2);
			center += value;
			A.ja(index) = i - 1;
			A.aa(index) = -value;
			index++;
		}
		if ( ix == N_[0] - 1 )
		{
			bndpos[0] = end_[0];
			bndpos[1] = position[1];

			if (traits::x1 == DIRICHLET) 	// Dirichlet Bnd
			{
				center += a[i]*2.0/(hx2);
				b[i] += bndfun(bndpos)*a[i]*2.0/(hx2);
			}
			else {
				b[i] -= bndfun(bndpos)/h_[0];
			}
		}
		else
		{
			value = 2.0/((1.0/a[i]+1.0/a[i+1])*hx2);
			center += value;
			A.ja(index)= i + 1;
			A.aa(index) = -value;
			index++;
		}

		/* South */
		if ( iy == 0 )
		{
			bndpos[0] = position[0];
			bndpos[1] = start_[1];

			if (traits::y0 == DIRICHLET) 	// Dirichlet Bnd
			{
				center +=  a[i]*2.0/(hy2);
				b[i] += bndfun(bndpos)*a[i]*2.0/(hy2);
			}
			else {
				b[i] -= bndfun(bndpos)/h_[1];
			}
		}
		else
		{
			value = 2.0/((1.0/a[i]+1.0/a[(iy-1)*N_[0]+ix])*hy2);
			center += value;
			A.ja(index) = i - N_[0];
			A.aa(index) = -value;
			index++;
		}

		/* North */
		if ( iy == N_[1]-1 )
		{
			bndpos[0] = position[0];
			bndpos[1] = end_[1];

			if (traits::y1 == DIRICHLET) 	// Dirichlet Bnd
			{
				center += a[i]*2.0/(hy2);
				b[i] += bndfun(bndpos)*a[i]*2.0/(hy2);
			}
			else {
				b[i] -= bndfun(bndpos)/h_[1];
			}
		}
		else
		{
			value = 2.0/((1.0/a[i]+1/a[i + N_[0]])*hy2);
			center += value;
			A.ja(index) = i + N_[0];
			A.aa(index) = -value;
			index++;
		}
		A.ja(index) = i;
		A.aa(index) = center;
		index++;
		A.ia(i + 1) = index;
	}
}

template <class Category, typename T>
void FVM<Category, T>::discretize3d(CRSMatrix<datatype> &A, NumVec<datatype>& b, const NumVec<datatype>& a) const{

	typename traits::boundary bndfun;
	typename traits::force force;
	size_t i, ix,iy,iz;
	size_t Nxy = N_[0]*N_[1];
	size_t Nxyz = Nxy*N_[2];

	b.resize(Nxyz);
	T value;
	T center;


	T hx2 = h_[0]*h_[0];
	T hxh = h_[0]/2.0;
	T hy2 = h_[1]*h_[1];
	T hyh = h_[1]/2.0;
	T hz2 = h_[2]*h_[2];
	T hzh = h_[2]/2.0;

	A.ia(0) = 0;
	int index = 0;
	T cx0 = start_[0] + hxh;
	T cy0 = start_[1] + hyh;
	T cz0 = start_[2] + hzh;

	pos position;
	pos bndpos;


	for ( i = 0; i < Nxyz; i++)
	{
		ix = (size_t)fmod(fmod((double)i,(double)Nxy),(double)N_[0]);
		iy = (size_t)floor(fmod((double)i,(double)Nxy)/(double)N_[0]);
		iz = (size_t)floor((double)i/(double)Nxy);
		position[0] = cx0 + h_[0]*ix;
		position[1] = cy0 + h_[1]*iy;
		position[2] = cz0 + h_[2]*iz;

		b[i] = force(position);
		center = 0.0;

		if ( ix == 0 )
		{
			bndpos[0] = start_[0];
			bndpos[1] = position[1];
			bndpos[2] = position[2];

			if (traits::x0 == DIRICHLET) 	// Dirichlet Bnd
			{
				center += a[i]*2.0/(hx2);
				b[i] += bndfun(bndpos)*a[i]*2.0/(hx2);
			}
			else {
				b[i] -= bndfun(bndpos)/h_[0];
			}
		}
		else
		{
			value = 2.0/((1.0/a[i]+1.0/a[i -1])*hx2);
			center += value;
			A.ja( index ) = i - 1;
			A.aa( index ) = -value;
			index++;
		}

		if ( ix == N_[0] - 1 )
		{
			bndpos[0] = end_[0];
			bndpos[1] = position[1];
			bndpos[2] = position[2];
			if (traits::x1 == DIRICHLET) 	// Dirichlet Bnd
			{
				center += a[i]*2.0/(hx2);
				b[i] += bndfun(bndpos)*a[i]*2.0/(hx2);
			}
			else {
				b[i] -= bndfun(bndpos)/h_[0];
			}
		}
		else
		{
			value = 2.0/((1.0/a[i]+1.0/a[i+1])*hx2);
			center += value;
			A.ja( index ) = i + 1;
			A.aa( index ) = -value;
			index++;
		}

		/* South */
		if ( iy == 0 )
		{
			bndpos[0] = position[0];
			bndpos[1] = start_[1];
			bndpos[2] = position[2];

			if (traits::y0 == DIRICHLET) 	// Dirichlet Bnd
			{
				center += a[i]*2.0/(hy2);
				b[i] += bndfun(bndpos)*a[i]*2.0/(hy2);
			}
			else {
				b[i] -= bndfun(bndpos)/h_[1];
			}
		}
		else
		{
			value = 2.0/((1.0/a[i]+1.0/a[i - N_[0]])*hy2);
			center += value;
			A.ja( index ) = i - N_[0];
			A.aa( index ) = -value;
			index++;
		}

		/* North */
		if ( iy == N_[1]-1 )
		{
			bndpos[0] = position[0];
			bndpos[1] = end_[1];
			bndpos[2] = position[2];

			if (traits::y1 == DIRICHLET) 	// Dirichlet Bnd
			{
				center += a[i]*2.0/(hy2);
				b[i] += bndfun(bndpos)*a[i]*2.0/(hy2);
			}
			else {
				b[i] -= bndfun(bndpos)/h_[1];
			}
		}
		else
		{
			value = 2.0/((1.0/a[i]+1.0/a[i + N_[0]])*hy2);
			center += value;
			A.ja( index ) = i +N_[0];
			A.aa( index ) = -value;
			index++;
		}

		if ( iz == 0 )
		{
			bndpos[0] = position[0];
			bndpos[1] = position[1];
			bndpos[2] = start_[2];

			if (traits::z0 == DIRICHLET) 	// Dirichlet Bnd
			{
				center += a[i]*2.0/(hz2);
				b[i] += bndfun(bndpos)*a[i]*2.0/(hz2);
			}
			else {
				b[i] -= bndfun(bndpos)/h_[2];
			}
		}
		else
		{
			value = 2.0/((1.0/a[i]+1.0/a[i-Nxy])*hz2);
			center += value;
			A.ja( index ) = i -Nxy;
			A.aa( index ) = -value;
			index++;
		}

		if ( iz == N_[2]-1 )
		{
			bndpos[0] = position[0];
			bndpos[1] = position[1];
			bndpos[2] = end_[2];
			if (traits::z1 == DIRICHLET) 	// Dirichlet Bnd
			{
				center += a[i]*2.0/(hz2);
				b[i] += bndfun(bndpos)*a[i]*2.0/(hz2);
			}
			else {
				b[i] -= bndfun(bndpos)/h_[2];
			}
		}
		else
		{
			value = 2.0/((1.0/a[i]+1.0/a[i+Nxy])*hz2);
			center += value;
			A.ja( index ) = i +Nxy;
			A.aa( index ) = -value;
			index++;
		}
		A.ja( index ) = i;
		A.aa( index ) = center;
		index++;
		A.ia(i+1) = index;
	}
}

template <class Category, typename T>
void FVM<Category, T>::save(const char* filename) const {
	size_t i;
	std::ofstream save;
	save.open(filename,std::ios::out);
	save << dim << std::endl;
	for (i = 0; i < dim; i++){
		save << start_[i] << " ";
	}
	save << std::endl;
	for (i = 0; i < dim; i++){
		save << end_[i] << " ";
	}
	save << std::endl;
	for (i = 0; i < dim; i++){
		save << N_[i] << " " ;
	}
	save << std::endl;
	save.close();
}

template <class Category, typename T>
void FVM<Category, T>::load(const char* filename) {
	std::ifstream load;
	size_t i ;
	pos start, end;
	N n;
	load.open(filename,std::ios::in);
	if (!load){
		std::cerr << "\n" ;
		std::cerr << "LOAD - Fatal error!\n";
		std::cerr << " Could not open the file: \"" << filename << "\"\n";
		abort();
	}
	int input_dim;
	load >> input_dim;
	if (dim != input_dim){
		std::cerr << "\n" ;
		std::cerr << "LOAD - Fatal error!\n";
		std::cerr << " dimension must be same\n";
		abort();
	}

	for (i = 0; i < dim; i++){
		load >> start[i];
	}
	for (i = 0; i < dim; i++){
		load >> end[i];
	}
	for (i = 0; i < dim; i++){
		load >> n[i];
	}
	init(start,end,n);
	load.close();
}
#endif /* FVM_HH_ */
