/*
 * full_matrix.hh
 *
 *  Created on: 2013. 9. 20.
 *      Author: parkmh
 */

#ifndef FULL_MATRIX_HH_
#define FULL_MATRIX_HH_

#define DEFAULT 		"copy"
#define TRANSPOSE 	"transpose"
#define TINY 			1.0e-20;

#include "../math/numvec.hh"
#include <iostream>
#include <cstring>
#include <cmath>


template <class T> class FMatrix;
template <class T> double determinant(const FMatrix<T>&);

template <class T>
class FMatrix{
public:
	FMatrix();
	FMatrix(const size_t nrow, const size_t ncol);
	FMatrix(const size_t n);

	FMatrix(const FMatrix &, const char *);

	~FMatrix();

	const FMatrix& operator=(const FMatrix& );
	const FMatrix& operator=(const T& );

	FMatrix transpose() const { return FMatrix<T>(*this,"transpose"); }
	size_t row() const {return nrow_;}
	size_t col() const {return ncol_;}
	void resize(const size_t nrow, const size_t ncol);


	FMatrix inv() const;
	FMatrix chol() const;
	T det() const ;
	T& operator() (size_t i, size_t j) {return value_[i][j];}
	const T& operator() (size_t i, size_t j) const{return value_[i][j]; }

	friend std::ostream&
	operator<< (std::ostream& os, const FMatrix &mat){
		os << "[ " << mat.nrow_ << "-by-" << mat.ncol_ << " ] MATRIX (full)" <<std::endl;
		size_t i, j;
		for (i = 0; i < mat.nrow_; i++){
			for (j = 0; j < mat.ncol_; j++){
				os << "[" << i << "][" <<  j << "] : "
						<< mat.value_[ i ][ j ] << std::endl;
			}
		}
		return os;
	}
protected:
	T **value_;
	size_t nrow_;
	size_t ncol_;
	void delete_container();
	void create_container();
	void ludcmp(FMatrix &, NumVec<size_t>& , T* ) const;
	void lubksb( NumVec<T> & , const NumVec<size_t> &) const;

};

template <class T>
FMatrix<T>::FMatrix(){
	nrow_ = 0;
	ncol_ = 0;
	value_ = 0;
}

template <class T>
FMatrix<T>::FMatrix(const size_t nrow, const size_t ncol) : ncol_(ncol), nrow_(nrow){
	create_container();
	for (size_t i = 0; i < nrow_; i++){
		for (size_t j = 0; j < ncol_; j++){
			value_[i][j] = 0;
		}
	}
}

template <class T>
FMatrix<T>::FMatrix(const size_t n) : ncol_(n), nrow_(n){
	create_container();
	for (size_t i = 0; i < nrow_; i++){
		for (size_t j = 0; j < ncol_; j++){
			value_[i][j] = 0;
		}
	}
}

template <class T>
FMatrix<T>::FMatrix(const FMatrix<T> &A, const char* opt = DEFAULT){
	if (!strcmp(opt,DEFAULT)){
		nrow_ = A.nrow_;
		ncol_ = A.ncol_;

		create_container();

		for (size_t i = 0; i < nrow_; i++){
			for (size_t j = 0; j < ncol_; j++){
				value_[i][j] = A.value_[i][j];
			}
		}
	} else if (!strcmp(opt,TRANSPOSE)){
		nrow_ = A.ncol_;
		ncol_ = A.nrow_;
		create_container();

		for (size_t i = 0; i < nrow_; i++){
			for (size_t j = 0; j < ncol_; j++){
				value_[i][j] = A.value_[j][i];
			}
		}
	}
}

template <class T>
FMatrix<T>::~FMatrix(){
	delete_container();
}

template <class T>
const FMatrix<T>& FMatrix<T>::operator=(const FMatrix<T> &rhs){
	if (this != &rhs){
		if (nrow_ != rhs.nrow_ | ncol_ != rhs.ncol_){
			delete_container();
			nrow_ = rhs.nrow_;
			ncol_ = rhs.ncol_;
			create_container();
		}
		for (size_t i = 0; i < nrow_; i++){
			std::copy(rhs.value_[i], rhs.value_[i]+ncol_, value_[i]);
		}
	}
	return *this;
}

template <class T>
const FMatrix<T>& FMatrix<T>::operator =(const T &rhs){
	for (size_t i = 0; i < nrow_; i++){
		std::fill(value_[i], value_[i]+ncol_,rhs);
	}
	return *this;
}

template <class T>
void FMatrix<T>::resize(const size_t nrow, const size_t ncol){
	if (nrow_ != nrow | ncol_ != ncol){
		delete_container();

		nrow_ = nrow;
		ncol_ = ncol;

		create_container();
	}
}

template <class T>
void FMatrix<T>::delete_container(){
	for (size_t i = 0; i < nrow_; i++){
		delete [] value_[i];
	}
	delete [] value_;
}

template <class T>
void FMatrix<T>::create_container(){
	value_ = new T*[ nrow_ ];
	size_t i, j;
	for (i = 0; i < nrow_; i++){
		*(value_+i) = new T[ ncol_ ];
	}
}


template <class T>
FMatrix<T> FMatrix<T>::inv() const {

	double d = det();
	if (nrow_ == ncol_ && d != 0.0){
		FMatrix<T> res(nrow_,ncol_);

		//		res.resize(nrow_, ncol_);
		if (nrow_ == 1){
			res(0,0) = (T)1/value_[0][0];
		} else if (nrow_ == 2){
			res(0,0) = value_[1][1]/(T)d;
			res(0,1) = -value_[0][1]/(T)d;
			res(1,0) = -value_[1][0]/(T)d;
			res(1,1) = value_[0][0]/(T)d;
		} else {
			FMatrix<T> LU;
			NumVec<size_t> indx;
			T flag;
			NumVec<T> e;
			e.resize(nrow_);
			ludcmp(LU,indx,&flag);
			for (size_t j = 0; j < ncol_; j++){
				e = 0.0;
				e[j] = 1.0;
				LU.lubksb(e,indx);
				for (size_t i = 0; i < nrow_; i++){
					res(i,j) = e[i];
				}
			}

		}
		return res;

	} else {
		if (nrow_ == ncol_){
			throw std::out_of_range("Determinant must be nonzero.");
		}
		else
			throw std::out_of_range("Matrix must be square.");
	}
}

template <class T>
FMatrix<T> FMatrix<T>::chol() const {
	int i,j,k;
	T sum;
	FMatrix<T> L = *this;;
	for (i = 0; i < nrow_; i++){
		for (j =  i; j < nrow_; j++){
			for (sum = L.value_[i][j], k = i-1; k >= 0; k--){
				sum -= L.value_[i][k]*L.value_[j][k];
			}
			if (i == j){
				if (sum <= 0.0)
					throw std::out_of_range("chol failed");
				L(i,i) = sqrt(sum);
			} else {
				L(j,i) = sum/L(i,i);
			}
		}

	}
	for (i = 0; i < nrow_-1; i++){
		for (j = i+1; j < nrow_;j++){
			L(i,j) = 0.0;
		}
	}
	return L;
}
template <class T>
FMatrix<T> operator+(const FMatrix<T>& A, const FMatrix<T>& B){
	if (A.row()!= B.row()|A.col()!= B.col()){
		throw std::out_of_range("Matrix size must agree");
	}
	FMatrix<T> C;
	C.resize(A.row(),A.col());
	for (size_t i = 0 ; i < A.row(); i++){
		for (size_t j = 0; j < A.col(); j++){
			C(i,j) = A(i,j)+ B(i,j);
		}
	}
	return C;
}

template <class T>
FMatrix<T> operator-(const FMatrix<T>& A, const FMatrix<T>& B){
	if (A.row()!= B.row()|A.col()!= B.col()){
		throw std::out_of_range("Matrix size must agree");
	}
	FMatrix<T> C;
	C.resize(A.row(),A.col());
	for (size_t i = 0 ; i < A.row(); i++){
		for (size_t j = 0; j < A.col(); j++){
			C(i,j) = A(i,j)- B(i,j);
		}
	}
	return C;
}

template <class T>
FMatrix<T> operator*(const FMatrix<T>& A, const FMatrix<T>& B){
	if (A.col()!= B.row()){
		throw std::out_of_range("Matrix size must agree(*).");
	}
	FMatrix<T> C;
	C.resize(A.row(),B.col());
	for (size_t i = 0; i < A.row(); i++){
		for (size_t j = 0; j < B.col(); j++){
			C(i,j) = 0.0;
			for (size_t k = 0; k < A.col(); k++){
				C(i,j) += A(i,k) * B(k,j);
			}
		}
	}
	return C;
}

template <class T>
void FMatrix<T>::ludcmp(FMatrix<T>& LU, NumVec<size_t>&indx, T* d) const{
	if (nrow_ == ncol_){
		size_t i, imax,j,k;
		T big, dum,sum,temp;
		NumVec<T> vv;			// vv stores the implicit scaling of each row
		vv.resize(nrow_);
		indx.resize(nrow_);
		for (size_t i = 0; i < nrow_; i++){
			indx[i] = i;
		}
		LU = *this;				// make a copy to LU
		*d = (T)1;

		for (i = 0; i < nrow_; i++ ){
			big = 0.0;
			for (j = 0; j < ncol_; j++){
				if ((temp = fabs(LU.value_[i][j]))>big) big = temp;
			}
			if (big == 0.0) throw std::out_of_range("Singular matrix in routine ludcmp.");
			// No nonzero largest element.
			vv[i] = 1.0/big;		// save th scaling
		}

		for (j = 0; j < ncol_; j++){		// This is the loop over columns of Crout's methos
			for (i = 0; i < j; i++) {
				sum = LU.value_[i][j];
				for (k = 0; k < i; k++) {
					sum -= LU.value_[i][k]*LU.value_[k][j];
				}
				LU.value_[i][j] = sum;
			}
			big = 0.0;		// Initialize for the search for largest pivot element

			for (i = j; i < nrow_; i++){
				sum = LU.value_[i][j];
				for (k = 0; k < j; k++){
					sum -= LU.value_[i][k]*LU.value_[k][j];
				}
				LU.value_[i][j] = sum;
				if ( (dum=vv[i]*fabs(sum)) >= big){
					big = dum;
					imax = i;
				}
			}
			if (j != imax){
				for (k = 0; k < nrow_; k++){
					dum = LU.value_[imax][k];
					LU.value_[imax][k] = LU.value_[j][k];
					LU.value_[j][k] = dum;
				}
				*d = -(*d);
				std::swap(vv[imax],vv[j]);
			}
//			std::swap(indx[j],indx[imax]);
			indx[j] = imax;
			if (LU.value_[j][j] == 0.0) LU.value_[j][j] = TINY;
			if (j!= ncol_-1){
				dum = 1.0/LU.value_[j][j];
				for (i = j+1; i < nrow_; i++) {
					LU.value_[i][j] *= dum;
				}
			}
		}

	}	else {
		throw std::out_of_range("Matrix must be square");
	}

}

template <class T>
void FMatrix<T>::lubksb(NumVec<T> &b , const NumVec<size_t> &indx) const{
	size_t i,ip,j;
	int ii = -1;
	T sum;
	for (i = 0; i < nrow_; i++){
		ip = indx[i];
		sum = b[ip];

		b[ip] = b[i];
		if (ii>=0)
			for (j = ii; j <i;j++) sum -= value_[i][j]*b[j];
		else if (sum) ii = i;
		b[i] = sum;
	}
	for (i = nrow_-1; i >= 0; i--){
		sum = b[i];
		for (j = i+1; j < nrow_; j++){
			sum -= value_[i][j]*b[j];
		}
		b[i] = sum/value_[i][i];
		if (i == 0) break;
	}
}


template <class T>
T FMatrix<T>::det() const {

	double d = 0.0;
	if (nrow_ == ncol_){
		if (nrow_ == 1)
			d = value_[0][0];
		else if (nrow_ == 2){
			d = value_[0][0]*value_[1][1] - value_[0][1]*value_[1][0];
		}
		else{
			FMatrix<T> LU;
			NumVec<size_t> indx;
			this->ludcmp(LU,indx,&d);
			for (size_t i = 0; i < nrow_; i++) d *= LU.value_[i][i];
		}
	} else {
		throw std::out_of_range("Matrix must be square");
	}
	return d;
}


#endif /* FULL_MATRIX_HH_ */
