/*
 * point.hh
 *
 *  Created on: 2013. 7. 28.
 *      Author: parkmh
 */

/*
 * Changelog
 	 * v.0.1 Added point.hh
 	 *
 */

#ifndef POINT_HH_
#define POINT_HH_

#include <cmath>
#include <ostream>
#include <stdexcept>


#include <iostream>

template<int dim, class T> class Point;
typedef Point<1,float> Pf1;
typedef Point<2,float> Pf2;
typedef Point<3,float> Pf3;

typedef Point<1,double> Pd1;
typedef Point<2,double> Pd2;
typedef Point<3,double> Pd3;

template<int dim, class T> class Point{
public:
	typedef T pointtype;
	static const int Dim = dim;
	Point(const T& );
	Point(T pos_in[]){ for (int i = 0; i < dim; i++) pos[i] = pos_in[i];};
	Point(const Point&);
	Point& operator=(const Point&);
	Point& operator=(const T&);


	friend std::ostream&
	operator<<(std::ostream& os, const Point &p){
		os << "(" << p.pos[0];
		for (int i = 1; i < dim; i++) os << ", " << p.pos[i];
		os << ")";
		return os;
	}

	T prod() const { T p = pos[0]; for (int i = 1; i < dim; i++) p *= pos[i]; return p;}

	const Point &operator+=(const Point&);
	const Point &operator-=(const Point&);
	const Point &operator*=(const Point&);
	const Point &operator/=(const Point&);


	const Point &operator+=(const T&);
	const Point &operator-=(const T&);
	const Point &operator*=(const T&);
	const Point &operator/=(const T&);


	T& operator[](size_t n){
		if (n>=dim)
			throw std::out_of_range("Bad point index");
		return pos[n];
	}

	const T& operator[](size_t n) const{
		if (n>=dim)
			throw std::out_of_range("Bad point index");
		return pos[n];
	}
private:
	T pos[dim];
};

template<int dim, class T>
Point<dim,T>::Point(const T &val = 0.0){
	for (int i = 0; i < dim; i++) pos[i] = val;
}

template<int dim, class T>
Point<dim,T>::Point(const Point &p){
	for (int i = 0; i < dim; i++) pos[i] = p.pos[i];
}

template<int dim, class T>
Point<dim,T>& Point<dim,T>::operator=(const Point<dim,T> &p){
	if (this != &p){
		for(int i = 0; i < dim; i++) {
			std::cout << i << ": " << p.pos[i] << std::endl;
			pos[i] = p.pos[i];
		}
	}
	return *this;
}

template<int dim, class T>
Point<dim,T>& Point<dim,T>::operator=(const T &pval){
	for(int i = 0; i < dim; i++) pos[i] = pval;
	return *this;
}


template<int dim, class T>
const Point<dim,T>& Point<dim,T>::operator +=(const Point& rhs){
	for(int i = 0; i < dim; i++){
		pos[i] += rhs[i];
	}
	return *this;
}

template<int dim, class T>
const Point<dim,T>& Point<dim,T>::operator -=(const Point& rhs){
	for(int i = 0; i < dim; i++){
		pos[i] -= rhs[i];
	}
	return *this;
}

template<int dim, class T>
const Point<dim,T>& Point<dim,T>::operator *=(const Point& rhs){
	for(int i = 0; i < dim; i++){
		pos[i] *= rhs[i];
	}
	return *this;
}

template<int dim, class T>
const Point<dim,T>& Point<dim,T>::operator /=(const Point& rhs){

	for(int i = 0; i < dim; i++){
		if (rhs[i] == 0)
			throw std::out_of_range("Can't divide by 0");

		pos[i] /= rhs[i];
	}
	return *this;
}

template<int dim, class T>
const Point<dim,T>& Point<dim,T>::operator +=(const T& rhs){
	for(int i = 0; i < dim; i++){
		pos[i] += rhs;
	}
	return * this;
}

template<int dim, class T>
const Point<dim,T>& Point<dim,T>::operator -=(const T& rhs){
	for(int i = 0; i < dim; i++){
		pos[i] -= rhs;
	}
	return *this;
}

template<int dim, class T>
const Point<dim,T>& Point<dim,T>::operator *=(const T& rhs){
	for(int i = 0; i < dim; i++){
		pos[i] *= rhs;
	}
	return *this;
}

template<int dim, class T>
const Point<dim,T>& Point<dim,T>::operator /=(const T& rhs){
	if (rhs == 0)
		throw std::out_of_range("Can't divide by 0");

	for(int i = 0; i < dim; i++){
		pos[i] /= rhs;
	}
	return *this;
}

template<int dim, class T>
const Point<dim,T> operator+(const Point<dim,T> &p1, const Point<dim,T> &p2){
	return Point<dim,T>(p1)+= p2;
}

template<int dim, class T>
const Point<dim,T> operator-(const Point<dim,T> &p1, const Point<dim,T> &p2){
	return Point<dim,T>(p1)-= p2;
}

template<int dim, class T>
const Point<dim,T> operator*(const Point<dim,T> &p1, const Point<dim,T> &p2){
	return Point<dim,T>(p1)*= p2;
}

template<int dim, class T>
const Point<dim,T> operator/(const Point<dim,T> &p1, const Point<dim,T> &p2){
	return Point<dim,T>(p1)/= p2;
}

template<int dim, class T>
const Point<dim,T> operator+(const Point<dim,T> &p1, const T &rhs){
	return Point<dim,T>(p1)+= rhs;
}

template<int dim, class T>
const Point<dim,T> operator-(const Point<dim,T> &p1, const T &rhs){
	return Point<dim,T>(p1)-= rhs;
}

template<int dim, class T>
const Point<dim,T> operator*(const Point<dim,T> &p1, const T &rhs){
	return Point<dim,T>(p1)*= rhs;
}

template<int dim, class T>
const Point<dim,T> operator/(const Point<dim,T> &p1, const T &rhs){
	return Point<dim,T>(p1)/= rhs;
}

template<int dim, class T>
const Point<dim,T> operator/(const T& val, const Point<dim,T> &p2){
	Point<dim,T> p;
	for (size_t i = 0; i < dim; i++){
		p[i] = val/p2[i];
	}
	return p;
}

template<int dim, class T>
const Point<dim,T> operator-(const Point<dim,T>& p){
	return Point<dim,T>(p)*= -1.0;
}

template<int dim, class T>
const T sum(const Point<dim,T> & p){
	T sum = 0.0;
	for (int i = 0; i < dim;i++){
		sum+=p[i];
	}
	return;
}

template<int dim, class T>
const T l2norm(const Point<dim,T> &p){
	T norm = 0.0;
	for (int i = 0; i < dim; i++){
		norm += p[i]*p[i];
	}
	return sqrt(norm);
}

template<int dim, class T>
const T dist(const Point<dim,T> &p1, const Point<dim,T> &p2){
	return l2norm(p1-p2);
}
#endif /* POINT_HH_ */
