/*
 * gmres.hh
 *
 *  Created on: 2013. 9. 1.
 *      Author: parkmh
 */

#ifndef GMRES_HH_
#define GMRES_HH_


#include "krylov.hh"

template <class PCOND>
class GMRES : public Krylov<PCOND>{
public:
	typedef typename PCOND::datatype datatype;
	typedef typename PCOND::matrix matrix;
	GMRES();
	GMRES(const matrix& , const PCOND &, size_t, size_t, double);
	~GMRES();
	void init(const matrix& , const PCOND &, size_t,size_t,double);
	void solve(const NumVec<datatype> &, NumVec<datatype> &x) const;
	void print(std::ostream&) const;
private:
	const matrix *A;
	const PCOND * conditioner_;
	double* Residual_;
	size_t mr_;

};

template <class PCOND>
GMRES<PCOND>::GMRES(){
	A = NULL;
	conditioner_ = NULL;
	Residual_ = NULL;
	mr_ = 0;
}

template <class PCOND>
GMRES<PCOND>::GMRES(const matrix& a, const PCOND & conditioner, size_t ncycle = 20, size_t mr = 10, double tol = 1e-10){
	Residual_ = NULL;
	init(a,conditioner,ncycle,mr,tol);
}

template <class PCOND>
void GMRES<PCOND>::init(const matrix& a, const PCOND &conditioner, size_t ncycle = 20, size_t mr = 10,double tol = 1e-10){
	this->A_ = &a;
	this->n_ = a.row();
	A = (this->A_);
	conditioner_ = &conditioner;
	this->ncycle_ = ncycle;
	this->tol_ = tol;
	mr_ = mr;
	delete[] Residual_;
	Residual_ = new double [this->ncycle_+1];
}

template <class PCOND>
GMRES<PCOND>::~GMRES(){
	delete [] Residual_;
}
template <class PCOND>
void GMRES<PCOND>::solve(const NumVec<datatype>& rhs, NumVec<datatype> &v) const {
	this->lastcycle_ = this->ncycle_;
	NumVec< datatype > r(rhs);

	r -= (*A) * v;
	Residual_[0] = r.l2norm();
	NumVec< datatype > z(A->row());
	conditioner_->solve(r,z);

	NumVec< datatype > p(z);
	datatype nu, alpha, beta;
	NumVec< datatype > Ap;
	Ap.resize(A->row());
	for (size_t i = 1; i <= this->ncycle_; i++) {
		nu = (r,z);
		axey<datatype>(*A,p,Ap);
		alpha = nu/(p,Ap);
		v += (alpha*p);
		r -= (alpha*Ap);
		Residual_[ i ] = (rhs - ((*A) * v)).l2norm();
//		if (printflag == 1){
			printf("       %04ld.............................%.4e\n",i,Residual_[i]);
//		}
		if (Residual_[ i ] / Residual_[ 0 ]  < this->tol_) {
			this->lastcycle_ = i;
			break;
		}
		z = 0;

		conditioner_->solve(r,z);
		beta = (r,z)/nu;
		p = z + beta*p;

	}
}

template <class PCOND>
void GMRES<PCOND>::print(std::ostream& os) const{
	os << "Preconditioned Conjugate Gradient" << std::endl;
	os << "[ GMRES -> Preconditioner ] " << std::endl;
	if (conditioner_ == NULL)
		os << "not assigned yet." << std::endl;
	else
		conditioner_->overview(os);
}




#endif /* GMRES_HH_ */
