/*
 * cem_traits.hh
 *
 *  Created on: 2013. 8. 14.
 *      Author: parkmh
 */

#ifndef CEM_TRAITS_HH_
#define CEM_TRAITS_HH_
#include <cmath>
#include <functional>
#include <sstream>
#include "../../mesh/point.hh"
#include "../../cem/covariance_functions.hh"
struct cem1d_tag{};
struct cem2d_tag{};
struct cem3d_tag{};
#include <iostream>

template <int dim, typename T> struct ZeroMean;
template <int dim, typename T> struct L1Norm;
template <int dim, typename T> struct L2Norm;


template<class Category, typename T = void>
struct cem_traits {};

template<typename T>
struct cem_traits<cem1d_tag,T>{
	const static int dim = 1;
	typedef T datatype;
	typedef ExpCov<datatype> covf;
	typedef L1Norm<dim,T> Norm;
	typedef ZeroMean<dim,T> Mu;
};

template<typename T>
struct cem_traits<cem2d_tag,T>{
	const static int dim = 2;
	typedef T datatype;
	typedef ExpCov<datatype> covf;
	typedef L2Norm<dim,T> Norm;
	typedef ZeroMean<dim,T> Mu;
};

template<typename T>
struct cem_traits<cem3d_tag,T>{
	const static int dim = 3;
	typedef T datatype;
	typedef ExpCov<datatype> covf;
	typedef L2Norm<dim,T> Norm;
	typedef ZeroMean<dim,T> Mu;
};


template <int dim, typename T>
struct ZeroMean{
	T operator()(const Point<dim,T>&d){
		return 0;
	}
	std::string Description() const {
		std::string description = "Zero mean value function";
		return description;
	}
};

template <int dim, typename T>
struct L1Norm{
	const T l1 = 1;
	const T l2 = 1;
	const T l3 = 1;

	double operator()(const Point<dim,T>& d){
		double r = 0.0;
		r += fabs(d[0])/l1;
		if (dim > 1)
			r += fabs(d[1])/l2;
		if (dim == 3)
			r += fabs(d[2])/l3;

		return r;
	}
	std::string Description() const {
		std::ostringstream convert1, convert2, convert3;
		convert1 << l1;
		std::string description = "L1 [lambda1 : " + convert1.str();
		if (dim > 1){
			convert2 << l2;
			description += ", lambda2 : " + convert2.str();
		}
		if (dim == 3){
			convert3 << l3;
			description += ", lambda3 : " + convert3.str();
		}
		description += "]";
		return description;
	}
};

template <int dim, typename T>
struct L2Norm{
	const T l1 = 1;
	const T l2 = 1;
	const T l3 = 1;

	double operator()(const Point<dim,T>& d){
		double r = 0.0;
		r += (d[0]*d[0])/(l1*l1);
		if (dim > 1)
			r += (d[1]*d[1])/(l2*l2);
		if (dim == 3)
			r += (d[2]*d[2])/(l3*l3);
		return sqrt(r);
	}
	std::string Description() const {
		std::ostringstream convert1, convert2, convert3;
		convert1 << l1;
		std::string description = "L2 [lambda1 : " + convert1.str();
		if (dim > 1){
			convert2 << l2;
			description += ", lambda2 : " + convert2.str();
		}
		if (dim == 3){
			convert3 << l3;
			description += ", lambda3 : " + convert3.str();
		}
		description += "]";
		return description;
	}
};



#endif /* CEM_TRAITS_HH_ */
