/*
 * cem.hh
 *
 *  Created on: 2013. 8. 14.
 *      Author: Minho Park
 */

#ifndef CEM_HH_
#define CEM_HH_
#define REAL true
#define COMPLEX false

#include "../mesh/point.hh"
#include "../math/numvec.hh"
#include "../math/cnumvec_op.hh"
#include "../util/config.hh"

#include "fftw_wrapper.hh"
#include <vector>
#include <cstdio>
#include <iostream>
#include <iomanip>
#include <string>

template <class Category, typename T>
class CEM{
public:
	typedef cem_traits<Category,T> 	traits;
	const static int dim = traits::dim;
	typedef typename traits::covf 	COVFUN;
	typedef typename traits::Norm 	Norm;
	typedef typename traits::Mu		MuFUN;
	typedef Point<dim,T> 			pos;
	typedef Point<3,T> 				pos3;
	typedef Point<dim,size_t> 		N;
	typedef Point<3,size_t> 			N3;
	typedef std::complex<T>			Complex;

	CEM(){ isFirst = true; rSample = NULL; iSample = NULL; nSampling = 0;}
	CEM(const pos &, const pos &, const N&, const pos &);
	CEM(const CEM&);
	virtual ~CEM();
	virtual void init(const pos &, const pos &, const N&, const pos &);
	virtual void generate(const NumVec<std::complex<T> > &, NumVec<T> &) const;
	void generate(NumVec<T> &) const;
	void generate(NumVec<T> &,NumVec<T> & ) const;
	size_t sampleSize() const;
	size_t extendedSampleSize() const;
	void print_first_column() const;
	virtual void complexWhiteNoise(NumVec<std::complex<T> >&) const;
	std::string getDescription() const {
		std::string s = "Circulant Embedding Method (" + norm_.Description() + " " + covf_.Description() +")";
		return s;
	}

	friend std::ostream&
	operator<<(std::ostream& os, const CEM& cem){
		os << std::endl << "***** " << dim << "D Circulant Embedding Method ***** " << std::endl;
		os << "start 					: " << cem.start_ << std::endl;
		os << "end   					: " << cem.end_ << std::endl;
		os << "N     			: " << cem.initNGrid3_ << std::endl;
		os << "h 			: " << cem.h_ << std::endl;
		os << "Covariance Function 	: " << cem.covf_.Description() << std::endl;
		os << "Norm  					: " << cem.norm_.Description() << std::endl;
		return os;
	}

protected:
	COVFUN covf_;
	Norm 	norm_;
	MuFUN	mufun_;
	std::vector<std::vector<std::vector<double> > > circ_;
	N3 		initNGrid3_;
	N3 		nExtendedGrid3_;
	N3 		nWholeGrid3_;

	N		nExtendedGrid_;
	N 		nWholeGrid_;

	pos 	start_;
	pos		end_;
	pos3 	start3_;
	pos3 	end3_;
	pos3 	h_;

	NumVec<double> sqrtEigVal_;
	mutable NumVec<T> mu_;
	mutable bool isFirst;

	T* rSample;
	T* iSample;
	size_t nSampling;

	// Protected member function
	void update();
	void computeFirstColumn(NumVec<double> &);
	size_t lengthC();
	size_t lengthT();

	void increasePadX(size_t);
	void increasePadY(size_t);
	void increasePadZ(size_t);

	void sqrtLambdaX1(const NumVec<Complex>&, NumVec<Complex>&) const;
	void sqrtLambdaX2(const NumVec<Complex>&, NumVec<Complex>&) const;
	void sqrtLambdaX3(const NumVec<Complex>&, NumVec<Complex>&) const;

	void trim1(const NumVec<Complex>& out) const;
	void trim2(const NumVec<Complex>& out) const;
	void trim3(const NumVec<Complex>& out) const;

	void run_cem(const NumVec<Complex>&) const;
	void compute_mean();

};


template <class Category, typename T>
CEM<Category, T>::CEM(const pos & start, const pos &end, const N& n, const pos &h){
	init(start, end, n,h);
}

template <class Category, typename T>
CEM<Category, T>::~CEM(){
	delete []  rSample;
	delete []  iSample;
}

template <class Category, typename T>
void CEM<Category, T>::init(const pos & start, const pos &end, const N& n, const pos &h){
	NumVec<double> firstColumn;
	FFT_R2C<dim> FFTR2C;
	int nNegEVal;

	isFirst = true;
	nExtendedGrid3_ = 1;
	start_ = start;
	end_ = end;
	for (size_t i = 0; i < dim; i++){
		start3_[i] 	= start[i];	end3_[i]   	= end[i];
		nExtendedGrid_[i] = nExtendedGrid3_[i]     	= n[i];
		h_[i]    	= h[i];
	}
	nSampling = nExtendedGrid3_[0] + 1;
	if (dim > 1) nSampling *= nExtendedGrid3_[1] + 1;
	if (dim > 2) nSampling *= nExtendedGrid3_[2] + 1;


	rSample = new T[nSampling];
	iSample = new T[nSampling];

	initNGrid3_ = nExtendedGrid3_;

	update();

	// Compute eigenvalues of circulant matrix.
	computeFirstColumn(firstColumn);

	// product of nwholeGrid3 is equivalent to the degrees of freedom of the circulant matrix
	nWholeGrid3_ = 1;
	for (size_t i = 0; i < dim; i++) {
		nWholeGrid_[i] = nWholeGrid3_[i] = nExtendedGrid3_[i]*2;
	}

	// Initialise the FFT
	FFTR2C.init(firstColumn,nWholeGrid_);

	// Execute FFT
	FFTR2C.execute();

	sqrtEigVal_.resize(FFTR2C.outSize());
	FFTR2C.real(sqrtEigVal_);

	// compute the number of negative eigenvalues.
	nNegEVal = sqrtEigVal_.nNeg();

	// Choose the size of padding depending the number of negative eigenvalues.
	int padsize = (int)ceil((double)nExtendedGrid3_[0]*0.2);
	while (nNegEVal > 0){	// increase padding to make circulant matrix be SPD matrix
		std::cout << "Padding increased (nNegEVal=" << nNegEVal << ")"<<std::endl;
		increasePadX(padsize);
		if (dim > 1) increasePadY(padsize);
		if (dim > 2) increasePadZ(padsize);

		computeFirstColumn(firstColumn);
		for (size_t i = 0; i < dim; i++) {
			nWholeGrid_[i] = nWholeGrid3_[i] = nExtendedGrid3_[i]*2;
		}
		FFTR2C.init(firstColumn,nWholeGrid_);
		FFTR2C.execute();
		sqrtEigVal_.resize(FFTR2C.outSize());
		FFTR2C.real(sqrtEigVal_);
		nNegEVal = sqrtEigVal_.nNeg();
		if (nNegEVal < 10) padsize = 1;
	}

	sqrtEigVal_/=(double)nWholeGrid_.prod();
	sqrtEigVal_.sqrt();
	compute_mean();
}

template <class Category, typename T>
void CEM<Category, T>::update(){
	size_t i,j,k;
	size_t nx,ny,nz;
	T r;

	// Note that unconditional covariance matrix is invariant under shift
	// so it doesn't matter where to start
	pos position = 0.0;
	nx = nExtendedGrid3_[0]+1;
	if (dim > 1)	ny = nExtendedGrid3_[1]+1;
	else ny = 1;
	if (dim > 2)  nz = nExtendedGrid3_[2]+1;
	else nz = 1;

	if (dim == 3){
		circ_.resize(nz);
	} else {
		circ_.resize(1);
	}
	for (k = 0; k < nz; k++){
		if (dim > 2){	position[2] = h_[2]*k;  }//position.set(2,hz*k);
		if (dim > 1){ circ_[k].resize(ny); }
		else { circ_[k].resize(1); 	}
		for (j = 0; j < ny; j++){
			if (dim > 1) position[1] = h_[1] * j;//position.set(1,hy*j);
			circ_[k][j].resize(nx);
			for (i = 0; i < nx; i++){
				position[0] = h_[0]*i; //.set(0,hx*i);
				r = norm_(position);
				circ_[k][j][i] = covf_(r);// (covfun.evaluate(position,correlationLength_,sigma_));
			}
		}
	}
}

template <class Category, typename T>
size_t CEM<Category, T>::lengthC(){
	size_t clen = 1;
	clen *= 2*(nExtendedGrid3_[0]);
	if (dim > 1)	clen *= 2*(nExtendedGrid3_[1]);
	if (dim > 2)	clen *= 2*(nExtendedGrid3_[2]);
	return clen;
}

template <class Category, typename T>
size_t CEM<Category, T>::lengthT(){
	size_t tlen = 1;
	tlen *= nExtendedGrid3_[0]+1;
	if (dim > 1) 	tlen *= nExtendedGrid3_[1]+1;
	if (dim > 2)	tlen *= nExtendedGrid3_[2]+1;
	return tlen;
}

template <class Category, typename T>
void CEM<Category, T>::computeFirstColumn(NumVec<double>& firstColumn){
	size_t clen = lengthC();
	firstColumn.resize(clen);

	int i,j,k;
	size_t nx, ny, nz;
	nx = nExtendedGrid3_[0] + 1;
	if (dim > 1)	ny = nExtendedGrid3_[1] + 1; 	else ny = 1;
	if (dim > 2)  nz = nExtendedGrid3_[2] + 1; 	else nz = 1;


	/* Note that the first column is stored in row-major format */
	int cindex = 0;
	for (k = 0; k < nz; k++){
		for (j = 0; j < ny; j++){
			for (i = 0; i < nx; i++){
				firstColumn[cindex] =  circ_[k][j][i]; cindex++;
			}
			for (i = nx-2; i > 0; i--){
				firstColumn[cindex] =  circ_[k][j][i]; cindex++;
			}
		}
		for (j = ny-2; j > 0; j--){
			for (i = 0; i < nx; i++){
				firstColumn[cindex] =  circ_[k][j][i]; cindex++;
			}
			for (i = nx-2; i > 0; i--){
				firstColumn[cindex] =  circ_[k][j][i]; cindex++;
			}
		}
	}
	for (k = nz-2; k > 0; k--){
		for (j = 0; j < ny; j++){
			for (i = 0; i < nx; i++){
				firstColumn[cindex] =  circ_[k][j][i]; cindex++;
			}
			for (i = nx-2; i > 0; i--){
				firstColumn[cindex] =  circ_[k][j][i]; cindex++;
			}
		}
		for (j = ny-2; j > 0; j--){
			for (i = 0; i < nx; i++){
				firstColumn[cindex] =  circ_[k][j][i]; cindex++;
			}
			for (i = nx-2; i > 0; i--){
				firstColumn[cindex] =  circ_[k][j][i]; cindex++;
			}
		}
	}
}

template <class Category, typename T>
void CEM<Category, T>::print_first_column() const{
	size_t i, j, k;
	size_t isize, jsize, ksize;
	ksize = circ_.size();
	for (k = 0; k < ksize; k++){
		jsize = circ_[k].size();
		for (j = 0; j < jsize; j++){
			isize = circ_[k][j].size();
			for (i = 0; i < isize; i++){
				printf("%7.5f ", circ_[k][j][i]);
			}
			printf("\n");
		}
		printf("\n");
	}
	printf("\n");
}

template <class Category, typename T>
void CEM<Category, T>::complexWhiteNoise(NumVec<std::complex<T> >& x) const{
	x.resize(extendedSampleSize());
	randn<T>(x);
}

template <class Category, typename T>
void CEM<Category, T>::increasePadX(size_t extra){
	size_t i, j, k;
	size_t nx, ny, nz;
	nx = nExtendedGrid3_[0]+1;
	if (dim > 1)	ny = nExtendedGrid3_[1]+1;
	else ny = 1;
	if (dim > 2)  nz = nExtendedGrid3_[2]+1;
	else nz = 1;

	T r;
	pos position;

	for (k = 0; k < nz; k++){
		if (dim>2) position[2] = h_[2]*k;
		for (j = 0; j < ny; j++){
			if (dim>1) position[1] = h_[1]*j;
			circ_[k][j].resize(nx+extra);
			for (i = nx; i < nx+extra; i++){
				position[0] = h_[0]*i;
				r = norm_(position);
				circ_[k][j][i] = covf_(r);
			}
		}
	}
	nExtendedGrid3_[0] += extra;
	nExtendedGrid_[0] += extra;
}

template <class Category, typename T>
void CEM<Category, T>::increasePadY(size_t extra){
	size_t i, j, k;
	size_t nx, ny, nz;
	nx = nExtendedGrid3_[0]+1;
	if (dim > 1)	ny = nExtendedGrid3_[1]+1;
	else ny = 1;
	if (dim > 2)  nz = nExtendedGrid3_[2]+1;
	else nz = 1;

	T r;
	pos position;

	for (k = 0; k < nz; k++){
		if (dim>2) position[2] = h_[2]*k;
		circ_[k].resize(ny+extra);
		for (j = ny; j < ny+extra; j++){
			if (dim>1) position[1] = h_[1]*j;
			circ_[k][j].resize(nx);
			for (i = 0; i < nx; i++){
				position[0] = h_[0]*i;
				r = norm_(position);
				circ_[k][j][i] = covf_(r);
			}
		}
	}
	nExtendedGrid3_[1] += extra;
	nExtendedGrid_[1] += extra;
}

template <class Category, typename T>
void CEM<Category, T>::increasePadZ(size_t extra){
	size_t i, j, k;
	size_t nx, ny, nz;
	nx = nExtendedGrid3_[0]+1;
	if (dim > 1)	ny = nExtendedGrid3_[1]+1;
	else ny = 1;
	if (dim > 2)  nz = nExtendedGrid3_[2]+1;
	else nz = 1;

	T r;
	pos position;

	circ_.resize(nz+extra);
	for (k = nz; k < nz + extra; k++){
		position[2] = h_[2]*k;
		circ_[k].resize(ny);
		for (j = 0; j < ny; j++){
			position[1]= h_[1]*j;
			circ_[k][j].resize(nx);
			for (i = 0; i < nx; i++){
				position[0] = h_[0]*i;
				r = norm_(position);
				circ_[k][j][i] = covf_(r);
			}
		}
	}
	nExtendedGrid3_[2] += extra;
	nExtendedGrid_[2] += extra;
}

/*
 * Generates a random field using a complex white noise IN and save it in X.
 * This is useful when IN is used again in other part of code.
 */
template <class Category, typename T>
void CEM<Category,T>::generate(const NumVec<std::complex<T> >& in, NumVec<T> &x) const{
	if (isFirst){
		run_cem(in);
		x.assign(rSample,nSampling);
		x += mu_;
		isFirst = false;
	} else {
		isFirst = true;
		x.assign(iSample,nSampling);
		x += mu_;
	}
}

/*
 * Generates a random field without a complex white noise.
 */
template <class Category, typename T>
void CEM<Category,T>::generate(NumVec<T> &x) const{
	if (isFirst){
		NumVec<std::complex<T> > e;
		complexWhiteNoise(e);
		run_cem(e);
		x.assign(rSample,nSampling);
		x += mu_;
		isFirst = false;
	} else {
		isFirst = true;
		x.assign(iSample,nSampling);
		x += mu_;
	}
}


/*
 * Generates two random fields which are antithetic pairs.
 */
template <class Category, typename T>
void CEM<Category,T>::generate(NumVec<T> &x,NumVec<T> &y) const{
	if (isFirst){
		NumVec<std::complex<T> > e;
		complexWhiteNoise(e);
		run_cem(e);
		x.assign(rSample,nSampling);
		y.assign_negative(rSample,nSampling);
		x += mu_;
		y += mu_;
		isFirst = false;
	} else {
		isFirst = true;
		x.assign(iSample,nSampling);
		y.assign_negative(iSample,nSampling);
		x += mu_;
		y += mu_;
	}
}

/*
 * This returns the size of a random samples. Basically it is the same as prod(initNGrid3_+1) in
 * MATLAB notations.
 */
template <class Category, typename T>
size_t CEM<Category,T>::sampleSize() const {
	size_t ss = 1;
	for (int i = 0; i < dim; i++){
		ss *= (initNGrid3_[i] + 1);
	}
	return ss;
}

/*
 * This returns the degrees of freedom of the circulant matrix.
 */
template <class Category, typename T>
size_t CEM<Category,T>::extendedSampleSize() const {
	size_t ss = 1;
	for (size_t i = 0; i < dim; i++){
		ss *= 2*nExtendedGrid3_[i];
	}
	return ss;
}

/*
 *  This computes the square roots of Lambda times a white noise IN on 1D.
 */
template <class Category, typename T>
void CEM<Category,T>::sqrtLambdaX1(const NumVec<Complex>&in, NumVec<Complex>& out) const{
	out.resize(in.size());
	size_t i;
	size_t index = 0;
	size_t clenx = nExtendedGrid3_[0]+1;
	for (i = 0; i < clenx; i++){
		out[index] = sqrtEigVal_[i]*in[index];
		index++;
	}
	for (i = clenx-2; i > 0; i--){
		out[index] = sqrtEigVal_[i]*in[index];
		index++;
	}
}

/*
 *  This computes the square roots of Lambda times a white noise IN on 2D.
 */
template <class Category, typename T>
void CEM<Category,T>::sqrtLambdaX2(const NumVec<Complex>&in, NumVec<Complex>& out) const{
	out.resize(in.size());
	size_t i,j,jsize;
	size_t index = 0;
	size_t clenx = nExtendedGrid3_[0]+1;
	size_t cleny = nExtendedGrid3_[1]+1;

	for (j = 0; j < cleny; j++){
		jsize = j*clenx;
		for (i = 0; i < clenx; i++){
			out[index] = sqrtEigVal_[jsize+i]*in[index];
			index++;
		}
		for (i = clenx-2; i > 0; i--){
			out[index] = sqrtEigVal_[jsize+i]*in[index];
			index++;
		}
	}
	for (j = cleny-2; j > 0; j--){
		jsize = j*clenx;
		for (i = 0; i < clenx; i++){
			out[index] = sqrtEigVal_[jsize+i]*in[index];
			index++;
		}
		for (i = clenx-2; i > 0; i--){
			out[index] = sqrtEigVal_[jsize+i]*in[index];
			index++;
		}
	}
}

/*
 *  This computes the square roots of Lambda times a white noise IN on 3D.
 */
template <class Category, typename T>
void CEM<Category,T>::sqrtLambdaX3(const NumVec<Complex>&in, NumVec<Complex>& out) const{
	out.resize(in.size());
	size_t i, j ,k, kjsize;
	size_t index = 0;
	size_t clenx = nExtendedGrid3_[0]+1;
	size_t cleny = 2*nExtendedGrid3_[1];
	size_t clenz = 2*nExtendedGrid3_[2];

	for ( k = 0; k < clenz; k++){
		for ( j = 0; j < cleny; j++){
			kjsize = clenx*(k*cleny+j);
			for ( i = 0; i < clenx; i++){
				out[index] = sqrtEigVal_[i+ kjsize]*in[index];
				index++;
			}
			for ( i = clenx-2; i > 0; i--){
				out[index] = sqrtEigVal_[i+ kjsize]*in[index];
				index++;
			}
		}
	}
}

/*
 * This just trims extra parts which ensure the covariance to be circulant.
 */
template <class Category, typename T>
void CEM<Category,T>::trim1(const NumVec<Complex>& out) const{
	size_t nx = initNGrid3_[0] + 1;

	for (size_t i = 0; i < nx; i++){
		rSample[i] = out[i].real();
		iSample[i] = out[i].imag();
	}
}

/*
 * This just trims extra parts which ensure the covariance to be circulant.
 */
template <class Category, typename T>
void CEM<Category,T>::trim2(const NumVec<Complex>& out) const{
	size_t nx = initNGrid3_[0] + 1;
	size_t ny = initNGrid3_[1] + 1;
	size_t dx = 2*nExtendedGrid3_[0];
	for (size_t j = 0; j < ny; j++){
		for (size_t i = 0; i < nx; i++){
			rSample[j*nx+i] = out[j*dx+i].real();
			iSample[j*nx+i] = out[j*dx+i].imag();
		}
	}
}

/*
 * This just trims extra parts which ensure the covariance to be circulant.
 */
template <class Category, typename T>
void CEM<Category,T>::trim3(const NumVec<Complex>& out) const{
	size_t nx = initNGrid3_[0] + 1;
	size_t ny = initNGrid3_[1] + 1;
	size_t nz = initNGrid3_[2] + 1;
	size_t dx = 2*nExtendedGrid3_[0];
	size_t dy = 2*nExtendedGrid3_[1];

	for (size_t k = 0; k < nz; k++){
		for (size_t j = 0; j < ny; j++){
			for (size_t i = 0; i < nx; i++){
				rSample[(ny*k+j)*nx+i] = out[(dy*k+j)*dx+i].real();
				iSample[(ny*k+j)*nx+i] = out[(dy*k+j)*dx+i].imag();
			}
		}
	}
}

template <class Category, typename T>
void CEM<Category,T>::run_cem(const NumVec<std::complex<T> >& in) const {
	FFT<dim> fft;

	NumVec<Complex> in2;

	switch (dim){
	case 1:
		sqrtLambdaX1(in,in2);
		break;
	case 2:
		sqrtLambdaX2(in,in2);
		break;
	case 3:
		sqrtLambdaX3(in,in2);
		break;
	}

	fft.init(in2,nWholeGrid_);
	fft.execute();
	const NumVec<Complex> out = fft.result();

	switch (dim){
	case 1:
		trim1(out);
		break;
	case 2:
		trim2(out);
		break;
	case 3:
		trim3(out);
		break;
	}

}
template <class Category, typename T>
void CEM<Category,T>::compute_mean(){
	mu_.resize(sampleSize());
	size_t i,j,k;
	size_t nx,ny,nz;
	T r;
	pos position = start_;
	nx = initNGrid3_[0]+1;
	if (dim > 1)	ny = initNGrid3_[1]+1;
	else ny = 1;
	if (dim > 2)  nz = initNGrid3_[2]+1;
	else nz = 1;

	size_t index = 0;
	for (k = 0; k < nz; k++){
		if (dim > 2){	position[2] = start_[2]+h_[2]*k;  }//position.set(2,hz*k);
		for (j = 0; j < ny; j++){
			if (dim > 1) position[1] = start_[1] + h_[1] * j;//position.set(1,hy*j);
			for (i = 0; i < nx; i++){
				position[0] = start_[0] + h_[0]*i; //.set(0,hx*i);
				mu_[index] = mufun_(position);
				index++;
			}
		}
	}
}

#endif /* CEM_HH_ */
