/*
 * ccem.hh
 *
 *  Created on: 2013. 9. 20.
 *      Author: parkmh
 */

#ifndef CCEM_HH_
#define CCEM_HH_

#include "cem.hh"
#include "observation.hh"
#include "../math/complex_full_matrix.hh"



template <class Category, typename T>
class CCEM: public CEM<Category, T>{
public:
	typedef cem_traits<Category,T> traits;
	const static int dim = traits::dim;
	typedef typename traits::covf COVFUN;
	typedef typename traits::Norm Norm;
	typedef Point<dim,T> pos;
	typedef Point<3,T> pos3;
	typedef Point<dim,size_t> N;
	typedef Point<3,size_t> N3;
	CCEM():CEM<Category,T>(){ observation_ = Observation<dim>::getInstance(); isFirst = true;
		mPz2r_ = NULL; mPz2i_ = NULL;};
	~CCEM(){ delete [] mPz2r_; delete [] mPz2i_;};

	void init(const pos &, const pos &, const N&, const pos &);
	void generate(NumVec<T> &) const;
	void generate(NumVec<T> &,NumVec<T> & ) const;
	void complexWhiteNoise(NumVec<std::complex<T> >&) const;
	std::string getDescription() const {
		return "Conditional Circulant Embedding Method (" + this->norm_.Description() + " " + this->covf_.Description() +")";
	}

	friend std::ostream&
	operator<<(std::ostream& os, const CCEM& cem){
		os << std::endl << "***** " << dim << "D Conditional Circulant Embedding Method ***** " << std::endl;
		os << "start 			: " << cem.start_ << std::endl;
		os << "end   			: " << cem.end_ << std::endl;
		os << "N     			: " << cem.nExtendedGrid_ << std::endl;
		os << "h 			: " << cem.h_ << std::endl;
		os << "Covariance Function 	: " << cem.covf_.Description() << std::endl;
		os << "Norm  			: " << cem.norm_.Description() << std::endl;
		return os;
	}
private:
	Observation<dim> * observation_;
	void computeR12(FMatrix<T> &);
	void computeR22(FMatrix<T> &);
	void computeK(FMatrix<T> &);
	void computeP(const FMatrix<T>&,const FMatrix<T> &);
	void computeL(const FMatrix<T> &,const FMatrix<T> &);
	void conditioning(const NumVec<std::complex<T> >&) const;
	void compute_mean();
	void compute_muc();
	CFMatrix<T> K;
	FMatrix<T> L, P;
	NumVec<T> z2s;
	NumVec<T> mu2_;
	NumVec<T> muc_;
	T* mPz2r_;
	T* mPz2i_;
	mutable bool isFirst;

};


template <class Category, typename T>
void CCEM<Category,T>::init(const pos & start, const pos &end, const N& n, const pos &h){

	FMatrix<T> R12, R22, KKH, R22inv;
	CEM<Category, T>::init(start,end,n,h);
	std::cout << "Building R12" << std::endl;
	computeR12(R12);
	std::cout << "Building R22" << std::endl;
	computeR22(R22);

	std::cout << "Building R22inv"<< std::endl;
	R22inv = R22.inv();
	std::cout << "Building K and KKH" << std::endl;
	computeK(KKH);

	std::cout << "Building P" << std::endl;
	computeP(R12,R22inv);
	std::cout << "Building R22" << std::endl;
	computeL(R22, KKH);
	z2s = observation_->returnOberservation();
	isFirst = true;

	compute_mean();
	compute_muc();
	mPz2r_ = new T[this->nSampling];
	mPz2i_ = new T[this->nSampling];
}


template <class Category, typename T>
void CCEM<Category,T>::computeR12(FMatrix<T> & R12){
	size_t nobsv = observation_->nobsv();
	size_t i,j,k;
	size_t nx,ny,nz;
	nx = this->initNGrid3_[0]+1;
	if (dim > 1)	ny = this->initNGrid3_[1]+1;
	else ny = 1;
	if (dim > 2)  nz = this->initNGrid3_[2]+1;
	else nz = 1;
	T r;
	pos position;
	switch (dim){
	case 1:
		R12.resize(nx,nobsv);
		for (size_t i = 0; i < nx; i++){
			for(size_t l = 0; l < nobsv; l++){
				position[0] = this->start_[0] + this->h_[0]*i;
				r = this->norm_(position-observation_->loc(l));
				R12(i,l) =  this->covf_(r);
			}
		}
		break;
	case 2:
		R12.resize(nx*ny,nobsv);
		for (size_t j = 0; j < ny; j++){
			position[1] = this->start_[1] + this->h_[1]*j;
			for (size_t i = 0; i < nx; i++){
				for(size_t l = 0; l < nobsv; l++){
					position[0] = this->start_[0] + this->h_[0]*i;
					r = this->norm_(position-observation_->loc(l));
					R12(i+j*ny,l) =  this->covf_(r);
//					std::cout << "R12(" << i+j*ny+1 << ", " << l+1 << ") : " << R12(i+j*ny,l) << std::endl;
				}
			}
		}
		break;
	case 3:
		R12.resize(nx*ny*nz,nobsv);
		for (size_t k = 0; k < nz; k++){
			position[2] = this->start_[2] + this->h_[2]*k;

			for (size_t j = 0; j < ny; j++){
				position[1] = this->start_[1] + this->h_[1]*j;
				for (size_t i = 0; i < nx; i++){
					for(size_t l = 0; l < nobsv; l++){
						position[0] = this->start_[0] + this->h_[0]*i;
						r = this->norm_(position-observation_->loc(l));
						R12(i+(j+k*nx)*ny,l) = this->covf_(r);
					}
				}
			}
		}
		break;
	}

}

template <class Category, typename T>
void CCEM<Category,T>::computeR22(FMatrix<T> & R22){
	size_t nobsv = observation_->nobsv();
	R22.resize(nobsv,nobsv);
	T r;
	for (size_t i = 0; i < nobsv; i++){
		for (size_t j = 0; j < nobsv; j++){
			r = this->norm_(observation_->loc(i)-observation_->loc(j));
			R22(i,j) = this->covf_(r);
		}
	}
}

template <class Category, typename T>
void CCEM<Category,T>::computeK(FMatrix<T> &KKH){
	FFT<dim> fft;
	size_t nobsv = observation_->nobsv();
	CFMatrix<T> KKHC;
	size_t nrow, index;
	pos position, distance, obsv;
	NumVec<std::complex<T> > column_k, z;
	T r;
	switch (dim){
	case 1:
		nrow = 2*this->nExtendedGrid3_[0];
		K.resize(nobsv,nrow);
		column_k.resize(nrow);
		for (size_t k = 0; k < nobsv; k++){
			obsv = observation_->loc(k);
			for (size_t i = 0; i < this->nExtendedGrid3_[0]; i++){
				position[0] = this->start_[0] + this->h_[0]*i;
				r = this->norm_(position-obsv);
				column_k[i] = std::complex<T>(this->covf_(r),0.0);
			}
			for (size_t i = this->nExtendedGrid3_[0]; i < nrow; i++){
				position[0] = this->start_[0] + this->h_[0]*i;
				distance = position-obsv;
				if (distance[0] > this->h_[0]*this->nExtendedGrid3_[0]) distance[0] = 2*this->h_[0]*this->nExtendedGrid3_[0]-distance[0];
				r = this->norm_(distance);
				column_k[i] = std::complex<T>(this->covf_(r),0.0);
			}

			fft.init(column_k,this->nWholeGrid_,FFTW_BACKWARD);
			fft.execute();
			z  = fft.result();

			for (size_t i = 0; i < nrow; i++){
				K(k,i) = z[i];
			}
		}
		KKHC = K*K.hermitian();
		KKH = KKHC.real();
		break;
	case 2:
		size_t jsize;
		nrow = 4*this->nExtendedGrid3_[0]*this->nExtendedGrid3_[1];
		K.resize(nobsv,nrow);
		column_k.resize(nrow);

		for (size_t k = 0; k < nobsv; k++){
			obsv = observation_->loc(k);
			for (size_t j = 0; j < this->nExtendedGrid3_[1]; j++){
				position[1] = this->start_[1] + this->h_[1]*j;
				for (size_t i = 0; i < this->nExtendedGrid3_[0]; i++){
					position[0] = this->start_[0] + this->h_[0]*i;
					r = this->norm_(position-obsv);
					column_k[j*2*this->nExtendedGrid3_[0]+i] = std::complex<T>(this->covf_(r),0.0);
				}
				for (size_t i = this->nExtendedGrid3_[0]; i < 2*this->nExtendedGrid3_[0];i++){
					position[0] = this->start_[0] + this->h_[0]*i;
					distance = position-obsv;
					if (distance[0] > this->h_[0]*this->nExtendedGrid3_[0]) distance[0] = 2*this->h_[0]*this->nExtendedGrid3_[0]-distance[0];
					r = this->norm_(distance);
					column_k[j*2*this->nExtendedGrid3_[0]+i] = std::complex<T>(this->covf_(r),0.0);
				}
			}

			for (size_t j = this->nExtendedGrid3_[1]; j < 2*this->nExtendedGrid3_[1]; j++){
				position[1] = this->start_[1] + this->h_[1]*j;
				for (size_t i = 0; i < this->nExtendedGrid3_[0]; i++){
					position[0] = this->start_[0] + this->h_[0]*i;
					distance = position-obsv;
					if (distance[1] > this->h_[1]*this->nExtendedGrid3_[1]) distance[1] = 2*this->h_[1]*this->nExtendedGrid3_[1]-distance[1];
					r = this->norm_(distance);
					column_k[j*2*this->nExtendedGrid3_[0]+i] = std::complex<T>(this->covf_(r),0.0);
				}
				for (size_t i = this->nExtendedGrid3_[0]; i < 2*this->nExtendedGrid3_[0];i++){
					position[0] = this->start_[0] + this->h_[0]*i;
					distance = position-obsv;
					if (distance[1] > this->h_[1]*this->nExtendedGrid3_[1]) distance[1] = 2*this->h_[1]*this->nExtendedGrid3_[1]-distance[1];
					if (distance[0] > this->h_[0]*this->nExtendedGrid3_[0]) distance[0] = 2*this->h_[0]*this->nExtendedGrid3_[0]-distance[0];
					r = this->norm_(distance);
					column_k[j*2*this->nExtendedGrid3_[0]+i] = std::complex<T>(this->covf_(r),0.0);
				}
			}
			fft.init(column_k,this->nWholeGrid_,FFTW_BACKWARD);
			fft.execute();
			z  = fft.result();

			index = 0;
			for (size_t j = 0; j < this->nExtendedGrid3_[1]+1; j++){
				jsize = j*(this->nExtendedGrid3_[0]+1);
				for (size_t i = 0; i < this->nExtendedGrid3_[0]+1; i++){
					K(k,index) = std::conj(z[index])/this->sqrtEigVal_[jsize+i];
					index++;
				}
				for (size_t i = this->nExtendedGrid3_[0]-1; i > 0; i--){
					K(k,index) = std::conj(z[index])/this->sqrtEigVal_[jsize+i];
					index++;
				}
			}
			for (size_t j = this->nExtendedGrid3_[1]-1; j > 0; j--){
				jsize = j*(this->nExtendedGrid3_[0]+1);
				for (size_t i = 0; i < this->nExtendedGrid3_[0]+1; i++){
					K(k,index) = std::conj(z[index])/this->sqrtEigVal_[jsize+i];
					index++;
				}
				for (size_t i = this->nExtendedGrid3_[0]-1; i > 0; i--){
					K(k,index) = std::conj(z[index])/this->sqrtEigVal_[jsize+i];
					index++;
				}
			}

		}
		K.aah(KKHC);
		KKH = KKHC.real();
		break;
	case 3:
//		size_t kjsize;
//		nrow = 8*this->nExtendedGrid3_[0]*this->nExtendedGrid3_[1]*this->nExtendedGrid3_[2];
//		K.resize(nobsv,nrow);
		// TODO compueK for 3D model

		break;
	}

}

template <class Category, typename T>
void CCEM<Category,T>::computeP(const FMatrix<T>&R12, const FMatrix<T> &R22inv){
	P = R12*R22inv;
}

template <class Category, typename T>
void CCEM<Category,T>::computeL(const FMatrix<T> &R22,const FMatrix<T> &KKH){
	FMatrix<T> temp = R22-KKH;
	L = temp.chol();
}

template <class Category, typename T>
void CCEM<Category,T>::conditioning(const NumVec<std::complex<T> >&z2) const{
	size_t nSampling = this->sampleSize();
	size_t nObsv 		= z2s.size();

	T sum_real;
	T sum_imag;
	for (size_t i = 0; i < nSampling; i++){
		sum_real = .0;
		sum_imag = .0;
		for (size_t j = 0;j <  nObsv;j++){
			sum_real -= P(i,j)*z2[j].real();
			sum_imag -= P(i,j)*z2[j].imag();
		}
		mPz2r_[i] = sum_real;
		mPz2i_[i] = sum_imag;
	}
}


template <class Category, typename T>
void CCEM<Category,T>::generate(NumVec<T> &x) const{
	size_t nSampling = this->sampleSize();

	if (isFirst){
		NumVec<std::complex<T> > z2;
		NumVec<std::complex<T> > e1, e2;	// unconditional random vector

		CEM<Category,T>::complexWhiteNoise(e1);
		complexWhiteNoise(e2);

		CEM<Category,T>::run_cem(e1);

		ke1le2ey<T>(K,e1,L,e2,z2);

		conditioning(z2);

		x.assign_two_sum(this->rSample,mPz2r_,nSampling);
		x += muc_;
		isFirst = false;
	}
	else{
		isFirst = true;
		x.assign_two_sum(this->iSample,mPz2i_,nSampling);
		x += muc_;
	}
}

/*
 * Generates two random fields which are antithetic pairs.
 */
template <class Category, typename T>
void CCEM<Category,T>::generate(NumVec<T> &x,NumVec<T> &y) const{
	size_t nSampling = this->sampleSize();

	if (isFirst){
		NumVec<std::complex<T> > z2;
		NumVec<std::complex<T> > e1, e2;	// unconditional random vector

		CEM<Category,T>::complexWhiteNoise(e1);
		complexWhiteNoise(e2);

		CEM<Category,T>::run_cem(e1);

		ke1le2ey<T>(K,e1,L,e2,z2);

		conditioning(z2);

		x.assign_two_sum(this->rSample,mPz2r_,nSampling);
		y.assign_negative(x.begin(),nSampling);
		x += muc_;
		y += muc_;
		isFirst = false;

	}
	else{
		isFirst = true;
		x.assign_two_sum(this->iSample,mPz2i_,nSampling);
		y.assign_negative(x.begin(),nSampling);
		x += muc_;
		y += muc_;
	}
}

template <class Category, typename T>
void CCEM<Category,T>::complexWhiteNoise(NumVec<std::complex<T> >&x) const{
	x.resize(observation_->nobsv());
	randn<T>(x);
}

template <class Category, typename T>
void CCEM<Category,T>::compute_mean(){
	size_t nobsv = observation_->nobsv();
	mu2_.resize(nobsv);
	for(size_t l = 0; l < nobsv; l++){
		mu2_[l] = this->mufun_(observation_->loc(l));
	}
}

template <class Category, typename T>
void CCEM<Category,T>::compute_muc(){
	muc_.resize(this->nSampling);

	size_t nSampling = this->nSampling;
	size_t nObsv 		= z2s.size();
	T temp;
	for (size_t i = 0; i < nSampling; i++){
		temp = this->mu_[i];

		for (size_t j = 0;j <  nObsv;j++){
			temp += P(i,j)*(z2s[j] - mu2_[j]);
		}
		muc_[i] = temp;
	}
}


#endif /* CCEM_HH_ */
