# Functions for PCO

logmap <- function(q,q0=quaternion(c(0,0,0,1))){
  if(!missing(q0)){
    q <- quatmult(quatinv(q0),q)
  }
  qm <- q@x
  if(nrow(qm) == 1){
    nqv <- vnorm(qm[-4])
    if(nqv < 1e-7) return(c(0,0,0))
    qm[-4]*2*acos(qm[4])/nqv
  }else{
    nqv <- apply(qm[,-4],1,vnorm)
    nqv <- pmax(nqv,1e-7)
    qm[,-4]*2*acos(pmin(qm[,4],1))/nqv
  }
}

quatmult <- function(q1,q2){
  if(class(q1) != "quaternion") stop("First argument not a quaternion")
  if(class(q2) != "quaternion") stop("First argument not a quaternion")
  qm1 <- q1@x
  qm2 <- q2@x
  if(nrow(qm1) == 1 & nrow(qm2) != 1){
    qm1 <- matrix(as.vector(qm1),nrow=nrow(qm2),ncol=4,byrow=T)
  }
  if(nrow(qm2) == 1 & nrow(qm1) != 1){
    qm2 <- matrix(as.vector(qm2),nrow=nrow(qm1),ncol=4,byrow=T)
  }
  vecp <- veccross(qm1[,-4],qm2[,-4])
  if(nrow(qm1) == 1 & nrow(qm2) == 1){
    scap <- sum(qm1*qm2)
    quaternion(c(qm1[4]*qm2[-4]+qm2[4]*qm1[-4]+vecp,qm1[4]*qm2[4]-scap))
  }else{
    scap <- apply(qm1[,-4]*qm2[,-4],1,sum)
    quaternion(cbind(qm1[,4]*qm2[,-4]+qm2[,4]*qm1[,-4]+vecp,qm1[,4]*qm2[,4]-scap))
  }

}

quatinv <- function(q){
  if(class(q) != "quaternion") stop("Argument not a quaternion")
  qm <- q@x
  nqm <- apply(qm,1,vnorm)
  qm[,1:3] <- -qm[,1:3]
  qm <- qm/(nqm^2)
  quaternion(qm)
}

# vectorized cross prod
"veccross" <-
  function(a,b)
  {
    if(is.vector(a) & is.vector(b)){
      cross(a,b)
    }else{
      cbind(a[,2]*b[,3]-b[,2]*a[,3],-a[,1]*b[,3]+b[,1]*a[,3],a[,1]*b[,2]-a[,2]*b[,1])
    }
  }

                                        # cross prod
"cross" <-
  function(a,b)
  {
    c(a[2]*b[3]-b[2]*a[3],-a[1]*b[3]+b[1]*a[3],a[1]*b[2]-a[2]*b[1])
  }

# norm of a vector
vnorm <- function(x) sqrt(sum(x^2))

# Compute the projection of a new point based on distances to new point ndist and MDS info qcmds
pcogow <- function(ndist,qcmds){
  k <- ncol(qcmds$points)
  dv <- -diag(qcmds$x)/2-ndist^2
  as.vector(t(qcmds$points) %*% dv)/(qcmds$eig[1:k] *2)
}

# Backscore from score space to quaternion space
# inputs are:
# mquat - quaternion data
# qm - mean of the quaternion data
# targlam - score
# mds - CMDS of data
# nrep - number of steps in search
backscoreq <- function(mquat,qm,targlam,mds,nrep){
  n <- nrow(mquat@x)
  nd <- numeric(n)
  scaled.score <- targlam*sd(mds$points)
  qorb <- quaternion(matrix(NA,nrep,4))
  acm <- matrix(NA,nrep,2)

  for(j in 1:nrep){
    trii <- sample((1:n),3)
    
                                        # step 0
    lam <- solve(rbind(1,t(mds$points[trii,])),c(1,scaled.score))
    projq <- quaternion(weighted.mean(mquat[trii,],lam))
    for(i in 1:n) nd[i] <- rotation.distance(mquat[i,],projq)
    actscore <- pcogow(nd,mds)
                                        # step 1
    lam1 <- solve(rbind(1,t(mds$points[trii,])),c(1,2*scaled.score-actscore))
    projq1 <- quaternion(weighted.mean(mquat[trii,],lam1))
    for(i in 1:n) nd[i] <- rotation.distance(mquat[i,],projq1)
    actscore1 <- pcogow(nd,mds)

    qorb[j,] <- projq1
    acm[j,] <- actscore1/sd(mds$points)
  }

  corb <- quatmult(quatinv(qm),qorb)
  pcorb <- logmap(corb)

  rotang <- numeric(nrep)
  for(j in 1:nrep) rotang[j] <- rotation.distance(qorb[j,],qm)

  list(qorb=qorb,actscore=acm,rotang=rotang,pcorb=pcorb)
}

# Shape functions

# Function for computing the distance between markers in the face
distm <- function(m,i,j){
  sqrt(apply((m[,(3*i-2):(3*i)]-m[,(3*j-2):(3*j)])^2,1,sum))
}

# data to score
scoredat <- function(newshape,shapedata,mds){
  ndist <- nrow(mds$points)
  k <- ncol(mds$points)
  for(i in 1:n){
    ndist[i] <- shapdist(shapedata[,,i],newshape)
  }
  dv <- -diag(mds$x)/2-ndist^2
  as.vector(t(mds$points) %*% dv)/(mds$eig[1:k] *2)
}

# score to data
# set meanshape = NA for mean calculation
backscore <- function(shapedata,meanshape,targscore,mds,nrep,verbose=FALSE){

  if(is.na(meanshape)){
    scaled.score <- c(0,0)
  }else{
    scaled.score <- targscore*sd(mds$points)
  }
  dimsol <- ncol(mds$points)
  k <- dim(shapedata)[1]
  m <- dim(shapedata)[2]
  n <- dim(shapedata)[3]

  # figure out set of trios

  if(is.numeric(nrep)){
    triset <- matrix(NA,3,nrep)
    for(j in 1:nrep) triset[,j] <- sample((1:n),3)
  }else{
    triset <- combn(1:n,3)
    nrep <- ncol(triset)
  }

  savscore <- matrix(NA,nrep,dimsol)
  savshape <- array(NA,dim=c(k,m,nrep))
  savtri <- matrix(NA,nrep,dimsol+1)
  savwt <- matrix(NA,nrep,dimsol+1)
  

  for(j in 1:nrep){
    trii <- triset[,j]
    if(verbose) cat(trii,"\n")

    tsol <- tripsol(shapedata,mds,targscore,trii)
    savscore[j,] <- tsol$score
    savshape[,,j] <- tsol$shape
    savwt[j,] <- tsol$weight
    savtri[j,] <- trii

  }

  rotang <- numeric(nrep)
  if(is.na(meanshape)){
    for(j in 1:nrep){
      dd <- numeric(n)
      for(i in 1:n){
        dd[i] <- shapdist(savshape[,,j],shapedata[,,i])
      }
      rotang[j] <- sqrt(mean(dd^2))
    }
  }else{
    for(j in 1:nrep) rotang[j] <- shapdist(savshape[,,j],meanshape)
  }

  list(shapes=savshape,scores=savscore,rotang=rotang,trios=savtri,weights=savwt)
}

# find weighted solution
tripsol <- function(shapedata,mds,score,trii){

  k <- dim(shapedata)[1]
  m <- 3
  scaled.score <- score*sd(mds$points)
  
                                        # step 0
  lam <- solve(rbind(1,t(mds$points[trii,])),c(1,scaled.score))
  spcas <- procGPA(shapedata[,,trii], scale=TRUE, tangentresiduals = TRUE)
  projshap <- spcas$mshape + matrix(spcas$tan %*% lam, k, m)
  actscore <- scoredat(projshap,shapedata,mds)
                                        # step 1
  lam1 <- solve(rbind(1,t(mds$points[trii,])),c(1,2*scaled.score-actscore))
  projshap1 <- spcas$mshape + matrix(spcas$tan %*% lam1, k, m)
  actscore1 <- scoredat(projshap1,shapedata,mds)

  list(score=actscore1/sd(mds$points),
       weights=lam1,
       shape=projshap1)
}

# compute data from score using PCA (must use approx tangent)
backpca <- function(spca,targscore){
  dimscore <- length(targscore)
#  sizescal <- mean(spca$size)
#  spca$mshape + sizescal*matrix(spca$pcar[,1:dimscore] %*% targscore, spca$k, spca$m)
  spca$mshape + matrix(spca$pcar[,1:dimscore] %*% (targscore*spca$pcasd[1:dimscore]), spca$k, spca$m)
}

# compare shapes

plotface <- function(fm,fm2=NULL,fm3=NULL,view="front", num=TRUE,main="",
                     dirm=c("arrows","points","segments")){
  n <- nrow(fm)
  dirm <- match.arg(dirm)

  cenpts <- c(4,10,32,36)
  lefpts <- c(5,6,7,11,12,17:20,22,27:30,33,37,38)
  cl <- union(cenpts,lefpts)

  lineseg <- matrix(
  c(1, 2, 2, 3, 3, 4, 4, 5, 5, 6, 6, 7, 4, 10, 9, 10, 10, 11, 4, 
    9, 4, 11, 24, 25, 25, 26, 26, 27, 27, 28, 28, 29, 24, 31, 31, 
    32, 32, 33, 33, 29, 34, 35, 35, 36, 36, 37, 37, 38),
                     byrow=TRUE,ncol=2)


  # adjust grimaces
  if(n == 36){
    cl <- cl[cl != 16]
    cl <- cl[cl != 17]
    cl[cl > 17] <- cl[cl > 17] - 2
  }
  allpts <- 1:n
  rigpts <- setdiff(allpts,cl)
  
  if(view == "front") {
    orthproj <- c(3,2)
    selpts <- allpts
  }else{
    orthproj <- c(1,2)
    selpts <- cl
  }

  afm <- rbind(fm[selpts,],fm2[selpts,],fm3[selpts,])
  xr <- range(afm[,orthproj[1]])
  yr <- range(afm[,orthproj[2]])

  maxsp <- max(diff(xr),diff(yr))
  nxlim <- mean(xr)+c(-1,1)*0.5*maxsp
  nylim <- mean(yr)+c(-1,1)*0.5*maxsp
  
  if(is.null(fm2)){
    if(num){
      plot(fm[selpts,orthproj[1]],fm[selpts,orthproj[2]],type="n",xlab="",ylab="",main=main,xlim=nxlim,ylim=nylim)
      text(fm[selpts,orthproj[1]],fm[selpts,orthproj[2]],as.character(selpts))
    }else{
      plot(fm[selpts,orthproj[1]],fm[selpts,orthproj[2]],xlab="",ylab="",main=main,xlim=nxlim,ylim=nylim)
    }
    if(dirm == "segments"){
      segments(fm[lineseg[,1],orthproj[1]],fm[lineseg[,1],orthproj[2]],
               fm[lineseg[,2],orthproj[1]],fm[lineseg[,2],orthproj[2]])
    }
  }else{
    if(is.null(fm3)){
      plot(fm[selpts,orthproj[1]],fm[selpts,orthproj[2]],xlab="",ylab="",main=main,xlim=nxlim,ylim=nylim)
      if(dirm == "arrows"){
        arrows(fm[selpts,orthproj[1]],fm[selpts,orthproj[2]],
               fm2[selpts,orthproj[1]],fm2[selpts,orthproj[2]],
               length=0.1)
      }
      if(dirm == "points"){
        points(fm2[selpts,orthproj[1]],fm2[selpts,orthproj[2]],pch=19)
      }
      if(dirm == "segments"){
        segments(fm[selpts,orthproj[1]],fm[selpts,orthproj[2]],
                 fm2[selpts,orthproj[1]],fm2[selpts,orthproj[2]])
      }
    }else{
      plot(fm[selpts,orthproj[1]],fm[selpts,orthproj[2]],xlab="",ylab="",main=main,xlim=nxlim,ylim=nylim,pch=".")
      points(fm2[selpts,orthproj[1]],fm2[selpts,orthproj[2]],pch=1)
      segments(fm[selpts,orthproj[1]],fm[selpts,orthproj[2]],
               fm2[selpts,orthproj[1]],fm2[selpts,orthproj[2]])
      points(fm3[selpts,orthproj[1]],fm3[selpts,orthproj[2]],pch=19)
      segments(fm[selpts,orthproj[1]],fm[selpts,orthproj[2]],
               fm3[selpts,orthproj[1]],fm3[selpts,orthproj[2]])
    }
  }
}

centroidfun <- function(shapedata,mds,niter,step=0.1,verbose=FALSE){

  dimsol <- ncol(mds$points)
  k <- dim(shapedata)[1]
  m <- dim(shapedata)[2]
  n <- dim(shapedata)[3]

  targscore <- rep(0,dimsol)

  twoshape <- array(NA,c(k,m,2))

  # initial guess
  oldtrii <- sample(n,3)
  tsol <- tripsol(shapedata,mds,targscore,oldtrii)
  optcrit <- cendist(shapedata,tsol$shape)
  twoshape[,,1] <- tsol$shape

  if(verbose) cat(0,optcrit,step,"\n")
  
  # search
  for(j in 1:niter){
    trii <- sample(n,3)
    tsol <- tripsol(shapedata,mds,targscore,trii)
    twoshape[,,2] <- tsol$shape
    stepcrit <- cendist(shapedata,tsol$shape)
    
    if(stepcrit < optcrit){
      outshape <- wavg(twoshape,c(-step,1+step))
      inshape <- wavg(twoshape,c(step,1-step))

      outcrit <- cendist(shapedata,outshape)
      incrit <- cendist(shapedata,inshape)

      isel <- which.min(c(incrit,stepcrit,outcrit))
      if(isel == 1){
        twoshape[,,1] <- inshape
        optcrit <- incrit
      }
      if(isel == 2){
        twoshape[,,1] <- twoshape[,,2]
        optcrit <- stepcrit
      }
      if(isel == 3){
        twoshape[,,1] <- outshape
        optcrit <- outcrit
      }
      step = step * 1.05
    }else{
      outshape <- wavg(twoshape,c(1+step,-step))
      inshape <- wavg(twoshape,c(1-step,step))

      outcrit <- cendist(shapedata,outshape)
      incrit <- cendist(shapedata,inshape)

      isel <- which.min(c(incrit,optcrit,outcrit))
      if(isel == 1){
        twoshape[,,1] <- inshape
        optcrit <- incrit
        step = step * 1.05
      }
      if(isel == 2){
        step = step * 0.9
      }
      if(isel == 3){
        twoshape[,,1] <- outshape
        optcrit <- outcrit
        step = step * 1.05
      }
    }

    oldtrii <- trii
    
    if(verbose) cat(j,optcrit,step,isel,"\n")
  }

  list(shape=twoshape[,,1],optcrit=optcrit)
}

# compute point corresponding to a score
backpco <- function(shapedata,mds,targscore,meanshape,niter,step=0.1,verbose=FALSE){

  dimsol <- ncol(mds$points)
  k <- dim(shapedata)[1]
  m <- dim(shapedata)[2]
  n <- dim(shapedata)[3]

  twoshape <- array(NA,c(k,m,2))

  # initial guess
  oldtrii <- sample(n,3)
  tsol <- tripsol(shapedata,mds,targscore,oldtrii)
  optcrit <- shapdist(meanshape,tsol$shape)
  twoshape[,,1] <- tsol$shape

  if(verbose) cat(0,optcrit,step,"\n")
  
  # search
  for(j in 1:niter){
    trii <- sample(n,3)
    tsol <- tripsol(shapedata,mds,targscore,trii)
    twoshape[,,2] <- tsol$shape
    stepcrit <- shapdist(meanshape,tsol$shape)
    
    if(stepcrit < optcrit){
      outshape <- wavg(twoshape,c(-step,1+step))
      inshape <- wavg(twoshape,c(step,1-step))

      outcrit <- shapdist(meanshape,outshape)
      incrit <- shapdist(meanshape,inshape)

      isel <- which.min(c(incrit,stepcrit,outcrit))
      if(isel == 1){
        twoshape[,,1] <- inshape
        optcrit <- incrit
      }
      if(isel == 2){
        twoshape[,,1] <- twoshape[,,2]
        optcrit <- stepcrit
      }
      if(isel == 3){
        twoshape[,,1] <- outshape
        optcrit <- outcrit
      }
      step = step * 1.05
    }else{
      outshape <- wavg(twoshape,c(1+step,-step))
      inshape <- wavg(twoshape,c(1-step,step))

      outcrit <- shapdist(meanshape,outshape)
      incrit <- shapdist(meanshape,inshape)

      isel <- which.min(c(incrit,optcrit,outcrit))
      if(isel == 1){
        twoshape[,,1] <- inshape
        optcrit <- incrit
        step = step * 1.05
      }
      if(isel == 2){
        step = step * 0.9
      }
      if(isel == 3){
        twoshape[,,1] <- outshape
        optcrit <- outcrit
        step = step * 1.05
      }
    }

    oldtrii <- trii
    currentscore <- scoredat(twoshape[,,1],shapedata,mds)/sd(mds$points)

    # renorm
    if(j %% 2){
      bgi <- which.max(abs(targscore))
      wtadj <- (abs(targscore[bgi])-abs(currentscore[bgi]))/abs(targscore[bgi])
      twoshape[,,2] <- meanshape
      twoshape[,,1] <- wavg(twoshape,c(1+wtadj,-wtadj))
      adjscore <- scoredat(twoshape[,,1],shapedata,mds)/sd(mds$points)
    }else{
      wtadj <- vnorm(targscore)/vnorm(currentscore)-1
      twoshape[,,2] <- meanshape
      twoshape[,,1] <- wavg(twoshape,c(1+wtadj,-wtadj))
      adjscore <- scoredat(twoshape[,,1],shapedata,mds)/sd(mds$points)
    }
    
    if(verbose) cat(j,optcrit,round(currentscore,3),wtadj,round(adjscore,3),step,isel,sep="\t","\n")
  }

  list(shape=twoshape[,,1],optcrit=optcrit)
}

# weighted average of shapes
wavg <- function(shapex,weights){
  ff <- procGPA(shapex)
  apply(ff$rotated,c(1,2),function(x) sum(x*weights))
}

# centroid distance measure
cendist <- function(shapex,targshape){
  n <- dim(shapex)[3]
  dd <- numeric(n)
  for(i in 1:n) dd[i] <- shapdist(shapex[,,i],targshape)
  sqrt(mean(dd^2))
}

# Lele distance
leledist <- function(shape1,shape2){
  dd1 <- dist(shape1)
  dd2 <- dist(shape2)
  sqrt(sum((log(dd1/mean(dd1))-log(dd2/mean(dd2)))^2))
}

# Functions for PCO on functional data

dtwdist <- function(f1,f2,if1,if2){
  dx2 <- (if1-if2)^2/(length(f1)^2)
  dy2 <- (f1[if1]-f2[if2])^2/40^2
  sqrt(mean(dx2+dy2))
}

targetdist <- function(fd,qf,diff=FALSE){
  p <- nrow(fd)
  dv <- numeric(p)
  for(i in 1:p){
    if(diff){
      al <- dtw(diff(qf),diff(fd[i,]),window.type="itakura")
      dv[i] <- dtwdist(qf,fd[i,],al$index1,al$index2)
    }else{
      al <- dtw(qf,fd[i,],window.type="itakura")
      dv[i] <- al$distance/length(al$index1)
    }
  }
  mean(dv)
}

w2f <- function(fun1,fun2,lambda,diff=FALSE){
  if(diff){
    al <- dtw(diff(fun1),diff(fun2),window.type="itakura")
  }else{
    al <- dtw(fun1,fun2,window.type="itakura")
  }
  y <- lambda*fun1[al$index1]+(1-lambda)*fun2[al$index2]
  tx <- lambda*al$index1+(1-lambda)*al$index2
  approx(tx,y,n=length(fun1))$y
}


w3f <- function(fm,lam,fp=c(1,2),diff=FALSE){
  lambda <- lam[fp[1]]/(lam[fp[1]]+lam[fp[2]])
  wtf <- w2f(fm[,fp[1]],fm[,fp[2]],lambda,diff=diff)
  lambda <- lam[fp[1]]+lam[fp[2]]
  w2f(wtf,fm[,-fp],lambda,diff=diff)
}

pcoas <- function(newf,fd,mds,diff=FALSE){
  n <- nrow(mds$points)
  k <- ncol(mds$points)
  ndist <- numeric(n)
  for(i in 1:n){
    if(diff){
      al <- dtw(diff(newf),diff(fd[i,]),window.type="itakura")
      ndist[i] <- dtwdist(newf,fd[i,],al$index1,al$index2)
    }else{
      al <- dtw(newf,fd[i,],window.type="itakura")
      ndist[i] <- al$distance/length(al$index1)
    }
  }
  dv <- -diag(mds$x)/2-ndist^2
  as.vector(t(mds$points) %*% dv)/(mds$eig[1:k] *2)
}


# distance matrix
getdist <- function(fd,diff=FALSE){
  p <- nrow(fd)
  distd <- matrix(0,p,p)
  for(i in 2:p){
    for(j in 1:(i-1)){
      if(diff){
        al <- dtw(diff(fd[i,]),diff(fd[j,]),window.type="itakura")
        nd <- dtwdist(fd[i,],fd[j,],al$index1,al$index2)
      }else{
        dd <- dtw(fd[i,],fd[j,],window.type="itakura")
        nd <- dd$distance/length(dd$index1)
      }
      distd[i,j] <- nd
      distd[j,i] <- nd
    }
  }
  distd
}

# version that covers all triples
backscorea <- function(fd,meanf,targlam,mds,verbose=TRUE,diff=FALSE){
  frep <- 0

  bestsd <- 10000
  n <- nrow(fd)
  scaled.score <- targlam*sd(mds$points)
  savres <- matrix(NA,n^3,4)

  for(i1 in 1:(n-2)){
    for(i2 in (i1+1):(n-1)){
      for(i3 in (i2+1):n){

        trii <- c(i1,i2,i3)
        lam <- solve(rbind(1,t(mds$points[trii,])),c(1,scaled.score))
    
        if(any(lam < 0)){
          next
        }else{
          frep <- frep + 1
        }

        fm <- t(fd[trii,])
        m3f <- w3f(fm,lam,diff=diff)
                                        # recenter
        actl <- pcoas(m3f,fd,mds,diff=diff)
        lam <- solve(rbind(1,t(mds$points[trii,])),c(1,2*scaled.score-actl))
        m3f <- w3f(fm,lam,diff=diff)

        if(diff){
          al <- dtw(diff(meanf),diff(m3f),window.type="itakura")
          dd <- dtwdist(meanf,m3f,al$index1,al$index2)
          savres[frep,] <- c(trii,dd)
        }else{
          al <- dtw(meanf,m3f,window.type="itakura")
          dd <- al$distance/length(al$index1)
          savres[frep,] <- c(trii,dd)
        }
    
        if(dd < bestsd){
          bestsd <- dd
          bests3 <- trii
          albest <- al
          bestfun <- m3f
        }
        if(verbose) cat(trii,dd,"\n",sep="\t")
      }
    }
  }
  savres <- savres[!is.na(savres[,1]),]
  list(dtw=albest,mindist=bestsd,best3=bests3,bestfun=bestfun,frep,savres)
}

cenfuna <- function(fd,mds,verbose=TRUE,diff=FALSE){
  bestd <- 10000
  frep <- 1
  n <- nrow(fd)

  for(i1 in 1:(n-2)){
    for(i2 in (i1+1):(n-1)){
      for(i3 in (i2+1):n){

        trii <- c(i1,i2,i3)
        lam <- solve(rbind(1,t(mds$points[trii,])),c(1,0,0))
    
        if(any(lam < 0)){
          next
        }else{
          frep <- frep + 1
        }

        if(any(lam < 0)) next

        fm <- t(fd[trii,])
        m3f <- w3f(fm,lam,diff=diff)
                                        # recenter
        actl <- pcoas(m3f,fd,mds,diff=diff)
        lam <- solve(rbind(1,t(mds$points[trii,])),c(1,-actl))
        m3f <- w3f(fm,lam,diff=diff)
    
        dd <- targetdist(fd,m3f,diff=diff)
    
        if(dd < bestd){
          bestd <- dd
          bestrii <- trii
          bestmean <- m3f
        }
        if(verbose) cat(trii,dd,"\n",sep="\t")
      }
    }
  }
  list(mean=bestmean,best3=bestrii,bestdist=bestd,frep)
}


cenfunak <- function(fd,mds,trii,verbose=TRUE,diff=FALSE){

  n <- nrow(fd)

  dweights <- solve(rbind(1,t(mds$points[trii,])),c(1,0,0))
    
  fm <- t(fd[trii,])
  m3f <- w3f(fm,dweights,diff=diff)
                                        # recenter
  for(irep in 1:10){
    pscore <- pcoas(m3f,fd,mds,diff=diff)
    if(verbose) cat(pscore/sd(mds$points),dweights,"\n")
    dweights <- solve(rbind(1,t(mds$points[trii,])),c(1,-pscore/irep))
#    dweights <- dweights+c(1,-1,0)/100
    m3f <- w3f(fm,dweights,diff=diff)
  }

    
  dd <- targetdist(fd,m3f,diff=diff)
    
  list(mean=m3f,bestdist=dd,weights=dweights,score=pscore/sd(mds2$points))
}

backscoreak <- function(fd,meanf,targlam,mds,trii,verbose=TRUE,diff=FALSE){

  n <- nrow(fd)
  scaled.score <- targlam*sd(mds$points)

  lam <- solve(rbind(1,t(mds$points[trii,])),c(1,scaled.score))
    
  fm <- t(fd[trii,])
  m3f <- w3f(fm,lam,diff=diff)

  actl <- pcoas(m3f,fd,mds,diff=diff)

                                        # recenter
  if(FALSE){
    for(delta in seq(0.5,1.5,length=11)){
      if(verbose){
        cat(delta,actscore/sd(mds$points),"\n")
      }
    
    }
  }
  delta <- 1
  adjscore <- scaled.score + delta*(scaled.score - actl)
  lam <- solve(rbind(1,t(mds$points[trii,])),c(1,adjscore))
  m3f <- w3f(fm,lam,diff=diff)
  actscore <- pcoas(m3f,fd,mds,diff=diff)
  
  if(diff){
    al <- dtw(diff(meanf),diff(m3f),window.type="itakura")
    dd <- dtwdist(meanf,m3f,al$index1,al$index2)      
  }else{
    al <- dtw(meanf,m3f,window.type="itakura")
    dd <- al$distance/length(al$index1)
  }
    
  list(dtw=al,mindist=dd,bestfun=m3f,weights=lam)
}

# Functions for mixed variables analysis

dxtox <- function(x){
  va <- c(rep(0,3),rep(1,7))
  vm <- c(1,1,1,c(7,8,5)/20,c(3,5,6,6)/20)
  vv <- (x+va)*vm
  if(is.matrix(x)){
    usen <- c("hayfield","both","grazing")[apply(vv[4:6,],2,which.max)]
    manag <- c("BF","HF","NM","SF")[apply(vv[7:10,],2,which.max)]
    data.frame(A1=vv[1,],moisture=vv[2,],manure=vv[3,],use=usen,manag=manag)
  }else{
    usen <- c("hayfield","both","grazing")[which.max(vv[4:6])]
    manag <- c("BF","HF","NM","SF")[which.max(vv[7:10])]
    data.frame(A1=vv[1],moisture=vv[2],manure=vv[3],use=usen,manag=manag)
  }
}

discex <- function(x,vals){
  cutp <- vals[-1] - diff(vals)/2
  cutp <- c(min(x)-1,cutp,max(x)+1)
  vals[unclass(cut(x,cutp))]
}




gendata <- function(x){
  cm <- colMeans(dunedata$envir[,1:3])
  cs <- sd(dunedata$envir[,1:3])*sqrt(20/19)
  va <- c(rep(0,3),rep(1,7))
  vm <- c(1,1,1,c(7,8,5)/20,c(3,5,6,6)/20)
  vv <- (x+va)*vm
  genlev <- function(v){
    v[v < 0] <- 0
    which(as.logical(rmultinom(1,1,v)))
  }
  if(is.matrix(x)){
    vv[1:3,] <- sweep(vv[1:3,],1,cs,"*")
    vv[1:3,] <- sweep(vv[1:3,],1,cm,"+")
    usen <- c("hayfield","both","grazing")[apply(vv[4:6,],2,genlev)]
    manag <- c("BF","HF","NM","SF")[apply(vv[7:10,],2,genlev)]
#    data.frame(A1=round(vv[1,],1),moisture=discex(vv[2,],1:5),manure=discex(vv[3,],0:4),use=usen,manag=manag)
    data.frame(A1=vv[1,],moisture=vv[2,],manure=vv[3,],use=usen,manag=manag)
  }else{
    usen <- c("hayfield","both","grazing")[genlev(vv[4:6])]
    manag <- c("BF","HF","NM","SF")[genlev(vv[7:10])]
    data.frame(A1=round(vv[1],1),moisture=vv[2],manure=vv[3],use=usen,manag=manag)
  }
}
