/***************************************************************************
 * The Robust Colour Morphological Gradient (RCMG) for edge detection      *
 * in multichannel images                                                  *
 *                                                                         *
 * Adrian N. Evans                                                         *
 * University of Bath                                                      *
 * July 2007                                                               *
 * Copyright 2005-2007.  All rights reserved.                              *
 *                                                                         *
 * The RCMG find the median centered difference for a given window size    *
 * by rejecting the s pairs of pixels that are furthest apart. For full    *
 * details of its operation see A.N. Evans and X.U. Liu, A Morphological   *
 * Gradient Approach to Color Edge Detection, IEEE Transactions on Image   *
 * Processing, 15(6), pp. 1454-1463, June 2006.                            *
 *																		   *
 * This implementation uses the L2 norm                                    *
 *																		   *
 * This file contains the mex wrapper									   *
 *																		   *
 ***************************************************************************/

#include "mex.h"
#include "rcmg.h"
#include <math.h>
#include <memory.h>
#include "matrix.h"


void mexFunction(int nlhs, mxArray *plhs[], int nrhs,
const mxArray *prhs[])
{
double *img, *out_m, *out_dir;
int number_of_dimensions;
const mwSize;
const *dims;
int rows, cols, channels, mask, s;
bool direction =0;

/* Declarations*/
if (nrhs < 1) {
	mexErrMsgTxt("At least one input argument required.\n");
} else if (nlhs > 2) {
	mexErrMsgTxt("Too many output arguments.\n");
}
/* Check input image is a double */
if (!mxIsDouble(prhs[0])) {
	mexErrMsgTxt("Input image should be double.\n");
}

if (nrhs >=2) {
	/* Check to make sure mask is a scalar. */
	if (!mxIsDouble(prhs[1]) || mxIsComplex(prhs[1]) ||
		mxGetN(prhs[1])*mxGetM(prhs[1]) != 1) {
		mexErrMsgTxt("Input mask must be a scalar.");
	}
	mask = mxGetScalar(prhs[1]);
}
else {mask = 3;}


if (nrhs >=3) {
	/* Check to make sure s is a scalar. */
	if (!mxIsDouble(prhs[2]) || mxIsComplex(prhs[2]) ||
		mxGetN(prhs[2])*mxGetM(prhs[2]) != 1) {
		mexErrMsgTxt("Input s must be a scalar.");
	}
	s = mxGetScalar(prhs[2]);
}
else {s = 0;}

/* Check input image is 3 dimensional */
number_of_dimensions = mxGetNumberOfDimensions(prhs[0]);
if (number_of_dimensions >3) {
    mexErrMsgTxt("Input image can not have more than 3 dimensions!");
}

img = mxGetPr(prhs[0]);
dims = mxGetDimensions(prhs[0]);
rows = dims[0];
cols = dims[1];

if (number_of_dimensions < 3) {
	channels = 1;
}
else {
	channels = dims[2];
}


/* Create output matrices: */
plhs[0] = mxCreateDoubleMatrix(rows, cols, mxREAL);
out_m = mxGetPr(plhs[0]);
if (nlhs > 1) {
	plhs[1] = mxCreateDoubleMatrix(rows, cols, mxREAL);
	out_dir = mxGetPr(plhs[1]);
	direction = 1;
}

rcmg(img, rows, cols, channels, mask, s, direction, out_m, out_dir);

return;
}


