/***************************************************************************
 * The Robust Colour Morphological Gradient (RCMG) for edge detection      *
 * in multichannel images                                                  *
 *                                                                         *
 * Adrian N. Evans                                                         *
 * University of Bath                                                      *
 * July 2007                                                               *
 * Copyright 2005-2007.  All rights reserved.                              *
 *                                                                         *
 * The RCMG find the median centered difference for a given window size    *
 * by rejecting the s pairs of pixels that are furthest apart. For full    *
 * details of its operation see A.N. Evans and X.U. Liu, A Morphological   *
 * Gradient Approach to Color Edge Detection, IEEE Transactions on Image   *
 * Processing, 15(6), pp. 1454-1463, June 2006.                            *
 *																		   *
 * This implementation uses the L2 norm                                    *
 *																		   *
 ***************************************************************************/


#include <stdio.h>
#include <stdlib.h>
#include "mex.h"
#include "rcmg.h"
#include <math.h>
#include <memory.h>
#include "matrix.h"

void rcmg(double *img, int rows, int cols, int channels, int mask, int s, bool direction, double *out_m, double *out_dir)
{
	int i, j, k;
	int imask,jmask;
	int index,loop;
	int tmp1,tmp2;
	int hs = (mask-1)/2;
	int m2 = mask*mask;
	int v1, v2, location;
	int *locations;
	bool *not_removed;
	double max_dist, max_dist2;
	double *max_dists,*copyof_max_dists;
	double **window;
	double **dists;

	max_dists = malloc(m2*sizeof(double));
	copyof_max_dists = malloc(m2*sizeof(double));
	locations = malloc(m2*sizeof(int));
	not_removed = malloc(m2*sizeof(bool));
	dists = (double **)malloc(sizeof(double **)*m2);
	window = (double **)malloc(sizeof(double *)*m2);
	for(i=0;i<m2;i++) {
		dists[i] = malloc(m2*sizeof(double));
		window[i] = malloc(channels*sizeof(double));	
	}
	for(i=0;i<m2;i++) 
		for(j=0;j<m2;j++)
			dists[i][j] = -1.0;

		for (i=hs; i<rows-hs; i++) {
			for (j=hs; j<cols-hs; j++) {
				for (k=0; k<m2; k++)
					not_removed[k] = 1;
			
			index=0;
			for (jmask=j-hs; jmask<=j+hs; jmask++) {		/*fill col-1 first */
				for (imask=i-hs; imask<=i+hs; imask++) {
					for (k=0; k<channels; k++){
						window[index][k] = img[(k*rows*cols) +jmask*rows +imask];					
					}
					index++;
				}
			}						
			if (j==hs) /*for first col	*/
				l2norms(window, dists, max_dists, locations, m2, channels);
			else 
				updatel2norms(window, dists, max_dists, locations, m2, mask, channels);

			max_dist = -1.0;
			for (k=0; k<m2-1; k++) {			/*m2-1 as last row only -1s */
				if (max_dists[k] > max_dist) {
					max_dist = max_dists[k];
					v1 = k;						/* Note: v1 and v2 also needed for direction */
					v2 = locations[k];							
				}
			}	
			
			if (s>0) { /* true if any pairs of vector going to be rejected */
				for (tmp1=0; tmp1<m2; tmp1++) {
					copyof_max_dists[tmp1] = max_dists[tmp1];
				}
				for (index=1; index<=s; index++) {
					not_removed[v1] = 0;
					not_removed[v2] = 0;
					max_dist = -1.0;
					for (k=0; k<m2-1; k++) {
						if (not_removed[k]) {
							if (copyof_max_dists[k] > max_dist) {							
								if (not_removed[locations[k]]) {	/* vector locations[k] not removed */
									max_dist = copyof_max_dists[k];
									v1 = k;
									v2 = locations[k];							
								}
								else {		/*update max_dists */
									max_dist2=-1.0;
									for (loop=k+1; loop<m2; loop++) {
										if (dists[k][loop] > max_dist2) {
											if (not_removed[loop]) {
												max_dist2 = dists[k][loop];
												location = loop;											
											}
										}
										copyof_max_dists[k] = max_dist2;
									}
									/* repeat check for > max_dist2 with updated max_dists[k] */
									if (copyof_max_dists[k] > max_dist) {						
										max_dist = copyof_max_dists[k];
										v1 = k;
										v2 = location;
									}
								}								
							}
						}
					}
				}	
			}

			out_m[j*rows +i] = max_dist;	
			if (direction) {
				out_dir[j*rows +i] = find_dir(v1,v2,mask);
			}
			

		}
	}
}


void l2norms(double **window, double **dists, double *max_dists, int *locations, int m2, int chans)
{
	int i,j,k,location;
	double norm,temp,max_norm;
	
	for (i=0; i<m2-1; i++) {
		max_norm = -1.0;
		for (j=i+1; j<m2; j++) {
			norm =0;
			for (k=0; k<chans; k++){
				temp = window[i][k] - window[j][k];
				norm += temp*temp;
			}
			norm = sqrt(norm);
			if (norm > max_norm) {
				max_norm = norm;
				location = j;
			}
			dists[i][j] = norm;			
		}
		max_dists[i] = max_norm;
		locations[i] = location;	
	}
}

void updatel2norms(double **window, double **dists, double *max_dists, int *locations, int m2, int mask, int chans)
{
	int i,j,k,location;
	double norm,temp,max_norm;

	for (i=0; i<m2-mask-1; i++) {		/* copy existing values into new positions */
			max_dists[i] = max_dists[i+mask];		/* copy max dists and location and */
			locations[i] = locations[i+mask]-mask;	/* adjust by mask for move */
			for (j=i+1; j<m2-mask; j++) {	/* copy dists */
				dists[i][j] = dists[i+mask][j+mask];
			} 

			max_norm = max_dists[i];//-1.0;	/* set max_norm to -1, so that can find max_dist of remaining in next loop */
			for (j=i+1; j<m2-mask; j++) {	/* copy dists */
				dists[i][j] = dists[i+mask][j+mask];
				if ( dists[i][j] > max_dists[i]) {
					max_dists[i] = dists[i][j];
					locations[i] = j;
				}
			}
	}

	for (i=m2-mask-1; i<m2-1; i++)	/* clear remaining max_dists */
		max_dists[i] = -1.0;
	
	for (i=0; i<m2-1; i++) {	/* add new values and update max_dists */
		for (j=max(i+1,mask); j<m2; j++) {
			norm = 0;
			for (k=0; k<chans; k++){
				temp = window[i][k] - window[j][k];
				norm += temp*temp;
			}
			dists[i][j] = sqrt(norm);
			if (dists[i][j] > max_dists[i]) {
				max_dists[i] = dists[i][j];
				locations[i] = j;
			}
		}
	}
}


double find_dir(int v1, int v2, int mask)
{
	int r1,r2,c1,c2;

	c1 = floor(v1/mask)-1;
	c2 = floor(v2/mask)-1;
	r1 = v1%mask -1;
	r2 = v2%mask -1;

	return (double)atan((double)(r2-r1)/(double)(c2-c1)); 
}

