#define NRANSI
#include "nrutil.h"
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <complex.h>


extern int nvar;
extern double x1,x2;

extern int kmax,kount;
extern double *xp,**yp,dxsav;

void rkck(double y[], double dydx[], int n, double x, double h, double yout[],
	  double yerr[], void (*derivs)(double, double [], double []))
{
  int i;
  static double a2=0.2,a3=0.3,a4=0.6,a5=1.0,a6=0.875,b21=0.2,
                b31=3.0/40.0,b32=9.0/40.0,b41=0.3,b42 = -0.9,b43=1.2,
                b51 = -11.0/54.0, b52=2.5,b53 = -70.0/27.0,b54=35.0/27.0,
                b61=1631.0/55296.0,b62=175.0/512.0,b63=575.0/13824.0,
                b64=44275.0/110592.0,b65=253.0/4096.0,c1=37.0/378.0,
                c3=250.0/621.0,c4=125.0/594.0,c6=512.0/1771.0,
                dc5 = -277.00/14336.0;
  double dc1=c1-2825.0/27648.0,dc3=c3-18575.0/48384.0,
         dc4=c4-13525.0/55296.0,dc6=c6-0.25;
  double *ak2,*ak3,*ak4,*ak5,*ak6,*ytemp;

  ak2=dvector(1,n);
  ak3=dvector(1,n);
  ak4=dvector(1,n);
  ak5=dvector(1,n);
  ak6=dvector(1,n);
  ytemp=dvector(1,n);
  for (i=1;i<=n;i++)
     ytemp[i]=y[i]+b21*h*dydx[i];
  (*derivs)(x+a2*h,ytemp,ak2);
  
  for (i=1;i<=n;i++)
     ytemp[i]=y[i]+h*(b31*dydx[i]+b32*ak2[i]);
  (*derivs)(x+a3*h,ytemp,ak3);
  
  for (i=1;i<=n;i++)
     ytemp[i]=y[i]+h*(b41*dydx[i]+b42*ak2[i]+b43*ak3[i]);
  (*derivs)(x+a4*h,ytemp,ak4);
  
  for (i=1;i<=n;i++)
     ytemp[i]=y[i]+h*(b51*dydx[i]+b52*ak2[i]+b53*ak3[i]+b54*ak4[i]);
  (*derivs)(x+a5*h,ytemp,ak5);
 
  for (i=1;i<=n;i++)
     ytemp[i]=y[i]+h*(b61*dydx[i]+b62*ak2[i]+b63*ak3[i]+b64*ak4[i]+b65*ak5[i]);
  (*derivs)(x+a6*h,ytemp,ak6);

  for (i=1;i<=n;i++)
    yout[i]=y[i]+h*(c1*dydx[i]+c3*ak3[i]+c4*ak4[i]+c6*ak6[i]);

  for (i=1;i<=n;i++)
    yerr[i]=h*(dc1*dydx[i]+dc3*ak3[i]+dc4*ak4[i]+dc5*ak5[i]+dc6*ak6[i]);

  free_dvector(ytemp,1,n);
  free_dvector(ak6,1,n);
  free_dvector(ak5,1,n);
  free_dvector(ak4,1,n);
  free_dvector(ak3,1,n);
  free_dvector(ak2,1,n);
}


#define SAFETY 0.9
#define PGROW -0.2
#define PSHRNK -0.25
#define ERRCON 1.89e-4


void rkqs(double y[],double dydx[], int n, double *x, double htry, double eps,
	  double yscal[], double *hdid, double *hnext,
	  void (*derivs)(double, double [], double []))
{
  void rkck(double y[], double dydx[], int n, double x, double h, double yout[], 
            double yerr[], void (*derivs)(double, double [], double []));
  int i;
  double errmax,h,htemp,xnew,*yerr,*ytemp;

  yerr=dvector(1,n);
  ytemp=dvector(1,n);
  h=htry;
  for (;;) 
  {
     rkck(y,dydx,n,*x,h,ytemp,yerr,derivs);
     errmax=0.0;
     for (i=1;i<=n;i++) 
        errmax=FMAX(errmax,fabs(yerr[i]/yscal[i]));
     errmax /= eps;
     if (errmax <= 1.0) 
        break;
     htemp=SAFETY*h*pow(errmax,PSHRNK);
     h=(h >= 0.0 ? FMAX(htemp,0.1*h) : FMIN(htemp,0.1*h));
     xnew=(*x)+h;
     if (xnew == *x) 
        nrerror("stepsize underflow in rkqs");
  }
  if (errmax > ERRCON) 
     *hnext=SAFETY*h*pow(errmax,PGROW);
  else 
     *hnext=5.0*h;
  
  *x += (*hdid=h); 
  for (i=1;i<=n;i++) 
     y[i]=ytemp[i];
  
  free_dvector(ytemp,1,n);
  free_dvector(yerr,1,n);
}
#undef SAFETY
#undef PGROW
#undef PSHRNK
#undef ERRCON


#define MAXSTP 1000000000
#define TINY 1.0e-30


void odeint(double ystart[], int nvar, double xx1, double xx2, double eps, 
            double h1, double hmin, int *nok, int *nbad,
	    void (*derivs)(double, double [], double []),
	    void (*rkqs)(double [], double [], int, double *, double, double, 
                 double [], double *, double *, void (*)(double, double [], 
                 double [])))
{
  int nstp,i;
  double xsav,x,hnext,hdid,h;
  double *yscal,*y,*dydx;

  yscal=dvector(1,nvar);
  y=dvector(1,nvar);
  dydx=dvector(1,nvar);
  x=xx1;
  h=SIGN(h1,xx2-xx1);
  *nok = (*nbad) = kount = 0;
  for (i=1;i<=nvar;i++) 
     y[i]=ystart[i];
  if (kmax > 0) 
     xsav=x-dxsav*2.0;
  for (nstp=1;nstp<=MAXSTP;nstp++) 
  {
     (*derivs)(x,y,dydx);
     for (i=1;i<=nvar;i++)
        yscal[i]=fabs(y[i])+fabs(dydx[i]*h)+TINY;
    
     if (kmax > 0 && kount < kmax-1 && fabs(x-xsav) > fabs(dxsav)) 
     {
        xp[++kount]=x;
        for (i=1;i<=nvar;i++) 
           yp[i][kount]=y[i];
        xsav=x;
     }
     if ((x+h-xx2)*(x+h-xx1) > 0.0) 
        h=xx2-x;
     (*rkqs)(y,dydx,nvar,&x,h,eps,yscal,&hdid,&hnext,derivs);
     if (hdid == h) 
        ++(*nok); 
     else 
        ++(*nbad);
     if ((x-xx2)*(xx2-xx1) >= 0.0) 
     {
        for (i=1;i<=nvar;i++) 
           ystart[i]=y[i];
        if (kmax) 
        {
	   xp[++kount]=x;
	   for (i=1;i<=nvar;i++) 
              yp[i][kount]=y[i];
        }
        free_dvector(dydx,1,nvar);
        free_dvector(y,1,nvar);
        free_dvector(yscal,1,nvar);
        return;
     }
     if (fabs(hnext) <= hmin) nrerror("Step size too small in odeint");
     h=hnext;
  }
  nrerror("Too many steps in routine odeint");
}
#undef MAXSTP
#undef TINY

#define ITMAX 100
#define EPS 3.0e-8

double zbrent(double (*func)(double), double x1, double x2, double tol)
{
  int iter;
  double a=x1,b=x2,c=x2,d,e,min1,min2;
  double fa=(*func)(a),fb=(*func)(b),fc,p,q,r,s,tol1,xm;

  if ((fa > 0.0 && fb > 0.0) || (fa < 0.0 && fb < 0.0))
     nrerror("Root must be bracketed in zbrent");
  fc=fb;
  for (iter=1;iter<=ITMAX;iter++) 
  {
     if ((fb > 0.0 && fc > 0.0) || (fb < 0.0 && fc < 0.0)) 
     {
        c=a;
        fc=fa;
        e=d=b-a;
     }
     if (fabs(fc) < fabs(fb)) 
     {
        a=b;
        b=c;
        c=a;
        fa=fb;
        fb=fc;
        fc=fa;
     }
     tol1=2.0*EPS*fabs(b)+0.5*tol;
     xm=0.5*(c-b);
     if (fabs(xm) <= tol1 || fb == 0.0) 
        return b;
     if (fabs(e) >= tol1 && fabs(fa) > fabs(fb)) 
     {
        s=fb/fa;
        if (a == c) 
        {
	   p=2.0*xm*s;
	   q=1.0-s;
        } 
        else 
        {
	   q=fa/fc;
	   r=fb/fc;
	   p=s*(2.0*xm*q*(q-r)-(b-a)*(r-1.0));
	   q=(q-1.0)*(r-1.0)*(s-1.0);
        }
        if (p > 0.0) 
           q = -q;
        p=fabs(p);
        min1=3.0*xm*q-fabs(tol1*q);
        min2=fabs(e*q);
        if (2.0*p < (min1 < min2 ? min1 : min2)) 
        {
           e=d;
	   d=p/q;
        } 
        else 
        {
	   d=xm;
	   e=d;
        }
     } 
     else 
     {
        d=xm;
        e=d;
     }
     a=b;
     fa=fb;
     if (fabs(d) > tol1)
        b += d;
     else
        b += SIGN(tol1,xm);
     fb=(*func)(b);
  }
  nrerror("Maximum number of iterations exceeded in zbrent");
  return 0.0;
}
#undef ITMAX
#undef EPS

extern int kmax,kount;
extern double *xp,**yp,dxsav;
extern double eps;

void shoot(int n, double v[], double f[])
{
  void derivs(double x, double y[], double dydx[]);
  void load(double x1, double v[], double y[]);
  void odeint(double ystart[], int nvar, double x1, double x2,
	      double eps, double h1, double hmin, int *nok, int *nbad,
	      void (*derivs)(double, double [], double []),
	      void (*rkqs)(double [], double [], int, double *, double, double,
			   double [], double *, double *, void (*)(double, 
                           double [], double [])));
  void rkqs(double y[], double dydx[], int n, double *x,
	    double htry, double eps, double yscal[], double *hdid, double *hnext,
	    void (*derivs)(double, double [], double []));
  void score(double xf, double y[], double f[]);
  int nbad,nok;
  double h1,hmin=0.0,*y;

  y=vector(1,nvar);
  kmax=0;
  load(x1,v,y);
  h1=(x2-x1)/100.0;
  odeint(y,nvar,x1,x2,eps,h1,hmin,&nok,&nbad,derivs,rkqs);
  score(x2,y,f);
  free_vector(y,1,nvar);
}

void shoot2(int n, double v[], double f[])
{
  void derivs(double x, double y[], double dydx[]);
  void load(double x1, double v[], double y[]);
  void odeint(double ystart[], int nvar, double x1, double x2,
	      double eps, double h1, double hmin, int *nok, int *nbad,
	      void (*derivs)(double, double [], double []),
	      void (*rkqs)(double [], double [], int, double *, double, double,
			   double [], double *, double *, void (*)(double, 
                           double [], double [])));
  void rkqs(double y[], double dydx[], int n, double *x,
	    double htry, double eps, double yscal[], double *hdid, double *hnext,
	    void (*derivs)(double, double [], double []));
  void score2(double xf, double y[], double f[]);
  int nbad,nok;
  double h1,hmin=0.0,*y;

  y=vector(1,nvar);
  kmax=0;
  load(x1,v,y);
  h1=(x2-x1)/100.0;
  odeint(y,nvar,x1,x2,eps,h1,hmin,&nok,&nbad,derivs,rkqs);
  score2(x2,y,f);
  free_vector(y,1,nvar);
}

#define EPS 1.0e-10

void fdjac(int n, double x[], double fvec[], double **df,
	   void (*vecfunc)(int, double [], double []))
{
  int i,j;
  double h,temp,*f;

  f=vector(1,n);
  for (j=1;j<=n;j++) 
  {
     temp=x[j];
     h=EPS*fabs(temp);
     if (h == 0.0) 
        h=EPS;
     x[j]=temp+h;
     h=x[j]-temp;
     (*vecfunc)(n,x,f);
     x[j]=temp;
     for (i=1;i<=n;i++) 
        df[i][j]=(f[i]-fvec[i])/h;
  }
  free_vector(f,1,n);
}
#undef EPS


extern int nn;
extern double *fvec;
extern void (*nrfuncv)(int n, double v[], double f[]);

double fmin_(double x[])
{
  int i;
  double sum;

  (*nrfuncv)(nn,x,fvec);
  for (sum=0.0,i=1;i<=nn;i++) 
     sum += SQR(fvec[i]);
  return 0.5*sum;
}

#define ALF 1.0e-4
#define TOLX 1.0e-10

void lnsrch(int n, double xold[], double fold, double g[], double p[], double x[],
	    double *f, double stpmax, int *check, double (*func)(double []))
{
  int i;
  double a,alam,alam2,alamin,b,disc,f2,rhs1,rhs2,slope,sum,temp,
    test,tmplam;

  *check=0;
  for (sum=0.0,i=1;i<=n;i++) 
     sum += p[i]*p[i];
  sum=sqrt(sum);
  if (sum > stpmax)
     for (i=1;i<=n;i++) 
        p[i] *= stpmax/sum;
  for (slope=0.0,i=1;i<=n;i++)
     slope += g[i]*p[i];
  if (slope >= 0.0) 
     nrerror("Roundoff problem in lnsrch.");
  test=0.0;
  for (i=1;i<=n;i++) 
  {
     temp=fabs(p[i])/FMAX(fabs(xold[i]),1.0);
     if (temp > test) 
        test=temp;
  }
  alamin=TOLX/test;
  alam=1.0;
  for (;;) 
  {
     for (i=1;i<=n;i++) 
        x[i]=xold[i]+alam*p[i];
     *f=(*func)(x);
     if (alam < alamin) 
     {
        for (i=1;i<=n;i++) 
           x[i]=xold[i];
        *check=1;
        return;
     } 
     else if (*f <= fold+ALF*alam*slope) 
        return;
     else 
     {
        if (alam == 1.0)
	   tmplam = -slope/(2.0*(*f-fold-slope));
        else 
        {
	   rhs1 = *f-fold-alam*slope;
	   rhs2=f2-fold-alam2*slope;
	   a=(rhs1/(alam*alam)-rhs2/(alam2*alam2))/(alam-alam2);
	   b=(-alam2*rhs1/(alam*alam)+alam*rhs2/(alam2*alam2))/(alam-alam2);
	   if (a == 0.0) 
              tmplam = -slope/(2.0*b);
	   else 
           {
	      disc=b*b-3.0*a*slope;
	      if (disc < 0.0) 
                 tmplam=0.5*alam;
	      else if (b <= 0.0) 
                 tmplam=(-b+sqrt(disc))/(3.0*a);
	      else 
                 tmplam=-slope/(b+sqrt(disc));
	   }
	   if (tmplam > 0.5*alam)
	      tmplam=0.5*alam;
        }
     }
     alam2=alam;
     f2 = *f;
     alam=FMAX(tmplam,0.1*alam);
  }
}
#undef ALF
#undef TOLX


void lubksb(double **a, int n, int *indx, double b[])
{
  int i,ii=0,ip,j;
  double sum;

  for (i=1;i<=n;i++) 
  {
     ip=indx[i];
     sum=b[ip];
     b[ip]=b[i];
     if (ii)
     {
        for (j=ii;j<=i-1;j++) 
           sum -= a[i][j]*b[j];
     }
     else if (sum) 
        ii=i;
    b[i]=sum;
  }
  for (i=n;i>=1;i--) 
  {
     sum=b[i];
     for (j=i+1;j<=n;j++) 
        sum -= a[i][j]*b[j];
     b[i]=sum/a[i][i];
  }
}

#define TINY 1.0e-20

void ludcmp(double **a, int n, int *indx, double *d)
{
  int i,imax,j,k;
  double big,dum,sum,temp;
  double *vv;

  vv=vector(1,n);
  *d=1.0;
  for (i=1;i<=n;i++) 
  {
     big=0.0;
     for (j=1;j<=n;j++)
        if ((temp=fabs(a[i][j])) > big) 
           big=temp;
     if (big == 0.0) 
        nrerror("Singular matrix in routine ludcmp");
     vv[i]=1.0/big;
  }
  for (j=1;j<=n;j++) 
  {
     for (i=1;i<j;i++) 
     {
        sum=a[i][j];
        for (k=1;k<i;k++) 
           sum -= a[i][k]*a[k][j];
        a[i][j]=sum;
     }
     big=0.0;
     for (i=j;i<=n;i++) 
     {
        sum=a[i][j];
        for (k=1;k<j;k++)
	   sum -= a[i][k]*a[k][j];
        a[i][j]=sum;
        if ( (dum=vv[i]*fabs(sum)) >= big) 
        {
	   big=dum;
	   imax=i;
        }
     }
     if (j != imax) 
     {
        for (k=1;k<=n;k++) 
        {
	   dum=a[imax][k];
	   a[imax][k]=a[j][k];
	   a[j][k]=dum;
        }
        *d = -(*d);
        vv[imax]=vv[j];
     }
     indx[j]=imax;
     if (a[j][j] == 0.0) 
        a[j][j]=TINY;
     if (j != n) 
     {
        dum=1.0/(a[j][j]);
        for (i=j+1;i<=n;i++) a[i][j] *= dum;
     }
  }
  free_vector(vv,1,n);
}
#undef TINY

#define MAXITS 200
#define TOLF 1.0e-8
#define TOLMIN 1.0e-6
#define TOLX 1.0e-8
#define STPMX 100.0

int nn;
double *fvec;
void (*nrfuncv)(int n, double v[], double f[]);
#define FREERETURN {free_vector(fvec,1,n);free_vector(xold,1,n);	\
    free_vector(p,1,n);free_vector(g,1,n);free_matrix(fjac,1,n,1,n);	\
    free_ivector(indx,1,n);return;}

void newt(double x[], int n, int *check,
	  void (*vecfunc)(int, double [], double []))
{
  int i,its,j,*indx;
  double d,den,f,fold,stpmax,sum,temp,test,**fjac,*g,*p,*xold;

  indx=ivector(1,n);
  fjac=matrix(1,n,1,n);
  g=vector(1,n);
  p=vector(1,n);
  xold=vector(1,n);
  fvec=vector(1,n);
  nn=n;
  nrfuncv=vecfunc;
  f=fmin_(x);
  test=0.0;
  for (i=1;i<=n;i++)
     if (fabs(fvec[i]) > test) 
        test=fabs(fvec[i]);
  if (test < 0.01*TOLF) 
  {
     *check=0;
     fprintf(stderr, "TOLF\n");		  
     FREERETURN
  }
  for (sum=0.0,i=1;i<=n;i++) 
     sum += SQR(x[i]);
  stpmax=STPMX*FMAX(sqrt(sum),(double)n);
  for (its=1;its<=MAXITS;its++) 
  {
     fdjac(n,x,fvec,fjac,vecfunc);
     for (i=1;i<=n;i++) 
     {
        for (sum=0.0,j=1;j<=n;j++) 
           sum += fjac[j][i]*fvec[j];
        g[i]=sum;
     }
     for (i=1;i<=n;i++) 
        xold[i]=x[i];
     fold=f;
     for (i=1;i<=n;i++) 
        p[i] = -fvec[i];
     ludcmp(fjac,n,indx,&d);
     lubksb(fjac,n,indx,p);
     lnsrch(n,xold,fold,g,p,x,&f,stpmax,check,fmin_);
     test=0.0;
     for (i=1;i<=n;i++)
        if (fabs(fvec[i]) > test) 
           test=fabs(fvec[i]);
     if (test < TOLF) 
     {
        fprintf(stderr, "TOLF\n");
        *check=0;
        FREERETURN
     }
     if (*check) 
     {
        test=0.0;
        den=FMAX(f,0.5*n);
        for (i=1;i<=n;i++) 
        {
	   temp=fabs(g[i])*FMAX(fabs(x[i]),1.0)/den;
	   if (temp > test) 
              test=temp;
        }
        *check=(test < TOLMIN ? 1 : 0);
        if (*check) 
           fprintf(stderr, "TOLMIN\n");
        FREERETURN
     }
     test=0.0;
     for (i=1;i<=n;i++) 
     {
        temp=(fabs(x[i]-xold[i]))/FMAX(fabs(x[i]),1.0);
        if (temp > test) 
           test=temp;
     }
     if (test < TOLX) 
     {
        fprintf(stderr, "TOLX\n"); 
        FREERETURN
     }
  }
  nrerror("MAXITS exceeded in newt");
}
#undef MAXITS
#undef TOLF
#undef TOLMIN
#undef TOLX
#undef STPMX
#undef FREERETURN

#define ITMAX 100
#define CGOLD 0.3819660
#define ZEPS 1.0e-10
#define SHFT(a,b,c,d) (a)=(b);(b)=(c);(c)=(d);

double brent(double ax, double bx, double cx, double (*f)(double), double tol,
	     double *xmin)
{
  int iter;
  double a,b,d,etemp,fu,fv,fw,fx,p,q,r,tol1,tol2,u,v,w,x,xm;
  double e=0.0;

  a=(ax < cx ? ax : cx);
  b=(ax > cx ? ax : cx);
  x=w=v=bx;
  fw=fv=fx=(*f)(x);
  for (iter=1;iter<=ITMAX;iter++) 
  {
     xm=0.5*(a+b);
     tol2=2.0*(tol1=tol*fabs(x)+ZEPS);
     if (fabs(x-xm) <= (tol2-0.5*(b-a))) 
     {
        *xmin=x;
        return fx;
     }
     if (fabs(e) > tol1) 
     {
        r=(x-w)*(fx-fv);
        q=(x-v)*(fx-fw);
        p=(x-v)*q-(x-w)*r;
        q=2.0*(q-r);
        if (q > 0.0) 
           p = -p;
        q=fabs(q);
        etemp=e;
        e=d;
        if (fabs(p) >= fabs(0.5*q*etemp) || p <= q*(a-x) || p >= q*(b-x))
	   d=CGOLD*(e=(x >= xm ? a-x : b-x));
        else 
        {
	   d=p/q;
	   u=x+d;
	   if (u-a < tol2 || b-u < tol2)
	      d=SIGN(tol1,xm-x);
        }
     } 
     else 
     {
        d=CGOLD*(e=(x >= xm ? a-x : b-x));
     }
     u=(fabs(d) >= tol1 ? x+d : x+SIGN(tol1,d));
     fu=(*f)(u);
     if (fu <= fx) 
     {
        if (u >= x) 
           a=x; 
        else 
           b=x;
        SHFT(v,w,x,u)
	SHFT(fv,fw,fx,fu)
     } 
     else 
     {
        if (u < x) 
           a=u; 
        else 
           b=u;
        if (fu <= fw || w == x) 
        {
	   v=w;
	   w=u;
	   fv=fw;
	   fw=fu;
        } 
        else if (fu <= fv || v == x || v == w) 
        {
	   v=u;
	   fv=fu;
        }
     }
  }
  nrerror("Too many iterations in brent");
  *xmin=x;
  return fx;
}
#undef ITMAX
#undef CGOLD
#undef ZEPS
#undef SHFT

#define EPS 1.0e-5
#define JMAX 20

double qtrap(double (*func)(double), double a, double b)
{
  double trapzd(double (*func)(double), double a, double b, int n);
  void nrerror(char error_text[]);
  int j;
  double s,olds=0.0;

  for (j=1;j<=JMAX;j++) 
  {
     s=trapzd(func,a,b,j);
     if (j > 5)
        if (fabs(s-olds) < EPS*fabs(olds) || (s == 0.0 && olds == 0.0)) 
           return s;
     olds=s;
  }
  nrerror("Too many steps in routine qtrap");
  return 0.0;
}
#undef EPS
#undef JMAX


#define FUNC(x) ((*func)(x))

double trapzd(double (*func)(double), double a, double b, int n)
{
  double x,tnm,sum,del;
  static double s;
  int it,j;

  if (n == 1) 
  {
     return (s=0.5*(b-a)*(FUNC(a)+FUNC(b)));
  } 
  else 
  {
     for (it=1,j=1;j<n-1;j++) 
        it <<= 1;
     tnm=it;
     del=(b-a)/tnm;
     x=a+0.5*del;
     for (sum=0.0,j=1;j<=it;j++,x+=del) 
        sum += FUNC(x);
     s=0.5*(s+(b-a)*sum/tnm);
     return s;
  }
}
#undef FUNC
#undef NRANSI
