/* arraytrace.c
 * Calls an_raytrace to implement the array raytracing
 *
 */

#include<stdlib.h>
#include<stdio.h>
#include<math.h>
#include "nrutil.h"
#include "raytrace.h"
#include "an_math.h"
#include "kerr.h"

#ifndef M_PI
#define M_PI	   3.14159265358979323846264338328      //pi
#endif

//vector dot product vec.vec
double dot(double *x, double *y)
{
  double prod;
  int i;
  prod = 0.0;
  for(i=0; i<4; i++)
  {
    prod += x[i]*y[i];
  }
  return prod;
}

//matrix/vector dot product mat.vec
void matdot(double **mat, double *vec, double *outvec)
{
  int i;
  for(i=0; i<4; i++)
  {
    outvec[i] = dot(mat[i], vec);
  }
  return;
}


//calculate impact parameters from other parameters
double ip_alpha(double ll, double q2, double a, double mu_o)
{
  double alpha;
  alpha = ll;
  alpha /= -sqrt(1.0 - mu_o*mu_o);

  return alpha;
}

double ip_beta(double ll, double q2, double a, double mu_o)
{
  double beta2;
  double cos2;
  cos2 = mu_o*mu_o;
  beta2 = q2 + a*a*cos2 - ll*ll*cos2/(1.0-cos2);
  return sqrt(beta2);
}

//calculate lambda from impact parameters
double llambda(double alpha, double beta, double a, double mu_o)
{
  double ll2;
  ll2 = -alpha;
  ll2*=sqrt(1.0-mu_o*mu_o);
  fprintf(stderr, "ll(%g, %g, %g, %g) = %g\n", alpha, beta, a, mu_o, ll2);
  return ll2;
}

//calculate carter constant from impact parameters
double qsquared(double alpha, double beta, double a, double mu_o)
{
  double ll;
  double cos2;
  double q2;
  ll = llambda(alpha, beta, a, mu_o);
  cos2 = mu_o*mu_o;
  q2 = beta*beta - a*a*cos2 + ll*ll*cos2/(1.0-cos2);
  fprintf(stderr, "q2(%g, %g, %g, %g) = %g\n", alpha, beta, a, mu_o, q2);

  return q2;
}

//kepler 4-velocity of the disk at position x
void Kepler4v(double a, double* x, double *u)
{
  double r;
  double temp;

  r = x[1]*sin(x[2]);

  temp = r*r-3.0*r + 2.0*a*sqrt(r);

  u[0] = (r*r + a*sqrt(r))/(r*sqrt(temp));
  u[1] = 0.0;
  u[2] = 0.0;
  u[3] = 1.0/sqrt(r*temp);
  
  return;
}


//normal vector to the disk at position x
void discNormal(double a, double *x, double *n)
{
  double r;
  r = x[1];
  n[0] = 0.0;
  n[1] = 0.0;
  n[2] = -1.0/r;
  n[3] = 0.0;
  return;
}


//Calculate the redshift factor at position x_em -> position xobs
double Calc_g(double a, double* x_em, double* k_em, double* u_em)
{
  double **g_em, **g_obs;
  double *k_obs;
  double *u_obs;

  double *tempvec1;
  double *tempvec2;

  double g;

  g_em = g_init();
  g_obs = g_init();
  g_down(x_em, a, g_em);

  g_obs[0][0] = -1.0;
  g_obs[0][1] = 0.0;
  g_obs[0][2] = 0.0;
  g_obs[0][3] = 0.0;

  g_obs[1][0] = 0.0;
  g_obs[1][1] = 1.0;
  g_obs[1][2] = 0.0;
  g_obs[1][3] = 0.0;

  g_obs[2][0] = 0.0;
  g_obs[2][1] = 0.0;
  g_obs[2][2] = 1.0;
  g_obs[2][3] = 0.0;

  g_obs[3][0] = 0.0;
  g_obs[3][1] = 0.0;
  g_obs[3][2] = 0.0;
  g_obs[3][3] = 1.0;

  k_obs = init_4v();
  u_obs = init_4v();
  
  tempvec1 = init_4v();
  tempvec2 = init_4v();

  k_obs[0] = 1.0; k_obs[1] = 1.0; k_obs[2] = 0.0, k_obs[3] = 0.0;
  u_obs[0] = 1.0; u_obs[1] = 0.0; u_obs[2] = 0.0, u_obs[3] = 0.0;
  
  matdot(g_obs, u_obs, tempvec1);
  matdot(g_em, u_em, tempvec2);
  
  g = dot(k_obs, tempvec1)/dot(k_em, tempvec2);

  free(k_obs);
  free(u_obs);
  free(tempvec1);
  free(tempvec2);
  g_free(g_obs);
  g_free(g_em);

  return g;
}

//Calculate the emission angle cosine at position x_em
double Calc_emcos(double a, double *x_em, 
		  double *k_em, double *u_em, double *norm)
{
  double **g_em;
  double *tempvec1;
  double *tempvec2;
  double emcos;

  g_em = g_init();
  g_down(x_em, a, g_em);

  tempvec1 = init_4v();
  tempvec2 = init_4v();

  matdot(g_em, norm, tempvec1);
  matdot(g_em, u_em, tempvec2);
  
  emcos = -dot(k_em, tempvec1)/dot(k_em, tempvec2);
  g_free(g_em);
  free(tempvec1);
  free(tempvec2);

  return emcos;
}

/*int Calc_rtr(double ipmax, int res, double a, double mu_o, 
 *  double ***x, double ***k, int **hit)
 *
 * Input: ipmax maxiumum impaxt parameter
 *        res resolution
 *        a black hole spin in units of M
 *        mu_o cosine of the inclination angle
 * 
 * Output: x[res][res] is an array of 4-vectors positions at emission
 *         k[res][res] is an array of 4-vectors wavevectors at emission
 *         hit[res][res] is an array of integers, indicating a disc 
 *                       interesection
 *   assume the output vectors are initialized before calling this
 */

int Calc_rtr(double ipmax, int res, double a, 
	     double mu_o, double ***x, double ***k, int **hit)
{
  double alpha, beta;
  int i, j, m;
  double ll, q2;

  for(i=0;i<res;i++)
  {
    alpha = -ipmax*(1.0 + 1.0/((double) res)) 
      + 2.0*ipmax*((double) i)/((double) res);
     for(j=0;j<res;j++)
     {
        beta = -ipmax*(1.0 + 1.0/((double) res)) 
	        + 2.0*ipmax*((double) j)/((double) res);
        ll = llambda(alpha, beta, a, mu_o);
        q2 = qsquared(alpha, beta, a, mu_o);

        hit[i][j] = raytrace(ll, q2, a, mu_o, alpha, beta, 0.0, x[i][j], k[i][j]);
     }
  }
  return 0;
}

/*
 * int Calc_ImageArrays(int res, double a, double mu_o, 
 *                        double ***x, double ***k, int **hit, 
 *                        double ***g, double ***emcos, double ***M)
 *
 * Inputs: res resolution of the alpha/beta arrays
 *          a  black hole spin parameter in units of M
 *         mu_0  cosine of the inclination angle
 *          x   2-d array of position 4-vectors at emission
 *          k   2-d array of wave-4vectrs at the emission
 *         hit  2-d array of integers indicating intersection with the disc
 * 
 * Ouputs: g  2-d array of doubles, for the redshift (res-2,res-2)
 *         emcos 2-d array of doubles, for the cosine of the emission angle
 *          M 2-d array of doubles for the magnification/lensing of each pixel
 */

int Calc_ImageArrays(int res, double a, double mu_o, double ***x, double ***k, int **hit, double **g, double **emcos, double **M)
{
  int Mres;
  int i;
  int j;
  int ii;
  double risco;
  double horizon;
  int del, gam, mu, nu;

  double *dxa, *dxb;

  double temp1;
  double temp2;

  int badspot;
  double *x_em;
  double *p_em;
  double **g_d;
  double *u_em;
  double *norm;
  

  double e[4][4][4][4];  //Levi-Civita psuedo-tensor

  //Resolution of magnification grid is 2 less than the rtr array
  
  Mres = res - 2;

  //calculate horizon and risco
  horizon = 1.0 + sqrt(1.0-a*a);
  temp1 = 1.0 + pow(1.0-a*a, 1.0/3.0)*
    (pow(1.0+a, 1.0/3.0) + pow(1.0-a, 1.0/3.0));
  temp2 = sqrt((3.0*a*a + temp1*temp1));
  
  risco = 3.0 + temp2 - sqrt((3.0-temp1)*(3.0+temp1 + 2.0*temp2));

  //initiate vectors
  dxa = init_4v();
  dxb = init_4v();
  x_em = init_4v();
  p_em = init_4v();
  u_em = init_4v();
  g_d = g_init();
  norm = init_4v();


  for(i=0; i<Mres; i++)
  {
     for(j=0; j<Mres; j++)
     {
        badspot = 0;
        for(ii=0; ii<4; ii++)
        {
	   //set dxa and dxb
	   dxa[ii] = (x[i+2][j+1][ii] - x[i][j+1][ii])/2.0;
	   dxb[ii] = (x[i+1][j+2][ii] - x[i+1][j][ii])/2.0;
         }
        //determine if the spot is valid
        if((hit[i+2][j+1] == 0) || (hit[i][j+1] == 0) || (hit[i+1][j+2] == 0) ||
	   (hit[i+1][j] == 0))
        {
           badspot = 1;
        }
        //phi is 2pi periodic, make sure we take the smallest value for delta phi
        if (fabs(temp1 = fmod(dxa[3], M_PI)) < M_PI/2.0)
        {
           dxa[3] = fmod(dxa[3], M_PI);
        }
        else 
        {
           dxa[3] = fmod(dxa[3], M_PI)-(temp1/fabs(temp1))*M_PI;
        }
        if (fabs(temp1 = fmod(dxb[3], M_PI)) < M_PI/2.0)
        {	
           dxb[3] = fmod(dxb[3], M_PI);
        }   
        else 
        {
           dxb[3] = fmod(dxb[3], M_PI)-(temp1/fabs(temp1))*M_PI;
        }

        for(ii=0; ii<4;ii++)
        {
           x_em[ii] = x[i+1][j+1][ii];
	   p_em[ii] = k[i+1][j+1][ii];
        }
      
        if((x_em[1] > risco)&&(hit[i+1][j+1]==1))
        {
           g_down(x_em, a, g_d);
	   Kepler4v(a, x_em, u_em);
	   if(badspot == 0)
           {
	      M[i][j] = 0.0;
	      temp1 = 0.0;
	      levi_civita(x_em, a, e);
	      for(del=0; del<4; del++)
	         for(gam=0; gam<4; gam++)
                    for(mu=0; mu<4; mu++)
		       for(nu=0; nu<4; nu++)
		          M[i][j] -= u_em[del]*p_em[gam]*dxa[mu]*dxb[nu]
		                     *e[del][gam][mu][nu];
	       for(del=0; del<4; del++)
	          for(gam=0; gam<4; gam++)
		      temp1 += u_em[del]*p_em[gam]*g_d[del][gam];
	       M[i][j] /= temp1;
   	   } 
           else 
           {
	      M[i][j] = 0.0;
           }
	   discNormal(a, x_em, norm);
	   emcos[i][j] = Calc_emcos(a, x_em, p_em, u_em, norm);
	   g[i][j] = Calc_g(a, x_em, p_em, u_em);
        }
        else
        {
           g[i][j] = 0.0;
	   M[i][j] = 0.0;
	   emcos[i][j] = 0.0;
        }
     }
  }
  

  //free memory
  free(dxa);
  free(dxb);
  free(x_em);
  free(p_em);
  free(u_em);
  g_free(g_d);
  free(norm);

  return 0;

}


