/*
 *  an_math.c Math routines for analytic raytrace
 * dtsang@physics.cornell.edu
 *
 */

#include<stdlib.h>
#include<stdio.h>
#include<math.h>
#include "nrutil.h"
#include "an_math.h"

#undef MAXSTP
#undef TINY

#ifndef M_PI

#define M_PI	   3.14159265358979323846264338328      // pi
#endif

#ifndef M_PI_2
#define M_PI_2	   1.57079632679489661923132169164      // pi/2
#endif

#ifndef COMPLEX_H

double c_abs(_Complex double z)
{	
  // returns |z|
  return hypot ( __real__ z, __imag__ z);
}

double c_abs2 (_Complex double z)
{
  // returns |z|^2
  double x = __real__ z;
  double y = __imag__ z;

  return (x * x + y * y);
}


_Complex double cinv (_Complex double a)
{
  // z = 1/a
  _Complex double z;

  double s = 1.0 / c_abs (a);
  z = __real__ a * s * s - __imag__ a * s * s * 1.0i;
  return z;
}

_Complex double csqrt (_Complex double a)
{
  // z=sqrt(a) 
  _Complex double z;

  if (__real__ a  == 0.0 && __imag__ a == 0.0)
    {
      z = 0.0 + 0.0i;
    }
  else
    {
      double x = fabs (__real__ a);
      double y = fabs (__imag__ a);
      double w;

      if (x >= y)
	{
	  double t = y / x;
	  w = sqrt (x) * sqrt (0.5 * (1.0 + sqrt (1.0 + t * t)));
	}
      else
	{
	  double t = x / y;
	  w = sqrt (y) * sqrt (0.5 * (t + sqrt (1.0 + t * t)));
	}

      if (__real__ a >= 0.0)
	{
	  double ai = __imag__ a;
	  z = w + 1.0i*ai/(2.0*w);
	}
      else
	{
	  double ai = __imag__ a;
	  double vi = (ai >= 0) ? w : -w;
	  z = ai/(2.0*vi) + vi*1.0i;
	}
    }

  return z;
}

_Complex double casin_real (double a)
{				
  // z = arcsin(a)
  _Complex double z;
  
  if (fabs (a) <= 1.0)
    {
      z = asin(a) + 0.0i;
    }
  else
    {
      if (a < 0.0)
	{
	  z = -M_PI_2 + acosh(-a)*1.0i;
	}
      else
	{
	  z = M_PI_2 - acosh(a)*1.0i;
	}
    }
  
  return z;
}

_Complex double casin (_Complex double a)
{				
  // z = arcsin(a) 
  double R = __real__ a, I = __imag__ a;
  _Complex double z;
  
  if (I == 0)
    {
      z = casin_real(R);
    }
  else
    {
      double x = fabs (R), y = fabs (I);
      double r = hypot (x + 1, y), s = hypot (x - 1, y);
      double A = 0.5 * (r + s);
      double B = x / A;
      double y2 = y * y;
      
      double real, imag;
      
      const double A_crossover = 1.5, B_crossover = 0.6417;
      
      if (B <= B_crossover)
	{
	  real = asin (B);
	}
      else
	{
	  if (x <= 1)
	    {
	      double D = 0.5 * (A + x) * (y2 / (r + x + 1) + (s + (1 - x)));
	      real = atan (x / sqrt (D));
	    }
	  else
	    {
	      double Apx = A + x;
	      double D = 0.5 * (Apx / (r + x + 1) + Apx / (s + (x - 1)));
	      real = atan (x / (y * sqrt (D)));
	    }
	}
      
      if (A <= A_crossover)
	{
	  double Am1;
	  
	  if (x < 1)
	    {
	      Am1 = 0.5 * (y2 / (r + (x + 1)) + y2 / (s + (1 - x)));
	    }
	  else
	    {
	      Am1 = 0.5 * (y2 / (r + (x + 1)) + (s + (x - 1)));
	    }
	  
	  imag = log1p (Am1 + sqrt (Am1 * (A + 1)));
	}
      else
	{
	  imag = log (A + sqrt (A * A - 1));
	}
      
      z = ((R >= 0) ? real : -real) + ((I >= 0) ? imag : -imag)*1.0i;
    }
  
  return z;
}


_Complex double csin (_Complex double a)
{				

  // z = sin(a) 
  double R = __real__ a, I = __imag__ a;
  
  _Complex double z;
  
  if (I == 0.0) 
    {
      z = sin(R) + 0.0;
    } 
  else 
    {
      z = sin(R)*cosh(I) + cos(R)*sinh(I)*1.0i;
    }

  return z;
}


#endif
#define SMALL 1.0e-10
#define ERRTOL 0.08
#define TINY 1.5e-38
#define BIG 3.0e37
#define THIRD (1.0/3.0)
#define C1 (1.0/24.0)
#define C2 0.1
#define C3 (3.0/44.0)
#define C4 (1.0/14.0)

double rf(double x, double y, double z)
{
  double alamb,ave,delx,dely,delz,e2,e3,sqrtx,sqrty,sqrtz,xt,yt,zt;
  if (fabs(x) < SMALL) x = 0.0; 
  if (fabs(y) < SMALL) y = 0.0; 
  if (fabs(z) < SMALL) z = 0.0; 

  if (FMIN(FMIN(x,y),z) < 0.0 || FMIN(FMIN(x+y,x+z),y+z) < TINY ||
                                           FMAX(FMAX(x,y),z) > BIG)
    {
      fprintf(stderr, "x = %g, y = %g, z = %g\n", x, y, z);
      nrerror("invalid arguments in rf");
    }
  xt=x;
  yt=y;
  zt=z;
  do {
      sqrtx=sqrt(xt);
      sqrty=sqrt(yt);
      sqrtz=sqrt(zt);
      alamb=sqrtx*(sqrty+sqrtz)+sqrty*sqrtz;
      xt=0.25*(xt+alamb);
      yt=0.25*(yt+alamb);
      zt=0.25*(zt+alamb);
      ave=THIRD*(xt+yt+zt);
      delx=(ave-xt)/ave;
      dely=(ave-yt)/ave;
      delz=(ave-zt)/ave;
      } while (FMAX(FMAX(fabs(delx),fabs(dely)),fabs(delz)) > ERRTOL);
	
  e2=delx*dely-delz*delz;
  e3=delx*dely*delz;
  return (1.0+(C1*e2-C2-C3*e3)*e2+C4*e3)/sqrt(ave);
}


_Complex double crf(_Complex double x, _Complex double y, _Complex double z)
{
  _Complex double alamb,ave,delx,dely,delz,e2,e3,sqrtx,sqrty,sqrtz,xt,yt,zt;
  double mx, my, mz;
  mx = c_abs(x);
  my = c_abs(y);
  mz = c_abs(z);

  xt=x;
  yt=y;
  zt=z;
  do{
      sqrtx=csqrt(xt);
      sqrty=csqrt(yt);
      sqrtz=csqrt(zt);
      alamb=sqrtx*(sqrty+sqrtz)+sqrty*sqrtz;
      xt=0.25*(xt+alamb);
      yt=0.25*(yt+alamb);
      zt=0.25*(zt+alamb);
      ave=THIRD*(xt+yt+zt);
      delx=(ave-xt)/ave;
      dely=(ave-yt)/ave;
      delz=(ave-zt)/ave;
    } while (FMAX(FMAX(c_abs(delx),c_abs(dely)),c_abs(delz)) > ERRTOL);

  e2=delx*dely-delz*delz;
  e3=delx*dely*delz;
  _Complex double temp = (1.0+(C1*e2-C2-C3*e3)*e2+C4*e3)/csqrt(ave);
 
  return temp;
}

#undef ERRTOL
#undef TINY
#undef BIG
#undef THIRD
#undef C1
#undef C2
#undef C3
#undef C4

#define ERRTOL 0.05
#define TINY 1.0e-25
#define BIG 4.5e21
#define C1 (3.0/14.0)
#define C2 (1.0/6.0)
#define C3 (9.0/22.0)
#define C4 (3.0/26.0)
#define C5 (0.25*C3)
#define C6 (1.5*C4)


double rd(double x, double y, double z)
{
  double alamb,ave,delx,dely,delz,ea,eb,ec,ed,ee,fac,sqrtx,sqrty,
 	 sqrtz,sum,xt,yt,zt;

  if (FMIN(x,y) < 0.0 || FMIN(x+y,z) < TINY || FMAX(FMAX(x,y),z) > BIG)
    {		
      nrerror("invalid arguments in rd");
    }

  xt=x;
  yt=y;
  zt=z;
  sum=0.0;
  fac=1.0;
  do{
      sqrtx=sqrt(xt);
      sqrty=sqrt(yt);
      sqrtz=sqrt(zt);
      alamb=sqrtx*(sqrty+sqrtz)+sqrty*sqrtz;
      sum += fac/(sqrtz*(zt+alamb));
      fac=0.25*fac;
      xt=0.25*(xt+alamb);
      yt=0.25*(yt+alamb);
      zt=0.25*(zt+alamb);
      ave=0.2*(xt+yt+3.0*zt);
      delx=(ave-xt)/ave;
      dely=(ave-yt)/ave;
      delz=(ave-zt)/ave;
    } while (FMAX(FMAX(fabs(delx),fabs(dely)),fabs(delz)) > ERRTOL);
	
  ea=delx*dely;
  eb=delz*delz;
  ec=ea-eb;
  ed=ea-6.0*eb;
  ee=ed+ec+ec;

  return 3.0*sum+fac*(1.0+ed*(-C1+C5*ed-C6*delz*ee)
		+delz*(C2*ee+delz*(-C3*ec+delz*C4*ea)))/(ave*sqrt(ave));
}

_Complex double crd(_Complex double x, _Complex double y, _Complex double z)
{
  _Complex double alamb,ave,delx,dely,delz,ea,eb,ec,ed,ee,fac,sqrtx,
	          sqrty,sqrtz,sum,xt,yt,zt;

  double mx, my, mz;
  mx = c_abs(x);
  my = c_abs(y);
  mz = c_abs(z);
  if (FMIN(FMIN(mx+my,mx+mz),my+mz) < TINY || FMAX(FMAX(mx,my),mz) > BIG)
    {
      nrerror("invalid arguments in crd");
    }
  if((__imag__ x < TINY && __real__ x < 0.0)||
     (__imag__ y < TINY && __real__ y < 0.0)||
     (__imag__ z < TINY && __real__ z < 0.0))
    {
      nrerror("invalid arguments in crd");
    }
	
  xt=x;
  yt=y;
  zt=z;
  sum=0.0;
  fac=1.0;
  do{
      sqrtx=csqrt(xt);
      sqrty=csqrt(yt);
      sqrtz=csqrt(zt);
      alamb=sqrtx*(sqrty+sqrtz)+sqrty*sqrtz;
      sum += fac/(sqrtz*(zt+alamb));
      fac=0.25*fac;
      xt=0.25*(xt+alamb);
      yt=0.25*(yt+alamb);
      zt=0.25*(zt+alamb);
      ave=0.2*(xt+yt+3.0*zt);
      delx=(ave-xt)/ave;
      dely=(ave-yt)/ave;
      delz=(ave-zt)/ave;
    } while (FMAX(FMAX(fabs(delx),fabs(dely)),fabs(delz)) > ERRTOL);
	
  ea=delx*dely;
  eb=delz*delz;
  ec=ea-eb;
  ed=ea-6.0*eb;
  ee=ed+ec+ec;
  return 3.0*sum+fac*(1.0+ed*(-C1+C5*ed-C6*delz*ee)
	 +delz*(C2*ee+delz*(-C3*ec+delz*C4*ea)))/(ave*sqrt(ave));
}

#undef ERRTOL
#undef TINY
#undef BIG
#undef C1
#undef C2
#undef C3
#undef C4
#undef C5
#undef C6

#define ERRTOL 0.04
#define TINY 1.69e-38
#define SQRTNY 1.3e-19
#define BIG 3.e37
#define TNBG (TINY*BIG)
#define COMP1 (2.236/SQRTNY)
#define COMP2 (TNBG*TNBG/25.0)
#define THIRD (1.0/3.0)
#define C1 0.3
#define C2 (1.0/7.0)
#define C3 0.375
#define C4 (9.0/22.0)

double rc(double x, double y)
{
  double alamb,ave,s,w,xt,yt;
  if (x < 0.0 || y == 0.0 || (x+fabs(y)) < TINY || (x+fabs(y)) > BIG ||
		                    (y<-COMP1 && x > 0.0 && x < COMP2))
    {
      nrerror("invalid arguments in rc");
    }
  if (y > 0.0) 
    {
      xt=x;
      yt=y;
      w=1.0;
    } 
  else 
    {
      xt=x-y;
      yt = -y;
      w=sqrt(x)/sqrt(xt);
    }
  do {
      alamb=2.0*sqrt(xt)*sqrt(yt)+yt;
      xt=0.25*(xt+alamb);
      yt=0.25*(yt+alamb);
      ave=THIRD*(xt+yt+yt);
      s=(yt-ave)/ave;
    } while (fabs(s) > ERRTOL);
	
  return w*(1.0+s*s*(C1+s*(C2+s*(C3+s*C4))))/sqrt(ave);
}


_Complex double crc(_Complex double x, _Complex double y)
{
  _Complex double alamb,ave,s,w,xt,yt;
  double mx, my;
  mx = c_abs(x);
  my = c_abs(y);

  //Cauchy principle value if y is real and negative
  if((__imag__ y  < TINY) && (__real__ y < 0.0) && (__real__ (x-y) > 0.0))
    {
      if(c_abs(x) <= TINY) 
        {
          return 0.0 +0.0i;
        }
      
      return csqrt(x/(x-y))*crc(x-y, -y);
    }
  if ((mx+my) < TINY || (mx+my) > BIG ||
	    ((fabs(__imag__ x) < TINY)&& ((__real__ x) < 0.0)) ||
	    ((fabs(__imag__ y) < TINY)&& ((__real__ y) < TINY)))
    {
      nrerror("invalid arguments in crc");
    }

  xt=x;
  yt=y;
  w=1.0;
  do {
      alamb=2.0*csqrt(xt)*csqrt(yt)+yt;
      xt=0.25*(xt+alamb);
      yt=0.25*(yt+alamb);
      ave=THIRD*(xt+yt+yt);
      s=(yt-ave)/ave;
    } while (c_abs(s) > ERRTOL);
  return w*(1.0+s*s*(C1+s*(C2+s*(C3+s*C4))))/csqrt(ave);
}
#undef ERRTOL
#undef TINY
#undef SQRTNY
#undef BIG
#undef TNBG
#undef COMP1
#undef COMP2
#undef THIRD
#undef C1
#undef C2
#undef C3
#undef C4

#define ERRTOL 0.05
#define TINY 2.5e-13
#define BIG 9.0e11
#define C1 (3.0/14.0)
#define C2 (1.0/3.0)
#define C3 (3.0/22.0)
#define C4 (3.0/26.0)
#define C5 (0.75*C3)
#define C6 (1.5*C4)
#define C7 (0.5*C2)
#define C8 (C3+C3)

double rj(double x, double y, double z, double p)
{
  double rc(double x, double y);
  double rf(double x, double y, double z);
  double a,alamb,alpha,ans,ave,b,beta,delp,delx,dely,delz,ea,eb,ec,
  	 ed,ee,fac,pt,rcx,rho,sqrtx,sqrty,sqrtz,sum,tau,xt,yt,zt;

  if (FMIN(FMIN(x,y),z) < 0.0 || FMIN(FMIN(FMIN(x+y,x+z),y+z),fabs(p)) < TINY
		|| FMAX(FMAX(FMAX(x,y),z),fabs(p)) > BIG)
    {
      nrerror("invalid arguments in rj");
    }  
  sum=0.0;
  fac=1.0;
  if (p > 0.0) 
    {
      xt=x;
      yt=y;
      zt=z;
      pt=p;
    } 
  else 
    {
       xt=FMIN(FMIN(x,y),z);
       zt=FMAX(FMAX(x,y),z);
       yt=x+y+z-xt-zt;
       a=1.0/(yt-p);
       b=a*(zt-yt)*(yt-xt);
       pt=yt+b;
       rho=xt*zt/yt;
       tau=p*pt/yt;
       rcx=rc(rho,tau);
    }
  do {
      sqrtx=sqrt(xt);
      sqrty=sqrt(yt);
      sqrtz=sqrt(zt);
      alamb=sqrtx*(sqrty+sqrtz)+sqrty*sqrtz;
      alpha=SQR(pt*(sqrtx+sqrty+sqrtz)+sqrtx*sqrty*sqrtz);
      beta=pt*SQR(pt+alamb);
      sum += fac*rc(alpha,beta);
      fac=0.25*fac;
      xt=0.25*(xt+alamb);
      yt=0.25*(yt+alamb);
      zt=0.25*(zt+alamb);
      pt=0.25*(pt+alamb);
      ave=0.2*(xt+yt+zt+pt+pt);
      delx=(ave-xt)/ave;
      dely=(ave-yt)/ave;
      delz=(ave-zt)/ave;
      delp=(ave-pt)/ave;
    } while (FMAX(FMAX(FMAX(fabs(delx),fabs(dely)),
		fabs(delz)),fabs(delp)) > ERRTOL);

  ea=delx*(dely+delz)+dely*delz;
  eb=delx*dely*delz;
  ec=delp*delp;
  ed=ea-3.0*ec;
  ee=eb+2.0*delp*(ea-ec);
  ans=3.0*sum+fac*(1.0+ed*(-C1+C5*ed-C6*ee)+eb*(C7+delp*(-C8+delp*C4))
      +delp*ea*(C2-delp*C3)-C2*delp*ec)/(ave*sqrt(ave));
	
  if (p <= 0.0) 
    {
       ans=a*(b*ans+3.0*(rcx-rf(xt,yt,zt)));
    }
  return ans;
}

_Complex double crj(_Complex double x, _Complex double y, _Complex double z, _Complex double p)
{
  _Complex double crc(_Complex double x, _Complex double y);
  _Complex double crf(_Complex double x, _Complex double y, _Complex double z);
  _Complex double a,alamb,alpha,ans,ave,b,beta,dm,em,delta;
  _Complex double delp,delx,dely,delz,ea,eb,ec;
  _Complex double ed,ee,fac,pt,rcx,rho,sqrtx,sqrty,sqrtz,sqrtp;
  _Complex double sum,tau,xt,yt,zt;
  double mx, my, mz, mp, rx, ry, rz, rp, ix, iy, iz, ip;

  mx = c_abs(x);
  my = c_abs(y);
  mz = c_abs(z);
  mp = c_abs(p);
  rx = __real__ x;
  ry = __real__ y;
  rz = __real__ z;
  rp = __real__ p;
  ix = __imag__ x;
  iy = __imag__ y;
  iz = __imag__ z;
  ip = __imag__ p;

  if (FMIN(FMIN(FMIN(rx,ry),rz), rp) < 0.0
	    || FMIN(FMIN(FMIN(mx+my,mx+mz),my+mz),mp) < TINY
	    || FMAX(FMAX(FMAX(mx,my),mz),mp) > BIG)
    {
      if((fabs(ip) < TINY && rp < 0.0)
	     ||FMAX(FMAX(fabs(ix), fabs(iy)), fabs(iz)) > TINY 
	     || FMIN(FMIN(rx, ry), rz) < 0.0)
        {
	  if((fabs(ip) < TINY && rp < 0.0)
	        ||((c_abs(CONJ(x)-y) > TINY || fabs(iz) > TINY || rz < 0.0) 
		//xy not complex conjugates or z not real or z negative
		&& (c_abs(CONJ(x)-z) > TINY || fabs(iy) > TINY || ry < 0.0) 
		//yz not complex conjugates or x not real or y negative
		&& (c_abs(CONJ(y)-z) > TINY || fabs(ix) > TINY || rx < 0.0) 
		//xz not complex conjugates or y not real or x negative
		))
            {
	      nrerror("invalid arguments in crj");
            }
        }
    }
	

  sum=0.0;
  fac=1.0;
  xt=x;
  yt=y;
  zt=z; 
  pt=p;
  delta = (p-x)*(p-y)*(p-z);
  do {
      sqrtx=csqrt(xt);
      sqrty=csqrt(yt);
      sqrtz=csqrt(zt);
      sqrtp=csqrt(pt);
		
      alamb=sqrtx*(sqrty+sqrtz)+sqrty*sqrtz;

      dm = (sqrtp + sqrtx)*(sqrtp + sqrty)*(sqrtp + sqrtz);
      em = fac*fac*fac*delta/(dm*dm);

      sum += fac*crc(1.0, 1.0+em)/dm;
      fac=0.25*fac;
      xt=0.25*(xt+alamb);
      yt=0.25*(yt+alamb);
      zt=0.25*(zt+alamb);
      pt=0.25*(pt+alamb);
      ave=0.2*(xt+yt+zt+pt+pt);
      delx=(ave-xt)/ave;
      dely=(ave-yt)/ave;
      delz=(ave-zt)/ave;
      delp=(ave-pt)/ave;
    } while (FMAX(FMAX(FMAX(c_abs(delx),c_abs(dely)),
			   c_abs(delz)),c_abs(delp)) > ERRTOL);
	
  ea=delx*(dely+delz)+dely*delz;
  eb=delx*dely*delz;
  ec=delp*delp;
  ed=ea-3.0*ec;
  ee=eb+2.0*delp*(ea-ec);

  ans=6.0*sum+fac*(1.0+ed*(-C1+C5*ed-C6*ee)+eb*(C7+delp*(-C8+delp*C4))
 		   +delp*ea*(C2-delp*C3)-C2*delp*ec)/(ave*csqrt(ave));
  return ans;
}

#undef ERRTOL
#undef TINY
#undef BIG
#undef C1
#undef C2
#undef C3
#undef C4
#undef C5
#undef C6
#undef C7
#undef C8


double ellf(double phi, double ak)
{
  double rf(double x, double y, double z);
  double s;
  
  s=sin(phi);
  return s*rf(SQR(cos(phi)),(1.0-s*ak)*(1.0+s*ak),1.0);
}

double elle(double phi, double ak)
{
  double rd(double x, double y, double z);
  double rf(double x, double y, double z);
  double cc,q,s;
  
  s=sin(phi);
  cc=SQR(cos(phi));
  q=(1.0-s*ak)*(1.0+s*ak);
  return s*(rf(cc,q,1.0)-(SQR(s*ak))*rd(cc,q,1.0)/3.0);
}

double ellpi(double phi, double en, double ak)
{
  double rf(double x, double y, double z);
  double rj(double x, double y, double z, double p);
  double cc,enss,q,s;
  
  s=sin(phi);
  enss=en*s*s;
  cc=SQR(cos(phi));
  q=(1.0-s*ak)*(1.0+s*ak);
  return s*(rf(cc,q,1.0)-enss*rj(cc,q,1.0,1.0+enss)/3.0);
}

double cuberoot(double x)
{
  double arg, out;
  if (x < 0) arg = -x;
  else arg = x;
  out = SIGN(pow(arg, 1.0/3.0), x);
  return out;
  
}

#define CA 0.0003

void sncndn(double uu, double emmc, double *sn, double *cn, double *dn)
{
  double a,b,c,d,emc,u;
  double em[14],en[14];
  int i,ii,l,bo;
  
  emc=emmc;
  u=uu;
  if (emc) 
    {
      bo=(emc < 0.0);
      if (bo) 
        {
          d=1.0-emc;
          emc /= -1.0/d;
          u *= (d=sqrt(d));
         }
      a=1.0;
      *dn=1.0;
      for (i=1;i<=13;i++) 
        {
          l=i;
          em[i]=a;
          en[i]=(emc=sqrt(emc));
          c=0.5*(a+emc);
          if (fabs(a-emc) <= CA*a) break;
          emc *= a;
          a=c;
        }
      u *= c;
      *sn=sin(u);
      *cn=cos(u);
      if (*sn) 
        {
          a=(*cn)/(*sn);
          c *= a;
          for (ii=l;ii>=1;ii--) 
            {
	      b=em[ii];
	      a *= c;
	      c *= (*dn);
	      *dn=(en[ii]+a)/(b+a);
	      a=c/b;
            }
          a=1.0/sqrt(c*c+1.0);
          *sn=(*sn >= 0.0 ? a : -a);
          *cn=c*(*sn);
        }
      if (bo) 
        {
          a=(*dn);
          *dn=(*cn);
          *cn=a;
          *sn /= d;
        }
    } 
  else 
    {
      *cn=1.0/cosh(u);
      *dn=(*cn);
      *sn=tanh(u);
    }
}
#undef CA



void sncndnhelper(double u, double m, double *sn, double *cn, double *dn)
{
  double sr, cr, dr;
  if( m > 2)
    {
      sncndn(u*sqrt(m), 1.0/m, &sr, &cr, &dr);
      *sn = sr/(dr*sqrt(m));
      *cn = cr/dr;
      *dn = 1.0/dr;
      return;
    } 
  else if(m < 0)
    {
      sncndn(u*sqrt(1.0-m), -m/(1.0-m), &sr, &cr, &dr);
      *sn = sr/sqrt(1.0-m);
      *cn = dr;
      *dn = cr;
      return;
    } 
  else
    {
      sncndn(u, m, &sr, &cr, &dr);
      *sn = sr;
      *cn = cr;
      *dn = dr;
      return;
    }
}

/*
 * csncndn(zz, emmc, *sn, *cn, *dn)
 * _Complex routine for calculating sn cn and dn the Jacobi Elliptic functions
 * using Landen's transformations. Complementary Modulus emmc = 1-k^2 is 
 * assumed to be either purely real or have m = 1-emmc have norm of 1.
 *
 */
void csncndn(_Complex double zz, _Complex double emmc, _Complex double *sn,
             _Complex double *cn, _Complex double *dn)
{
  double u;
  double v;
  _Complex double k;
  double kappa;
  double si, ci, di, sr, cr, dr;
  _Complex double zprime;
  _Complex double temp;
  _Complex double sz, cz, dz;
  _Complex double emmcprime;
  double m;
  double denom;

  // Complex modulus
  if (__imag__ emmc  != 0.0)
    { 
      k = csqrt(1.0 - emmc);
      kappa = -SIGN(1.0, __imag__ k )
      *sqrt((1.0 - __real__ k)/(1.0 + __real__ k));
      temp = 1.0 + kappa*1.0i;
      zprime = zz/temp;
      emmcprime = -kappa*kappa + 0.0i;
      csncndn(zprime, emmcprime, &sz, &cz, &dz);
      *sn = temp*sz*cz/dz;
      *cn = ((1.0+0.0i)/dz) - temp*sz*sz/dz;
      temp = -1.0 + kappa*1.0i;
      *dn = ((1.0+0.0i)/dz) + temp*sz*sz/dz;
      return;
    }
  else
    {
      u = __real__ zz;
      v = __imag__ zz;
      m = __real__ emmc;
      if (v != 0.0)
        {
          sncndnhelper(u, m, &sr, &cr, &dr);
          sncndnhelper(v, 1.0 - m, &si, &ci, &di);
          denom = (1.0 + si*dr)*(1.0 - si*dr);
          *sn = sr*di/denom + cr*dr*si*ci*1.0i/denom;
          *cn = cr*ci/denom - sr*dr*si*di*1.0i/denom;
          *dn = dr*ci*di/denom - (1.0 - m)*sr*cr*si*1.0i/denom;
          return;
        }
      else
        {
          sncndnhelper(u, m, &sr, &cr, &dr);
          *sn = sr + 0.0i;
          *cn = cr + 0.0i;
          *dn = dr + 0.0i;
          return;
        }
    }
}

/*
 * Using Landen's transformation
 * Assuming that the norm of k is 1 or that k is pure real or pure imaginary
 * m is gauranteed to have  magnitude = 1, or be real, and 
 * z is gauranteed to have magnitude <= 1.
 */

#define TINY 1.5e-15

/*
 * Inverse Jacobi sn with complex argument and complex modulus. 
 * Is only garaunteed to be valid for |z| < 1. working on an extension
 */

_Complex double InverseSN(_Complex double z, _Complex double m)
{
  double kappa;
  double k;
  _Complex double ak;
  _Complex double theta;

  ak = csqrt(m);

  // ak real, pass it on
  if (fabs(__imag__ ak) <= TINY)
    {
      fprintf(stderr, "theta = %g + i%g\n", __real__ casin(z), __imag__ casin(z)); 
      return cellf(casin(z), __real__ ak);
    }
  // ak imaginary, use imag. transform
  if (fabs(__real__ ak) <= TINY)
    { 
      //theta = sin(beta) in this transform
      k = __imag__ ak;  
      theta = sqrt(1.0 + k*k)*z/csqrt(1.0 + k*k*z*z);
      kappa = 1.0/sqrt(1 + k*k);
      fprintf(stderr, "theta = %g + i%g, kappa = %g, casin(theta) = %g + i%g\n",
              __real__ theta, __imag__ theta, kappa, __real__ casin(theta),
              __imag__ casin(theta));
      return kappa*cellf(casin(theta), k*kappa);
    }
  
  theta = 0.5*(casin(ak*z) + casin(z));
  fprintf(stderr, "theta = %g + i%g, ak*z angle= %g\n", __real__ theta, 
          __imag__ theta, atan((__imag__ (ak*z))/(__real__ (ak*z))));
  kappa = -SIGN(1.0, __imag__ ak)*sqrt((1- __real__ ak)/(1 + __real__ ak));
  k = sqrt(1 + kappa*kappa);
  
  return (1.0 + kappa*1.0i)*cellf(theta, k);

}

//takes sinphi instead of phi as an argument
_Complex double cellf(_Complex double sinphi, _Complex double m)
{
  _Complex double temp = sinphi*crf(1.0-sinphi*sinphi, 1.0-m*sinphi*sinphi, 1.0);
   return temp;

}

_Complex double cellk(_Complex double m)
{
  fprintf(stderr, "m = %g + i%g\n", __real__ m, __imag__ m);
  return crf(0.0, 1.0-m, 1.0);
}


//takes sinphi instead of phi as an argument
_Complex double celle(_Complex double sinphi, _Complex double m)
{
  return cellf(sinphi, m) -
         (m/3.0)*sinphi*sinphi*sinphi*crd(1.0-sinphi*sinphi, 
					 1.0-m*sinphi*sinphi,1.0);
}


//takes sinphi instead of phi as an argument
_Complex double cellpi(_Complex double sinphi, _Complex double k, 
		       _Complex double n)
{
  if((__real__ (sinphi*sinphi) > 1.0) || (__real__ (k*k*sinphi*sinphi) > 1.0)||
     (__real__ (n*sinphi*sinphi) < -1.0))
    {
      fprintf(stderr, "Bad values for cellpi\n");
      return 0.0;
    }

  return sinphi*crf((1.0+sinphi)*(1.0-sinphi), 
         (1.0 + k*sinphi)*(1.0-k*sinphi), 1.0) - 
         n*(1.0/3.0)*sinphi*sinphi*sinphi*crj((1.0+sinphi)*(1.0-sinphi), 
         (1.0+k*sinphi)*(1.0-k*sinphi), 1.0, 1.0 + n*sinphi*sinphi);

}
