/*
 * kerr   - dtsang@physics.cornell.edu                 14/06/04
 * Routines to calculate useful quantities in the kerr metric.
 * Assuming Boyer-Linquist coords
 *
 * To add: Christoffel symbols, Riemann tensor, covariant deriv..
 *         not necessary yet.
 */

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include "kerr.h"
#include "raytrace.h"

double kerr_Sigma(double r, double mu, double a)
{
  double Sigma;

  Sigma  = r*r;
  Sigma += a*a*mu*mu;

  return Sigma;
}

double kerr_Delta(double r, double mu, double a)
{
  double Delta;

  Delta  = r*r;
  Delta += a*a;
  Delta -= 2.0*r;

  return Delta;
}

double kerr_A(double r, double mu, double a)
{
  double x, A;
  double mu2;
  double Sigma;
  
  Sigma = kerr_Sigma(r, mu, a);
  mu2 = mu*mu;
  x  = r*r + a*a;
    A  = x*x;
  A -= a*a*(1-mu*mu)*kerr_Delta(r,mu,a);
  return A;
}

double **g_init()
{
  int i,j;
  double **g;

  g = malloc(sizeof(double *)*4);
  for (i = 0; i < 4; i++)
  {
     g[i] = malloc(sizeof(double)*4);
     for(j=0; j < 4; j++)
     {
        g[i][j]=0.0;
     }
  }
  return g;
}

void g_free(double** g)
{
  int i;
  for(i=0; i < 4; i++) 
     free(g[i]);
  free(g);
  return;
}

void g_down(double *x, double a, double **g)
{
  double r, mu, mu2, Sigma, Delta, A; 

  r = x[1];
  mu = cos(x[2]);
  mu2 = mu*mu;
  Sigma = kerr_Sigma(r, mu, a);
  Delta = kerr_Delta(r, mu, a);
  A = kerr_A(r, mu, a);
 
  g[0][0] =  -(1.0 - 2.0*r/Sigma);
  g[1][1] = Sigma/Delta;
  g[2][2] = Sigma;
  g[3][3] = A*(1.0 - mu2)/Sigma;
  g[0][3] = -2.0*a*r*(1.0-mu2)/Sigma;
  g[3][0] = g[0][3];

  return;

}

void g_up(double*x, double a, double **g)
{
  double r, mu, mu2, Sigma, Delta, A; 

  r = x[1];
  mu = cos(x[2]);
  mu2 = mu*mu;
  Sigma = kerr_Sigma(r, mu, a);
  Delta = kerr_Delta(r, mu, a);
  A = kerr_A(r, mu, a);

  g[0][0] = -A/(Sigma*Delta);
  g[1][1] = Delta/Sigma;
  g[2][2] = 1.0/Sigma;
  g[3][3] = Sigma/(A*(1.0-mu2));
  g[3][3] -= (4.0*r*r*a*a)/(A*Sigma*Delta);
  g[0][3] = -2.0*r*a/(Sigma*Delta);
  g[3][0] = g[0][3];

  return;
}

double *init_4v()
{
  int i;
  double *x;
  x = malloc(sizeof(double)*4);
  for(i=0;i<4;i++)
     x[i]=0;
  return x;
}

double kerr_ds(double *dx, double *x, double a)
{
  double **g;
  double ds;
  int i, j;
  g = g_init();
  g_down(x, a, g);
  for(i=0; i < 4; i++)
  {
     for(j=0; j < 4; j++)
     {
        ds = g[i][j]*dx[i]*dx[j];
     }
  }
  return ds;
}

double Calc_risco(double a){

  double temp1 = 1.0 + pow(1.0 - a*a, 1.0/3.0)*(pow(1.0+a, 1.0/3.0)
                 + pow(1.0-a, 1.0/3.0));
  double temp2 = sqrt(3.0*a*a + temp1*temp1);
   
  return(3.0 + temp2 - sqrt((3.0-temp1)*(3.0+temp1 + 2.0*temp2)));
}

double Determinant(double **a,int n){
  int i,j,j1,j2;
  double det = 0;
  double **m = NULL;

  if (n < 1) 
  { 
     fprintf(stderr, "Bad matrix size\n");
  } 
  else if (n == 1) 
  {
     det = a[0][0];
  } 
  else if (n == 2) 
  {
     det = a[0][0] * a[1][1] - a[1][0] * a[0][1];
  } 
  else 
  {
     det = 0;
     for (j1=0;j1<n;j1++) 
     {
        m = malloc((n-1)*sizeof(double *));
        for (i=0;i<n-1;i++)
	   m[i] = malloc((n-1)*sizeof(double));
        for (i=1;i<n;i++) 
        {
	   j2 = 0;
	   for (j=0;j<n;j++) 
           {		
	      if (j == j1)
	         continue;
	     m[i-1][j2] = a[i][j];
	     j2++;
	   }
        }
        det += pow(-1.0,j1+2.0) * a[0][j1] * Determinant(m,n-1);
        for (i=0;i<n-1;i++)
	   free(m[i]);
        free(m);
     }
  }
  return(det);
}

double lchelper(int a, int b, int c, int d)
{
  int A[4];
  double parity;
  int i, j;
  int temp;

  parity = 1.0;
  A[0] = a;
  A[1] = b;
  A[2] = c;
  A[3] = d;

  for(i = 0; i < 3; i++)
  {
     for(j = 0; j < 3-i; j++)
     {
        if (A[j] > A[j+1])
        {
	   temp =  A[j];
	   A[j] = A[j+1];
	   A[j+1] = temp;
	   parity *= -1.0;
        }
        else if(A[j] == A[j+1])
        {
	   return 0.0;
         }
      }
  }
  return parity;
}

//Assumed that e is initialized
void levi_civita(double *x, double a, double e[4][4][4][4]){
  double** gmat=NULL;
  double g;
  int i, j, k, l;

  gmat = g_init();
  g_down(x, a, gmat); 
  g = Determinant(gmat, 4);

  if (g > 0.0)
  {
     fprintf(stderr, "Error: Det(g_mu_nu) > 0, e uninitiated\n");
     return;
  }
  for(i = 0; i < 4; i++)
     for(j = 0; j < 4; j++)
        for(k = 0; k < 4; k++)
	   for(l = 0; l < 4; l++)
	      e[i][j][k][l] = sqrt(-g)*lchelper(i, j, k, l);
  
  g_free(gmat);
  return;
}
