#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <complex.h>
#include "nrmath.h"
#include <string.h>

#define OUTFACTOR 4.0
#define PI 3.141592653589793238462643


double m;
double cs2;
double omega_r;
double omega_i;
double p;
int offset;
double sign;
int setlog;
double Omegaunit;

int nvar;
double x1,x2;

int kmax,kount;
double *xp,**yp,dxsav;

double eps;

double *f;

double rolfunc(double r)
{

  double kappa;
  double Omega;

  Omega = 1.0/(r-2.0)/sqrt(r);
  kappa = Omega*sqrt((r-6.0)/(r-2.0));

  return (kappa - omega_r + m*Omega);
}

double rilfunc(double r)
{

  double kappa;
  double Omega;

  Omega = 1.0/(r-2.0)/sqrt(r);
  kappa = Omega*sqrt((r-6.0)/(r-2.0));

  return (kappa + omega_r - m*Omega);
}
double rcfunc(double r)
{

  double kappa;
  double Omega;

  Omega = 1.0/(r-2.0)/sqrt(r);
  kappa = Omega*sqrt((r-6.0)/(r-2.0));

  return (omega_r - m*Omega);
}

void derivscomplex( double x, double y[], double dydx[])
{
  double tomega_r;
  double tomega_i;

  _Complex double tomega;
  _Complex double tomega2;

  double kappa;

  double kappa2;
  double Omega;

  _Complex double dhdr;
  _Complex double diudr;

  Omega = 1.0/(x-2.0)/sqrt(x);
  kappa = Omega*sqrt((x-6.0)/(x-2.0));

  tomega_r = y[5] - m*Omega;
  tomega_i = y[6];
  tomega = tomega_r + 1.0i*tomega_i;
  tomega2 = tomega*tomega;

  kappa2 = kappa*kappa;
  
  dhdr = 2.0*Omega*m*(y[1] + 1.0i*y[2])/(x*tomega)
         + (tomega2-kappa2)*(y[3] + 1.0i*y[4])/tomega;


  diudr = (-tomega/(cs2*Omega*Omega*x*x) + m*m/(x*x*tomega))*(y[1]+1.0i*y[2])
           - (1.0/x - p/x + m*kappa2/(2.0*Omega*x*tomega))*(y[3] + 1.0i*y[4]);

  dydx[1] = __real__ dhdr;
  dydx[2] = __imag__ dhdr;

  dydx[3] = __real__ diudr;
  dydx[4] = __imag__ diudr;
  
  dydx[5] = 0;
  dydx[6] = 0;

  return;

}

void derivs(double x, double y[], double dydx[])
{
  double tomega_r;
  double tomega_i;

  _Complex double tomega;
  _Complex double tomega2;

  double kappa;
  double kappa2;
  double Omega;

  _Complex double dhdr;
  _Complex double diudr;
  
  Omega = 1.0/(x-2.0)/sqrt(x);
  kappa = Omega*sqrt((x-6.0)/(x-2.0));

  tomega_r = y[5] - m*Omega;
  tomega_i = y[6];
  tomega = tomega_r + 1.0i*tomega_i;
  tomega2 = tomega*tomega;

  kappa2 = kappa*kappa;
  
  dhdr = 2.0*Omega*m*(y[1] + 1.0i*y[2])/(x*tomega)
         + (tomega2-kappa2)*(y[3] + 1.0i*y[4])/tomega;

  diudr = (-tomega/(cs2*Omega*Omega*x*x) + m*m/(x*x*tomega))*(y[1]+1.0i*y[2])
           - (1.0/x - p/x + m*kappa2/(2.0*Omega*x*tomega))*(y[3] + 1.0i*y[4]);

  dydx[1] = __real__ dhdr;
  dydx[2] = __imag__ dhdr;

  dydx[3] = __real__ diudr;
  dydx[4] = __imag__ diudr;
  
  dydx[5] = 0;
  dydx[6] = 0;

  return;
}

double vortensity(double xval)
{
  double Sigma;
  double Omega, kappa;
  double D;
  double tomegr;
  double tomegi;
 
  Omega = 1.0/(xval-2.0)/sqrt(xval);
  kappa = Omega*sqrt((xval-6.0)/(xval-2.0));

  tomegr = omega_r - m*Omega; 
  tomegi = omega_i;

  D = kappa*kappa - tomegr*tomegr + tomegi*tomegi;

  Sigma = pow(xval, -p);
  
  return -kappa*kappa/(Omega*Sigma);
}

double dlnvort(double xval)
{
  double Omega;
  double Omegap;
  double kappa;
  double kappap;
  double tomegr;
  double tomegrp;
  
  Omega = 1.0/(xval-2.0)/sqrt(xval);
  kappa = Omega*sqrt((xval-6.0)/(xval-2.0));

  Omegap = (2.0-3.0*xval)*(xval - 2.0)*pow(xval*(xval-2.0)*(xval-2.0), -1.5)/2.0;
  kappap = Omegap*sqrt((xval-6.0)/(xval-2.0)) + Omega*2.0/(xval-2.0)/
                      (xval-2.0)/sqrt((xval-6.0)/(xval-2.0));

  tomegr = omega_r - m*Omega;
  tomegrp = -m*Omegap;

  return (2.0*kappap*kappa)/(kappa*kappa)+ p/xval - Omegap/Omega;
}


double nu(double xval)
{
  double dlnv;
  double D;

  double Omega, Omegap;
  double kappa;
  double tomegr;
  double q;
   
  Omega = 1.0/(xval-2.0)/sqrt(xval);
  kappa = Omega*sqrt((xval-6.0)/(xval-2.0));
  Omegap = (2.0-3.0*xval)*(xval - 2.0)*pow(xval*(xval-2.0), -1.5)/2.0;
 
  tomegr = omega_r - m*Omega;

  D = kappa*kappa - tomegr*tomegr;

  dlnv = dlnvort(xval);

  q = -xval*Omegap/Omega;
  
  return dlnv*sqrt(cs2*Omega*Omega*xval*xval/D)/q;
  
}

double flux(double xval, double y[])
{
  _Complex double duphi;
  double Sigma;
  _Complex double tomega;
  double kappa2;
  double Omega;
  double F;
  double kappa;

  _Complex double dh, idu;

  dh = y[1] + 1.0i*y[2];
  idu = y[3] + 1.0i*y[4];
 
  Omega = 1.0/(xval-2.0)/sqrt(xval);
  kappa = Omega*sqrt((xval-6.0)/(xval-2.0));

  tomega = omega_r + 1.0i*omega_i - m*Omega;
  kappa2 = kappa*kappa;

  Sigma = pow(xval, -p);

  duphi = (m*dh)/(xval*tomega) - kappa2*idu/(2*Omega*tomega);

  F = PI*xval*xval*Sigma*((__imag__ idu)*(__real__ duphi) 
			  - (__real__ idu)*(__imag__ duphi));

  return F;

}

double flux3(double xval, double y[])
{
  _Complex double duphi;
  _Complex double dur;
  double Sigma;
  _Complex double tomega;
  double kappa2;
  double Omega;
  double F;
  double kappa;

  _Complex double dh;

  dh = y[1] + 1.0i*y[2];
  dur = y[4] - 1.0i*y[3];
  
  Omega = 1.0/(xval-2.0)/sqrt(xval);
  kappa = Omega*sqrt((xval-6.0)/(xval-2.0));
  tomega = omega_r + 1.0i*omega_i - m*Omega;
  kappa2 = kappa*kappa;
  Sigma = pow(xval, -p);

  duphi = (m*dh)/(xval*tomega) - 1.0i*kappa2*dur/(2*Omega*tomega);

  F = PI*xval*xval*Sigma*cabs(dur)*cabs(duphi)*cos(carg(dur) - carg(duphi));

  return F;

}


double flux2(double xval, double y[]){

  double yprime[7];
  double Sigma;
  double kappa;
  double kappa2;
  double Omega;
  double D;
  double F;
  _Complex double dh;
  _Complex double dhdr;

  Omega = 1.0/(xval-2.0)/sqrt(xval);
  kappa = Omega*sqrt((xval-6.0)/(xval-2.0));
  kappa2 = kappa*kappa;
  Sigma = pow(xval, -p);

  derivs(xval, y, yprime);
  
  dh = y[1] + 1.0i*y[2];
  dhdr = yprime[1] + 1.0i*yprime[2];

  D = (kappa2 - pow(omega_r - m*Omega, 2.0) + omega_i*omega_i);
 
  F = PI*m*xval*Sigma*((y[2])*(yprime[1]) - (y[1])*(yprime[2]))/D;

  return F;
}

double kval(double xval)
{
  double kappa;
  double Omega;
  double tomegr;
  double D;
  double tomegi;

  Omega = 1.0/(xval-2.0)/sqrt(xval);
  kappa = Omega*sqrt((xval-6.0)/(xval-2.0));
  tomegr = omega_r - m*Omega;
  tomegi = omega_i;
 
  D = kappa*kappa - tomegr*tomegr+tomegi*tomegi;

  return sqrt(fabs(D/cs2/xval/xval/Omega));
}


void load(double xinit, double v[], double y[])
{  
  double kappa;
  double Omega;
  double Omegap;
  double kappa2;
  double kappap;
  _Complex double tomegap;
  _Complex double tomega;
  double tomega_r;
  double tomega_i;
  _Complex double tomega2;
  _Complex double idu;
  _Complex double dhdr;
  double rin;
  double rout;
  double vr, vrp;
  _Complex double k;

  rin = 6.0;

  Omegaunit = 1.0/(rin-2.0)/sqrt(rin);
  omega_r = Omegaunit*v[1+offset];
  if(setlog) 
    omega_i =  Omegaunit*sign*exp(v[2+offset]);
  else
    omega_i = Omegaunit*v[2+offset];

  fprintf(stderr, "v = %20.20E , %20.20E\n", v[1], v[2]);  

  rout = OUTFACTOR*zbrent(rolfunc, rin, 100, 1e-5);

  x1 = rout;
  x2 = rin;

  Omega = 1.0/(x1-2.0)/sqrt(x1);
  kappa = Omega*sqrt((x1-6.0)/(x1-2.0));

  tomega_r = omega_r - m*Omega;
  tomega_i = omega_i;
  tomega = tomega_r + 1.0i*tomega_i;
  tomega2 = tomega*tomega;

  kappa2 = kappa*kappa;

  Omegap = (2.0-3.0*x1)*(x1 - 2.0)*pow(x1*(x1-2.0)*(x1-2.0), -1.5)/2.0;
  kappap = Omegap*sqrt((x1-6.0)/(x1-2.0)) 
    + Omega*2.0/(x1-2.0)/(x1-2.0)/sqrt((x1-6.0)/(x1-2.0));

  tomegap = -m*Omegap;

  vr = x1*Omega;
  vrp = x1*Omegap + Omega;

  k = csqrt((tomega2- kappa2)/(cs2*Omega*Omega*x1*x1));
  
  dhdr = 0.5*( vrp/vr - (tomega*tomegap  - kappa*kappap)/(tomega2-kappa2))
    + 1.0i*k;

  y[1]=1.0;
  y[2]=0.0;

  idu = (-2.0*m*Omega*(y[1] + 1.0i*y[2])/x1 + tomega*dhdr)/(tomega2 - kappa2);

  y[3] = __real__ idu;
  y[4] = __imag__ idu;

  y[5] = omega_r;
  y[6] = omega_i;

  return;
}



void score(double xf, double y[], double f[])
{
  _Complex double DeltaP;
  _Complex double xi;
  _Complex double tomega;
  _Complex double h;
  double Omega;

  Omega = 1.0/(xf-2.0)/sqrt(xf);
  tomega = (omega_r + 1.0i*omega_i) - m*Omega;

  xi = (y[3] + 1.0i*y[4])/(tomega);
  h = y[1] + 1.0i*y[2];
  DeltaP = h - xi*(p*cs2*Omega*Omega*xf);

  f[1] = __real__ DeltaP;
  f[2] = __imag__ DeltaP;

  fprintf(stderr, "f1 = %10E, f2 = %10E\n", f[1], f[2]);
  return;
}

void score2(double xf, double y[], double f[])
{
  f[1] = y[1]*y[1] + y[2]*y[2];
  return;
}


int main(int argc, char *argv[])
{
  int i;
  double *v;
  int check;
  double omega_rguess;
  double omega_iguess;
  double f;
  double f2, f3, vort;
  double yprime[7];
  double yprime2[7];
  double ytemp[7];
  double y[7];
  double h1, hmin = 0.0;
  int nbad,nok;
  double scoreout[2];
  char outfile[100];

  nvar = 6;

  if(argc < 8)
  {
     fprintf(stderr,"Usage: %s <omega_r> <omega_i> <m> <p> <c> <eps>\
                    <outfile.perturb>\n", argv[0]);
     return 1;
  }

  setlog = 0;

  v = malloc(sizeof(double)*(2+1));
  omega_rguess = atof(argv[1]);
  omega_iguess = atof(argv[2]);
  m = atof(argv[3]);
  p = atof(argv[4]);
  cs2 = pow(atof(argv[5]),2.0);
  eps = atof(argv[6]);
  strcpy(outfile, argv[7]);

  v[1] = omega_rguess;
  if(setlog)
  {
     v[2] = log(fabs(omega_iguess));
     sign = (omega_iguess < 0 ? -1.0: 1.0);
  }
  else 
  {
     v[2] = omega_iguess;
  }
  offset = 0;
  load(x2, v, y);
  fprintf(stderr, "x1 = %20.20E, x2 = %20.20E\n", x1, x2);
  derivs(x1, y, yprime);
  fprintf(stderr, "y[1] = %20.20E, y'[1] = %20.20E\ny[2] = %20.20E, \
                   y'[2] = %20.20E\ny[3] = %20.20E, y'[3] = %20.20E\n\
                   y[4] = %20.20E, y'[4] = %20.20E\ny[5]= %20.20E, \
                   y'[5] = %20.20E\ny[6] = %20.20E, y'[6] = %20.20E\n", 
                   y[1], yprime[1], y[2], yprime[2], y[3], yprime[3], y[4], 
                   yprime[4], y[5], yprime[5], y[6], yprime[6]);
  newt(v, 2, &check, shoot);

  FILE *output_file;
  FILE *binary_file;

  if ((output_file = fopen("./discperturb.dat", "w")) == NULL)
    fprintf(stderr, "Cannot open %s\n", "./discperturb.dat");

  if((binary_file = fopen(outfile, "wb")) == NULL)
    fprintf(stderr, "Cannout open %s\n", outfile);

  if(check)
  {
     fprintf(stderr, "Shoot failed, bad initial guess\n");
     return 1;
  }
  
  load(x2, v, y);
  
  derivs(x1, y, yprime);
  derivscomplex(x1, y, yprime2);
  
  h1 = (x2-x1)/100.0;
  fprintf(stderr, "rin = %19.18E, rout = %19.18E\n", x2, x1);

  kmax = 10000000;
  xp = malloc(sizeof(double)*(kmax+1));
  yp = malloc(sizeof(double*)*(nvar+1));
  for(i = 1; i <= nvar;i++)
    yp[i] = malloc(sizeof(double)*(kmax + 1)); 
  
  odeint(y, nvar, x1, x2, eps, h1, hmin, &nok, &nbad, derivs, rkqs);
  
  score(x2, y, scoreout);

  f = flux(x2, y);
  fprintf(stderr, "dhr, dhi, idur, idui, omega_r, omega_i, f = \n");
  fprintf(stderr, "%25.18E %19.18E %19.18E %19.18E %19.18E %19.18E %19.18E\n",
                   y[1], y[2], y[3], y[4], y[5]/Omegaunit, y[6]/Omegaunit, f);
  fprintf(stderr, "Omu=%19.18E\n", Omegaunit);

  fprintf(stdout, "%g %g %g %20.20E %20.20E %20.20E %20.20E\n", m, p, sqrt(cs2), 
          y[5]/Omegaunit, y[6]/Omegaunit, scoreout[1], scoreout[2]);

  //Write basic parameters.
  fwrite(&m, sizeof(double), 1, binary_file);
  fwrite(&p, sizeof(double), 1, binary_file);
  fwrite(&cs2, sizeof(double), 1, binary_file);
  fwrite(&y[5], sizeof(double), 1, binary_file);
  fwrite(&y[6], sizeof(double), 1, binary_file);
  fwrite(&kount, sizeof(int), 1, binary_file);

  fprintf(stderr, "kount = %d\n", kount);
  

  for(i=1; i< kount; i++)
  {
     ytemp[1] = yp[1][i];
     ytemp[2] = yp[2][i];
     ytemp[3] = yp[3][i];
     ytemp[4] = yp[4][i];
     ytemp[5] = yp[5][i];
     ytemp[6] = yp[6][i];

     derivs(xp[i], ytemp, yprime);

     f = flux(xp[i], ytemp);
     f2 = flux2(xp[i], ytemp);
     f3 = flux3(xp[i], ytemp);
     vort = vortensity(xp[i]);
     fprintf(output_file, "%19.18E %19.18E %19.18E %19.18E %19.18E %19.18E \
                           %19.18E %19.18E %19.18E %19.18E %19.18E %19.18E \
                           %19.18E\n", xp[i], yp[1][i], yp[2][i], yp[3][i], 
                           yp[4][i], yp[5][i]/Omegaunit, yp[6][i]/Omegaunit, f, 
                           f2, f3, yp[1][i]/yp[2][i], yprime[1], yprime[2]);
  }

  fwrite(xp, sizeof(double), kount, binary_file);
  fwrite(yp[1], sizeof(double), kount, binary_file);
  fwrite(yp[2], sizeof(double), kount, binary_file);
  fwrite(yp[3], sizeof(double), kount, binary_file);
  fwrite(yp[4], sizeof(double), kount, binary_file);

  fclose(output_file);
  fclose(binary_file);

  return 0;
}
