/* an_raytrace
 * Analytic (in R and Theta) method for calculating ray trajectories.
 * Takes Boyer Linquist coordinates. Uses Quadrature calculation method for 
 * t and phi. 
 */


#include<stdlib.h>
#include<stdio.h>
#include<math.h>
#include<complex.h>
#include "nrutil.h"
#include "raytrace.h"
#include "an_math.h"
#include "kerr.h"

#define RBITRARY 200.0
#define APPROACH 1.0e-8
#define TINY 1.0e-30
#define RSMALL 0.001
#define RTINY 1.0e-5
#define STPLAM 1.0e-8


#define ALLREAL 1
#define TWOREAL 2
#define ALLCOMP 3

double alpha;
double beta;
double ll;
double q2;
double a;
double mu_o;
_Complex double r_1, r_2, r_3, r_4;
double z_m;
double mu_em;
double z_p;


int rtype;
int rcase;

int kmax,kount;
double *xp,**yp,dxsav;

int rcase;

#define NRANSI
#define SWAP(c,d) temp=(c);(c)=(d);(d)=temp;
#define M 7
#define NSTACK 50

double getx(double *x, int i)
{
   return x[i];
}

void sort(unsigned long n, double arr[])
{
   unsigned long i,ir=n,j,k,l=1,*istack;
   int jstack=0;
   double b,temp;

   istack=lvector(1,NSTACK);
   for (;;) {
      if (ir-l < M) {
         for (j=l+1;j<=ir;j++) {
            b=arr[j];
  	    for (i=j-1;i>=l;i--) { 
    	       if (arr[i] <= b) break;
               arr[i+1]=arr[i];
            }
	    arr[i+1]=b;
         }
	 if (jstack == 0){ 
            break;
         }
	 ir=istack[jstack--];
         l=istack[jstack--];
      } 
      else {
         k=(l+ir) >> 1;
	 SWAP(arr[k],arr[l+1])
	 if (arr[l] > arr[ir]) {
	    SWAP(arr[l],arr[ir])
	 }
	 if (arr[l+1] > arr[ir]) {
	    SWAP(arr[l+1],arr[ir])
	 }
	 if (arr[l] > arr[l+1]) {
	    SWAP(arr[l],arr[l+1])
	 }
	 i=l+1;
	 j=ir;
	 b=arr[l+1];
	 for (;;) {
	    do i++; while (arr[i] < b);
	    do j--; while (arr[j] > b);
	    if (j < i) break;
  	    SWAP(arr[i],arr[j]);
	 }
	 arr[l+1]=arr[j];
	 arr[j]=b;
	 jstack += 2;
	 if (jstack > NSTACK){
            nrerror("NSTACK too small in sort.");
         }
	 if (ir-i+1 >= j-l) {
	    istack[jstack]=ir;
	    istack[jstack-1]=i;
	    ir=j-1;
	 } 
         else {
	    istack[jstack]=j-1;
	    istack[jstack-1]=l;
	    l=i;
	 }
      }
   }
   free_lvector(istack,1,NSTACK);
}

#undef M
#undef NSTACK
#undef SWAP

void polint(double xa[], double ya[], int n, double x, double *y, double *dy)
{
   int i,m,ns=1;
   double den,dif,dift,ho,hp,w;
   double *c,*d;

   dif=fabs(x-xa[1]);
   c=vector(1,n);
   d=vector(1,n);
   for (i=1;i<=n;i++) {
      if ((dift=fabs(x-xa[i])) < dif) {
         ns=i;
	 dif=dift;
      }
      c[i]=ya[i];
      d[i]=ya[i];
   }
   *y=ya[ns--];
   for (m=1;m<n;m++) {
      for (i=1;i<=n-m;i++) {
         ho=xa[i]-x;
	 hp=xa[i+m]-x;
	 w=c[i+1]-d[i];
	 if ((den=ho-hp) == 0.0) {
            nrerror("Error in routine polint");
         }
	 den=w/den;
	 d[i]=hp*den;
	 c[i]=ho*den;
      }
      *y += (*dy=(2*ns < (n-m) ? c[ns+1] : d[ns--]));
   }
   free_vector(d,1,n);
   free_vector(c,1,n);
}

#define FUNC(x) ((*func)(x))

double trapzd(double (*func)(double), double aa, double b, int n)
{
	double x,tnm,sum,del;
	static double s;
	int it,j;

	if (n == 1) {
		return (s=0.5*(b-aa)*(FUNC(aa)+FUNC(b)));
	} else {
		for (it=1,j=1;j<n-1;j++) it <<= 1;
		tnm=it;
		del=(b-aa)/tnm;
		x=aa+0.5*del;
		for (sum=0.0,j=1;j<=it;j++,x+=del) sum += FUNC(x);
		s=0.5*(s+(b-aa)*sum/tnm);
		return s;
	}
}
#undef FUNC

#define EPS 1.0e-6
#define JMAX 20
#define JMAXP (JMAX+1)
#define KVAL 5

double qromb(double (*func)(double), double aa, double b)
{
   void polint(double xa[], double ya[], int n, double x, double *y, double *dy);
   double trapzd(double (*func)(double), double aa, double b, int n);
   void nrerror(char error_text[]);
   double ss,dss;
   double s[JMAXP],h[JMAXP+1];
   int j;

   h[1]=1.0;
   for (j=1;j<=JMAX;j++) {
      s[j]=trapzd(func,aa,b,j);	
      if (j >= KVAL) {
      polint(&h[j-KVAL],&s[j-KVAL],KVAL,0.0,&ss,&dss);
      if (fabs(dss) <= EPS*fabs(ss)) return ss;
      }
      h[j+1]=0.25*h[j];
   }
   fprintf(stderr, "error: alpha = %g, beta = %g\n", alpha, beta);
   nrerror("Too many steps in routine qromb");
   return 0.0;
}

#undef EPS
#undef JMAX
#undef JMAXP
#undef KVAL


#define EPS 1.0e-6
#define JMAX 14
#define JMAXP (JMAX+1)
#define KVAL 5

double qromo(double (*func)(double), double aa, double b,
	double (*choose)(double(*)(double), double, double, int))
{
  void polint(double xa[], double ya[], int n, double x, double *y, double *dy);
  void nrerror(char error_text[]);
  int j;
  double ss,dss,h[JMAXP+1],s[JMAXP];

  h[1]=1.0;
  for (j=1;j<=JMAX;j++) 
  {
     s[j]=(*choose)(func,aa,b,j);
     if (j >= KVAL) 
     {
        polint(&h[j-KVAL],&s[j-KVAL],KVAL,0.0,&ss,&dss);
	if (fabs(dss) <= EPS*fabs(ss)) return ss;
     }
     h[j+1]=h[j]/9.0;
  }
  nrerror("Too many steps in routing qromo");
  return 0.0;
}
#undef EPS
#undef JMAX
#undef JMAXP
#undef KVAL

#define FUNC(x) ((*func)(x))

double midpnt(double (*func)(double), double aa, double b, int n)
{
   double x,tnm,sum,del,ddel;
   static double s;
   int it,j;

   if (n == 1) {
      return (s=(b-aa)*FUNC(0.5*(aa+b)));
   } 
   else {
      for(it=1,j=1;j<n-1;j++) it *= 3;
      tnm=it;
      del=(b-aa)/(3.0*tnm);
      ddel=del+del;
      x=aa+0.5*del;
      sum=0.0;
      for (j=1;j<=it;j++) {
         sum += FUNC(x);
	 x += ddel;
	 sum += FUNC(x);
	 x += del;
      }
      s=(s+(b-aa)*sum/tnm)/3.0;
      return s;
   }
}
#undef FUNC

_Complex double an_r(double lambda)
{
   _Complex double u;
   _Complex double sn12;
   _Complex double sninf;
   _Complex double sninf2;
   _Complex double sn, cn, dn;
   _Complex double emmc;
   _Complex double r;

   u = csqrt((r_2-r_4)*(r_1-r_3))*lambda/2.0;
   emmc = 1.0 - (r_1-r_4)*(r_2-r_3)/((r_2-r_4)*(r_1-r_3));
   sninf = csqrt((r_2-r_4)/(r_1-r_4));
   sninf2 = (r_2-r_4)/(r_1-r_4);

   // Using InverseSN
   u = cellf(csqrt((r_2-r_4)/(r_1-r_4)), 1.0-emmc) - u;
   csncndn(u, emmc, &sn, &cn, &dn);
   sn12 = sn*sn;
   r = (r_1 - r_2*sn12/sninf2)/(1.0-sn12/sninf2);

   return r;
}

double an_r_real(double lambda)
{
   _Complex double temp = an_r(lambda);
   double final = (__real__ final);
   return final;
}

double r_real(_Complex double r_comp)
{
   return (__real__ r_comp);
}

double Chi(double mu)
{
   double zr, chi;
   zr = mu*mu/z_p;

   chi = acos(sqrt(zr));
   return chi;
}

double an_mu(double lambda) 
{
   double mu;
   double phi;
   double phi_o;
   double k;
   double sn, cn, dn;


   if (alpha == 0.0 && beta == 0.0) return mu_o;
   
   k = sqrt(z_p/(z_p - z_m));
   phi_o = ellf(-SIGN(1.0, beta)*Chi(mu_o),k);
   phi = a*sqrt(z_p-z_m)*lambda + phi_o;

   sncndn(phi, 1.0-k*k, &sn, &cn, &dn);

   mu = sqrt(z_p)*cn;
   return mu;
}

// For deltalambda after the turning point, take 
// 2*deltalambda_r(r_turn) - deltalambda_r(r_em)
_Complex double deltalambda_r(double r_em)
{
   _Complex double dlambda;
   _Complex double dlambda2;
   _Complex double isn_i, isn_f;
   _Complex double m;
   _Complex double z2_em;
   _Complex double K;
  
   m = (r_1-r_4)*(r_2-r_3)/((r_2-r_4)*(r_1-r_3));
   z2_em = (r_em - r_1)*(r_2-r_4)/((r_em-r_2)*(r_1-r_4));

   isn_f = cellf(csqrt((r_2-r_4)/(r_1-r_4)), m);
   isn_i = cellf(csqrt(z2_em), m);

   dlambda = 2.0*(isn_f - isn_i)/csqrt((r_1-r_3)*(r_2-r_4));
   // Ambiguity with the sign of the square root inside InverseSN
   // check both signs and see if either gives a real value.
   dlambda2 = 2.0*(isn_f + isn_i)/csqrt((r_1-r_3)*(r_2-r_4));

   if (fabs(__imag__ dlambda) >=  fabs(__imag__ dlambda2)){
     return dlambda2;
   }
   return dlambda;
}

// Using substitution r = r_tp cosh^2(psi)
double Phi3_tp(double psi)
{
   _Complex double temp;
   double coshpsi;
   double dldpsi;
   double out;
   double r;

   coshpsi = cosh(psi);

   if (rtype == ALLREAL){
      r = (__real__ r_1)*coshpsi*coshpsi;
      temp = (r-r_2)*(r-r_3)*(r-r_4);
   }
   if (rtype == TWOREAL){
      r = r_3*coshpsi*coshpsi;
      temp = (r-r_1)*(r-r_2)*(r-r_4);
   }
   dldpsi = 2.0*sqrt(r)/(sqrt(__real__ temp));
   out = a*r*r + a*a*a-ll*a*a;
   out /= r*r-2.0*r+a*a;
   out *= dldpsi;
  
   return out;
}

double Phi3_r(double r){
   double temp;
   double dldr;
   double out;

   temp = (r*r + a*a- ll*a);
   temp = temp*temp - (r*r-2*r+a*a)*((ll-a)*(ll-a)+q2);
   dldr = 1.0/sqrt(temp);
   out = a*r*r + a*a*a-ll*a*a;
   out /= r*r-2.0*r+a*a;
   out *= dldr;

   if(isnan(out)){
      fprintf(stderr, "(r*r + a*a- ll*a) = %g\n", (r*r + a*a- ll*a));
      fprintf(stderr, "(r*r-2*r+a*a)*((ll-a)*(ll-a)+q2) = %g\n", (r*r-2*r+a*a)*((ll-a)*(ll-a)+q2));
      fprintf(stderr, "a*r*r + a*a*a-ll*a*a/r*r-2.0*r+a*a =%g\n", (a*r*r + a*a*a-ll*a*a)/(r*r-2.0*r+a*a));
  }
  return out;
}


//Using substitution r = 1/eps 
double Phi3_inf(double eps)
{
   double dldeps;
   double r;
   double out;
   double temp;

   r = 1.0/eps;
   temp = (r*r + a*a- ll*a);
   temp = temp*temp - (r*r-2*r+a*a)*((ll-a)*(ll-a)+q2);
   dldeps = 1.0/(eps*eps*sqrt(temp));

   out = a*r*r + a*a*a-ll*a*a;
   out /= r*r-2.0*r+a*a;
   out *= dldeps;

   return out;
}

// Using substitution r = r_tp cosh^2(psi)
double T3_tp(double psi)
{
   _Complex double temp;
   double coshpsi;
   double dldpsi;
   double out;
   double r;

   coshpsi = cosh(psi);

   if (rtype == ALLREAL){
      r = (__real__ r_1)*coshpsi*coshpsi;
      temp = (r-r_2)*(r-r_3)*(r-r_4);
   }
   if (rtype == TWOREAL){
      r = (__real__ r_3)*coshpsi*coshpsi;
      temp = (r-r_1)*(r-r_2)*(r-r_4);
   }
   dldpsi = 2.0*sqrt(r)/sqrt(__real__ temp);
   out = ((r*r + a*a) - a*ll)*(r*r+a*a);
   out /= r*r-2.0*r+a*a;
   out *= dldpsi;

   return out;
}

//Using substitution r = 1/eps 
double T3_inf(double eps){
   double dldeps;
   double r;
   double out;
   double temp;

   r = 1.0/eps;
   temp = (r*r + a*a- ll*a);
   temp = temp*temp - (r*r-2*r+a*a)*((ll-a)*(ll-a)+q2);
   dldeps = 1.0/(eps*eps*sqrt(temp));

   out = ((r*r + a*a) - a*ll)*(r*r+a*a);
   out /= r*r-2.0*r+a*a;
   out -= sqrt(temp)*(1.0 + 2.0*r/(r*r-2.0*r+a*a));
   out *= dldeps;

   return out;
}

double T3_r(double r){
   double dldr;
   double out;
   double temp;

   temp = (r*r + a*a- ll*a);
   temp = temp*temp - (r*r-2*r+a*a)*((ll-a)*(ll-a)+q2);
   dldr = 1.0/(sqrt(temp));

  out = ((r*r + a*a) - a*ll)*(r*r+a*a);
  out /= r*r-2.0*r+a*a;
  out *= dldr;

  return out;
}

void derivs(double x, double *y, double *dydx){
   _Complex double cr;
   double mu;
   double p;
   double r;
   double mu2;
   double Delta;

   cr = an_r(x);
   r = __real__ cr;
   mu = an_mu(x);
   mu2 = mu*mu;
   Delta = kerr_Delta(r, mu, a);

   //dtdlambda  
   p  = (r*r + a*a)/Delta;
   p *= (r*r + a*a - ll*a);
   p -= a*(a*(1-mu2) - ll);
   dydx[1] = p;

   //dphidlambda
   p  = ll/(1- mu2);
   p += a*(r*r + a*a - ll*a)/Delta;
   p -= a;
   dydx[2] = p;
}




double deltalambda_mu(double chi, double chi_o){
   double F1, F2;
   double k;
   double dl;

   k = sqrt(z_p/(z_p - z_m));
  
   F1 = ellf(chi, k);
   F2 = ellf(chi_o, k);
   dl  = F1-F2;
   dl /= a*sqrt(z_p - z_m);
  
   return dl;
}


void wavevec(double lambda, double r, double mu, double *p){
  
   double sign_r, mu2, rho2, Delta;
   double temp;
   _Complex double lambdaturn;
   mu2 = mu*mu;
   rho2 = r*r + a*a*mu2;

   // set sign change of r, 
   if (rtype == ALLREAL){
      lambdaturn = deltalambda_r(r_1);
   }
  
   if (rtype == TWOREAL){
      lambdaturn = deltalambda_r(r_3);
   }
  
   if (rtype == ALLCOMP){
      sign_r = 1.0;
   }
   else{
      if ((lambda > __real__ lambdaturn)&&(fabs(__imag__ lambdaturn) <= TINY)){
         sign_r = -1.0;
      }
   else sign_r = 1.0;
   }
   Delta = kerr_Delta(r, sqrt(mu2), a);
  
   p[0]  = (r*r + a*a)/Delta;
   p[0] *= (r*r + a*a - ll*a);
   p[0] -= a*(a*(1-mu2) - ll);
   p[0] /= rho2;

   temp = (r*r + a*a- ll*a);
   temp = temp*temp - (r*r-2*r+a*a)*((ll-a)*(ll-a)+q2);

   p[1] = sign_r*sqrt(temp)/rho2;
   p[2] = - sqrt(a*a*(z_p - mu2)*(mu2 - z_m)/(1.0 - mu2))/rho2;

   p[3]  = ll/(1- mu2);
   p[3] += a*(r*r + a*a - ll*a)/Delta;
   p[3] -= a;
   p[3] /= -rho2;

   return;
}

#define TINY  1.0e-30

int raytrace(double llval, double q2val, double aval, double mu_oval, 
	      double alphaval, double betaval, double mu_emval, 
	      double *x, double *p 
	      ){

  double x2;
  double K;
  double A, B, C, D, E, F;
  double r;
  double mu;
  _Complex double dlr;
  _Complex double stoplambda;
  _Complex double ra, rb, rc, rd;
  double rturn;
  double rroots[5];
  double Phi_1, Phi_2, Phi_3;
  double T_1, T_2, T_3;
  double epsmax, ktheta;
  int i;

  // WE ASSUME XP AND YP ARE INITIALIZED BY THE CALLING FUNCTION. (AND FREED)
  // SET CONSTANTS FOR THIS LIGHTCURVE
  ll = llval;
  q2 = q2val;
  a = aval;
  mu_o = mu_oval;
  beta = betaval;
  alpha = alphaval;
  mu_em = mu_emval;


  // FIND ROOTS Setup root global variables
 
  //Find roots of R(r)
  C = (a - ll)*(a- ll) + q2;
  D = (2.0/3.0)*(q2 + ll*ll - a*a);
  E = (9.0/4.0)*D*D - 12.0*a*a*q2;
  F = -27.0*D*D*D/4.0 - 108.0*a*a*q2*D + 108.0*C*C;
  if (F*F >= 4.0*E*E*E){
    A = cuberoot((F + SIGN(1.0,F)*sqrt(F*F - 4.0*E*E*E))/2.0);
    A += E*cuberoot((2.0)/(F + SIGN(1.0, F)*sqrt(F*F - 4.0*E*E*E))); 
    A /= 3.0;
   }else{
    A = (2.0/3.0)*sqrt(E)*cos(acos(F/(2*sqrt(E*E*E)))/3.0);
    
  }

  B = sqrt(A + D);

  ra = (B + csqrt(-A + 2.0*D - 4.0*C/B))/2.0;
  rb = (B - csqrt(-A + 2.0*D - 4.0*C/B))/2.0;
  rc = (-B + csqrt(-A + 2.0*D + 4.0*C/B))/2.0;
  rd = (-B - csqrt(-A + 2.0*D + 4.0*C/B))/2.0;	
  
  if((-A + 2.0*D + 4.0*C/B) >= 0.0){ // rc rd real
    if((-A + 2.0*D -4.0*C/B) >= 0.0){ // all real
      //sort such that r1 is largest.
      rroots[1] = __real__ ra;
      rroots[2] = __real__ rb;
      rroots[3] = __real__ rc;
      rroots[4] = __real__ rd;
      sort(4, rroots);
      r_4 = rroots[1];
      r_3 = rroots[2];
      r_2 = rroots[3];
      r_1 = rroots[4];
      rtype = ALLREAL;
      
    }else{//rc rd real, ra rb complex
      r_1 = ra;
      r_2 = rb;
      r_3 = rc;
      r_4 = rd;
      rtype = TWOREAL;
    }    
  }else{ 
    if((-A + 2.0*D -4.0*C/B) >= 0.0){ //rc rd complex, ra rb real
      r_1 = rc;
      r_2 = rd;
      r_3 = ra;
      r_4 = rb;
      rtype = TWOREAL;

    }else{
      //all complex
      // check for different cases of k 
      r_1 = ra;
      r_2 = rb;
      r_3 = rc;
      r_4 = rd;
      rtype = ALLCOMP;

    }
  }

   //Find roots of Z(z)

  K = q2 + ll*ll - a*a;

  z_p = -K/(2*a*a) + sqrt(K*K/(4*a*a*a*a) + q2/(a*a));
  z_m =  -K/(2*a*a) - sqrt(K*K/(4*a*a*a*a) + q2/(a*a));

  //DETERMINE BOUNDS OF RAYTRACE

  if(alpha == 0.0 && beta == 0.0){
    x[0] = 0.0;
    x[1] = 1.0 + sqrt(1.0 - a*a );
    x[2] = acos(mu_o);
    x[3] = 0.0;
    p[0] = 0.0;
    p[1] = 0.0;
    p[2] = 0.0;
    p[3] = 0.0;
    return 0;
  }
  if(mu_em*mu_em - z_m <=  TINY){
      x[0] = 0.0;
      x[1] = 1.0 + sqrt(1.0 - a*a );
      x[2] = acos(mu_o);
      x[3] = 0.0;
      p[0] = 0.0;
      p[1] = 0.0;
      p[2] = 0.0;
      p[3] = 0.0;
      return 0;
  }else{
    x2 = deltalambda_mu(Chi(mu_em), -SIGN(1.0, beta)*Chi(mu_o));
  }



  stoplambda = deltalambda_r( 1.0 + sqrt(1.0 - a*a));
  r = __real__ an_r(x2);

    
  if ((fabs(__imag__ stoplambda) <= STPLAM && x2 > __real__ stoplambda)
     ||(r <= Calc_risco(a))){ //1.0 + sqrt(1.0 - a*a) + RSMALL)){
    x[0] = 0.0;
    x[1] = 1.0 + sqrt(1.0 - a*a );
    x[2] = acos(mu_em);
    x[3] = 0.0;
    p[0] = 0.0;
    p[1] = 0.0;
    p[2] = 0.0;
    p[3] = 0.0;
    return 0;
  }
  mu = an_mu(x2);
  ktheta = sqrt(z_p/(z_p - z_m));
  
  Phi_1 = -a*x2;
  T_1 = a*ll*x2;

  Phi_2 = ellpi(Chi(mu), z_p/(1.0-z_p) , ktheta);
  Phi_2 -= ellpi(-SIGN(1.0, beta)*Chi(mu_o), z_p/(1.0-z_p) , ktheta);
  Phi_2 *= ll/(a*(1.0-z_p)*sqrt(z_p-z_m));

  T_2 = elle(Chi(mu), ktheta);
  T_2 -= elle(-SIGN(1.0, beta)*Chi(mu_o), ktheta);
  T_2 *= -a*z_p/(ktheta*ktheta*sqrt(z_p-z_m));
  T_2 += a*a*(1.0 + z_p*(1.0-ktheta*ktheta)/(ktheta*ktheta))*x2;

  if (rtype == ALLCOMP){
    Phi_3 = qromb(Phi3_r, r, RBITRARY);
    T_3 = qromb(T3_r, r, RBITRARY);
  }else{
    if (rtype == ALLREAL) rturn = __real__ r_1;
    if (rtype == TWOREAL) rturn = __real__ r_3;


    if (r-rturn <= RSMALL){
      //r is on the turning point.;
      Phi_3 = qromb(Phi3_tp, 
		    acosh(sqrt(r/rturn)), 
		    acosh(sqrt(RBITRARY/rturn)));
      T_3 = qromb(T3_tp,
		  acosh(sqrt(r/rturn)),
		  acosh(sqrt(RBITRARY/rturn)));
    }else{
      //fprintf(stderr, "calc Phi_3 Phi3_r\n");
      Phi_3 = qromb(Phi3_r, r, RBITRARY);
      //fprintf(stderr, "calc T_3 T3_r\n");
      T_3 = qromb(T3_r, r, RBITRARY);
    }
    //check to see if turning point is encountered.
    dlr = deltalambda_r(rturn);
    if ((x2 > __real__ dlr) && (fabs(__imag__ dlr) <= TINY)){
      Phi_3 = 2*qromb(Phi3_tp, 0.0, acosh(sqrt(RBITRARY/rturn)))-Phi_3;
      T_3  = 2*qromb(T3_tp, 0.0,acosh(sqrt(RBITRARY/rturn)))-T_3;
        //check T3_tp
    }
    
  }
  Phi_3 += qromo(Phi3_inf, 0.0, 1.0/RBITRARY, midpnt);
  T_3 += qromo(T3_inf, 0.0, 1.0/RBITRARY, midpnt);

  x[0] = T_1 + T_2 + T_3;
  x[1] = r;
  x[2] = acos(mu);
  x[3] = Phi_1 + Phi_2 + Phi_3;
  FILE* outfile;
  outfile = fopen("lam.dat", "a+");
  wavevec(x2, r, mu, p);
  fprintf(outfile, "%f  %f  %f\n", alphaval, betaval, x2);
  fclose(outfile);
  return 1;
}

