function [meanQMC,stdQMC,costQMC] = qmc(l,R,skl,kinit,eps)
%
% Quasi-Monte Carlo method (randomised rank-1 lattice)
%
%   inputs:  l     ... grid level
%            R     ... number of random shifts
%            skl   ... number of KL modes (choose smaller than 500)
%            eps   ... tolerance for sampling error
%            kinit ... log_2 of initial number of lattice points
%
%   output:  costQMC  ... cost for run in FLOPs
%
%   Note: To balance the FE, truncation and sampling errors, the following 
%         values for skl and eps should be chosen
%
%    sig2=1, nu=1, lam=1:   skl = ceil(0.38*2^(l)), eps = 0.159*2^(-l+1)
%    sig2=1, nu=1, lam=0.1: skl = ceil(1.78*2^(l)), eps = 0.507*2^(-l+1)
%    sig2=1, nu=2, lam=1:   skl = ceil(0.76*2^(l/2)), eps = 0.154*2^(-l+1)
%    sig2=1, nu=1, lam=0.1: skl = ceil(2.38*2^(l/2)), eps = 0.400*2^(-l+1)
%
% ++++++++++++++++++++++++++++++++++
% Written March 2015 by R. Scheichl
% ++++++++++++++++++++++++++++++++++

% Open file with generating vector for lattice points
%fid = fopen('lattice-39101-1024-1048576.3600.txt');  % weights 1/j^1
fid = fopen('lattice-39102-1024-1048576.3600.txt');  % weights 1/j^2
%fid = fopen('lattice-33002-1024-1048576.9125.txt');  % weights 1/j^3

data = textscan(fid,'%f%f');
fclose(fid);

z = data{2};          % generating vector (3600 x 1)

% Choose the parameters for the covariance function

nu   = 1;    % smoothness parameter (in [1/2, \infty))
lam  = 1;    % correlation length (> 0) 
sig2 = 1;    % variance (> 0)

% Cost (in FLOPs) to produce one sample

cost = (2*skl+9)*2^l;

% Evaluate solution at x = 1/3

xs = 1/3;

% Grid parameters

hl = 2^(-l);
ml = 1/hl;

% RHs f=1

bl = ones(ml-1,1)*hl^2;

% Set up the KL expansions (using 1000 Gauss points in Nystrom method)

[ev,ef] = klsetup1D_matern(lam,nu,sig2,skl,hl,1000);

Ml = ef*diag(sqrt(ev));

sh = rand(R,skl);  % generate random shifts

N=2^kinit;  % initial number of lattice points
v = ([0:N-1] / N)' * z(1:skl)';

sumQMC  = zeros(1,R);
meanQMC = 0;
varQMC  = 0;

for r=1:R     % loop over shifts
        
    vs = v + repmat(sh(r,:),N,1);  % add random shift
    x  = vs-floor(vs);             % fractional part
        
    for n=1:N
        
        % Coordinate transformation -> Skl dimensional normal distribution!
        xiu = x(n,1:skl)';  
        xi = icdf('normal',xiu,0,1);

        kl = exp(Ml*xi);
        Ul = solverTh(ml,kl,bl);
        Pl = eval_pde1D(ml,Ul,xs);
                                   
        sumQMC(r) = sumQMC(r) + Pl;
            
    end
        
    % Evaluate the required sums.
    % Note: To limit the impact of round-off errors, we compute the running 
    % arithmetic mean and corrected sum of squares and use an update formula 
    % from 
    %
    % B.P. Welford: Note on a method for calculating corrected sums of
    % squares and products. Technometrics 4(3), 1962, pp. 419-420.
            
    varQMC = varQMC + (r-1)/r * (sumQMC(r)/N-meanQMC)^2;
    meanQMC = ((r-1)* meanQMC + sumQMC(r)/N)/r;
        
end

varQMC = varQMC/((R-1)*R);

maxit=100;
for it=1:maxit
          
    if varQMC < eps^2
        R*N
        costQMC = R*N*cost;
        break;
    end
    
    N = 2*N;
    
    meanQMC = 0;
    varQMC  = 0;
    
    v = ([1:2:N-1] / N )' * z(1:skl)';
    
    for r=1:R     % loop over shifts
        
        vs = v + repmat(sh(r,:),N/2,1);  % add random shift
        x  = vs-floor(vs);               % fractional part
       
        for n=1:N/2
        
            % Coordinate transformation -> mkl dimensional normal distribution !
            xiu = x(n,1:skl)';  
            xi = icdf('normal',xiu,0,1);
        
            kl = exp(Ml*xi);
            Ul = solverTh(ml,kl,bl);
            Pl = eval_pde1D(ml,Ul,xs);
                                   
            sumQMC(r) = sumQMC(r) + Pl;
            
        end
        
        % Evaluate the required sums.
        % Note: To limit the impact of round-off errors, we compute the running 
        % arithmetic mean and corrected sum of squares and use an update formula 
        % from 
        %
        % B.P. Welford: Note on a method for calculating corrected sums of
        % squares and products. Technometrics 4(3), 1962, pp. 419-420.
            
        varQMC = varQMC + (r-1)/r * (sumQMC(r)/N-meanQMC)^2;
        meanQMC = ((r-1)* meanQMC + sumQMC(r)/N)/r;
        
    end
    
    varQMC = varQMC/((R-1)*R);
    
end

stdQMC = sqrt(varQMC);

end

