
function cost = complexity_plots(l0,Lmin,Lmax,C_FE,C_trunc)

%
% Routine to do the complexity comparison plots in 1D
%
% l0 ... coarsests level in MLQMC and MLMC
% Lmin,Lmax ... range of finest level in MLQMC
% nu,lam,sig2  ... Matern parameters
% C_FE, C_trunc ... constants in error bounds
% R  ... number of shifts
%
% nu = 2, lam = 1:    C_FE = 0.154,  C_trunc = 0.051
% nu = 2, lam = 0.1:  C_FE = 0.400,  C_trunc = 12.8
%
% nu = 1, lam = 1:    C_FE = 0.159,  C_trunc = 0.023
% nu = 1, lam = 0.1:  C_FE = 0.507,  C_trunc = 1.61
%
% ++++++++++++++++++++++++++++++++++
% MLMC project function
% Written March-2015 by R. Scheichl
% ++++++++++++++++++++++++++++++++++
          
format long

% Choose the parameters for the covariance function

nu   = 1;    % smoothness parameter (in [1/2, \infty))
lam  = 1;    % correlation length (> 0) 
sig2 = 1;    % variance (> 0)

c_s = (C_trunc/C_FE)^(1/(2*nu));

cost = zeros(Lmax-Lmin+1,4);

for L=Lmin:Lmax
  
    % Set problem parameters for level L
    
    hL  = 2^(-L);
    skL = ceil(c_s * hL^(-1/nu));
    eps = 2*C_FE*hL^2;  
    
    % Estimate for standard MC
    [dum1,dum2,cost(L-Lmin+1,1)] = mc(L,skL,20,eps);
    
    % Using Giles' formula for MLMC
    [dum1,dum2,cost(L-Lmin+1,2)] = mlmc(l0,L,skL,4*L,eps);
        
    % Call qmc routine which calculates the QMC estimate        
    [dum1,dum2,cost(L-Lmin+1,3)] = qmc(L,16,skL,1,eps);
    
    % Call mlqmc routine which implements the Giles/Waterhouse algorithm
    [dum1,dum2,cost(L-Lmin+1,4)] = mlqmc(l0,L,16,skL,1,eps);
    
end

% Plot all costs against eps^{-1}

eps = 2*sqrt(2)*C_FE*4.^(-[Lmin:Lmax]);

loglog(eps,cost(:,1),'ro--');
hold on
loglog(eps,cost(:,2),'ro-');
loglog(eps,cost(:,3),'bd--');
loglog(eps,cost(:,4),'bd-');
hold off

end
