% Adaptive sparse grids -- constructs QoI directly
function test_sparse_grids(varargin)
global assem_solve_deterministic_evals
% Unfortunately there is no guaranteed way to calculate the number of
% solves in a black-box package except incrementing a global variable

% Check that we have a working spinterp
check_spinterp;

% Parse parameters or ask a user for them
params = parse_model_inputs(varargin{:});
% Extra parameters (only for SG)
if (~isfield(params, 'max_points'))
    params.max_points = input('Max number of samples = ? (default 200000): ');
    if (isempty(params.max_points))
        params.max_points = 200000;
    end
end
if (strcmpi(params.ydist, 'normal'))
    error('spinterp cannot sample normal y');
end
% Approximate mean (corresponds to Mass within build_grid_and_kle below)
Eu = 0.2;


% A priori fitted function to map spatial meshlevel into space discr. error
if (strcmpi(params.coeff, 'exp'))
    htolfun = @(x)(10.^(-0.6123*(x+1)-1.6795));  % For log-normal
else
    htolfun = @(x)(10.^(-0.59*(x+1)-3.42)); % for affine, av
end
% We use tolerances htolfun(lvl) to make plots more smooth


Q_sg = zeros(params.n_moments, params.runs, numel(params.lvls));
ttimes_sg = zeros(numel(params.lvls), 1);
evals_sg = zeros(numel(params.lvls), 1);
ilvl = 0;
for meshlevel=params.lvls
    ilvl = ilvl+1;
    tol = htolfun(meshlevel);
    fprintf('Solving for lvl=%d, tol=%3.3e\n', meshlevel, tol);
    
    % Build the discretization and KLE
    tol_kle = tol*3;
    [p,bound,W1g,W1m,spind,~,phi,lambda,Mass] = build_grid_and_kle(meshlevel, 'DN', params.nu, params.corr_length, tol_kle);
    
    % weighted KLE components
    L = numel(lambda);
    phil = full(phi*spdiags(sqrt(lambda), 0, L, L));
    
    % Coeff for the deterministic solver is evaluated as follows:
    if (strcmpi(params.coeff, 'exp'))
        cfunx = @(x)reshape(exp(phil*reshape(x, [], L).'*sqrt(params.sigma)), 1, size(p,2), size(x,1));
    else
        cfunx = @(x)reshape((10 + phil*reshape(x, [], L).'*sqrt(sigma)), 1, size(p,2), size(x,1));
    end
    
    % Sparse grids
    range = repmat([-sqrt(3) sqrt(3)],L,1);
    options = spset('DimensionAdaptive', 'on', ...
        'DimadaptDegree', 1, ...
        'FunctionArgType', 'vector', ...
        'RelTol', tol, ...
        'Vectorized', 'on', ...
        'NumberOfOutputs', params.n_moments, ...
        'MaxPoints', params.max_points, ...
        'GridType', 'Clenshaw-Curtis');
    
    fprintf('Evaluating sparse interpolant\n');
    for irun=1:params.runs
        tic;
        assem_solve_deterministic_evals = 0;
        Usg = spvals(@(x)mysplit(assem_solve_deterministic(cfunx(x),bound,W1g,W1m,spind, Mass,params.n_moments,Eu)), L, range, options);
        evals_sg(ilvl) = evals_sg(ilvl) + assem_solve_deterministic_evals;
        
        for j=1:params.n_moments
            Usg.selectOutput = j;
            Q_sg(j,irun,ilvl) = spquad(Usg)/(2*sqrt(3))^L;
        end
        ttimes_sg(ilvl) = ttimes_sg(ilvl) + toc;
        clear Usg
    end % irun
    
    save(sprintf('test_sparse_grids_%d.mat', meshlevel), 'Q_sg', 'evals_sg', 'ttimes_sg', 'L', 'meshlevel', 'tol', 'params');
end % meshlevel

if (numel(params.lvls)>1)
    % Estimate the error here
    [~,imax] = max(params.lvls);
    icomp = 1:numel(params.lvls);
    icomp(imax) = [];
    Q_ex = mean(Q_sg(:,:,imax), 2);
    err = exp(mean(log(sqrt(sum((Q_sg(:,:,icomp) - repmat(Q_ex,1,params.runs,numel(params.lvls)-1)).^2))),2))/norm(Q_ex);
    err = err(:);
    ttimes = sum(ttimes_sg(icomp, :), 2);
    fprintf('Estimated errors, corresponding to lvls: \n\t %s\n', num2str(err'));
    try
        lin_fit = fit(log10(err), log10(ttimes), 'poly1');
        plot(log10(err), log10(ttimes), '*-', log10(err), lin_fit(log10(err)));
        xlabel('log_{10} error');
        ylabel('log_{10} CPU time');
        legend('measured', sprintf('slope %g', lin_fit.p1));
        title('sparse grids');
    catch ME
        fprintf(ME.message); fprintf('\n');
    end
    err_sg = err;                                                      %#ok
end

% Copy vars to main space
vars = whos;
for i=1:numel(vars)
    if (exist(vars(i).name, 'var'))
        assignin('base', vars(i).name, eval(vars(i).name));
    end
end
end

% Housekeeping for spvals
function [varargout]=mysplit(v)
varargout = num2cell(v,1);
end

% Check/patch/initialize spinterp
function check_spinterp()
% check files
if (exist('spvals', 'file')==0)
    if (exist('spinterp_v5.1.1', 'dir')==0)
        try
            fprintf('spinterp_v5.1.1 is not found, downloading...\n');
            urlwrite('http://www.ians.uni-stuttgart.de/spinterp/download/spinterp_v5.1.1.zip', 'spinterp_v5.1.1.zip');
            unzip('spinterp_v5.1.1.zip');
        catch ME
            error('%s. Automatic spinterp download failed. Please download it from http://www.ians.uni-stuttgart.de/spinterp/download.html and unzip into this directory', ME.message);
        end
    end
    cd('spinterp_v5.1.1');
    spinit;
    cd('..');
end
% check patches
if (exist('spinterp_patched', 'file')==0)
    try
        system('patch -p0 -t -i spevalf.patch');
        system('patch -p0 -t -i spadaptvals.patch');
    catch ME
        error('%s. Automatic spinterp patching failed. Please apply spevalf.patch and spadaptvals.patch', ME.message);
    end
    f = fopen('spinterp_v5.1.1/spinterp_patched.m', 'w');
    fprintf(f, 'function [v]=spinterp_patched()\n');
    fprintf(f, 'v=true;\n');
    fprintf(f, 'end\n');
    fclose(f);
end
end

