function [U,A,F] = assem_solve_deterministic(coeff,bound,W1g,W1m,spind, Mass,n_moments,Eu)
% Deterministic solver for the fruitfly
% If Mass is not given, returns a cell of U,A,F at the coefficients coeff
% (used in als_cross).
% If Mass is given, returns U only, as a I x n_moments double matrix of the
% moments of the solution

global assem_solve_deterministic_evals
% Unfortunately there is no guaranteed way to calculate the number of
% solves in a black-box package except incrementing a global variable

% Parse coeff
[num_coeffs,nx,I]=size(coeff);
if (num_coeffs>1)
    % Extract RHS separately
    rhs = coeff(2,:,:);
    rhs = reshape(rhs, nx, I);
end
coeff = coeff(1,:,:); % the actual coeff (permeability) is here
% We need to generate separate matrices
n = size(W1g,2);
coeff = reshape(coeff, nx, I);
if (min(coeff(:))<=0)
    warning('non-positive coeff, %g elements', sum(coeff(:)<0)/numel(coeff));
    coeff(coeff<=0) = 1e-8;
end
if (any(isinf(coeff(:))))
    warning('infinite coefficient');
    coeff(isinf(coeff))=1e9;
end

if (nargout==3)
    % Return U,A,F
    A = cell(1,I);
    F = cell(1,I);
    U = cell(1,I);
elseif (nargout==1)&&(nargin<=5)
    % Return U only
    U = cell(1,I);
elseif (nargout==1)&&(nargin>5)
    % solve and evaluate QoI
    U = zeros(I,n_moments);
else
    error('Wrong input/output combination on assem_solve_deterministic');
end
for i=1:I
    % Assemble matrix from W2g and coeff
    B = reshape(coeff(1:n^2,i), n, n);
    B = sparse(B);
    B = W1g*B*W1m' + W1m*B*W1g';
    B = sparse(double(spind(:,1)),double(spind(:,2)),B(spind(:,3)),n^2,n^2); % permute
    if (num_coeffs>1)
        g = rhs(:,i);
    else
        % Eliminate the BC to the RHS
        bound_l = bound(1:numel(bound)/2);
        g = B(:,bound_l)*ones(numel(bound_l),1);
        g = -g;
    end
    g(bound) = [];
    B(bound,:) = [];                                                   %#ok
    B(:,bound) = [];                                                   %#ok
    
    % Prolong matrices
    Pb = speye(n^2);
    Pb(:,bound) = [];                                                  %#ok
    ud = zeros(n^2,1);
    if (num_coeffs==1)
        ud(bound_l) = 1;
    end
    
    % Solve
    u = B\g;
    u = Pb*u;
    u = u+ud;
    
    if (nargout==3)
        % Find the full solution and Return U,A,F
        U{i} = u;
        A{i} = B;
        F{i} = g;
        if (mod(i,100)==0); fprintf('assem_solve_deterministic: assemble/solve %d problems (full)\n', i); end
    elseif (nargout==1)&&(nargin<=5)
        U{i} = u;
        if (mod(i,100)==0); fprintf('assem_solve_deterministic: assemble/solve %d problems (U)\n', i); end
    elseif (nargout==1)&&(nargin>5)
        % Integral of u moments
        u_av = sum(Mass*u) - Eu; % Center all moments
        for j=1:n_moments
            U(i,j) = u_av.^j;
        end
        if (mod(i,100)==0); fprintf('assem_solve_deterministic: assemble/solve %d problems (q)\n', i); end
    end
end

assem_solve_deterministic_evals = assem_solve_deterministic_evals + I;
end
