/*
 * numvec2.hh
 *
 *  Created on: 2013. 9. 9.
 *      Author: parkmh
 */

#ifndef NUMVEC2_HH_
#define NUMVEC2_HH_

#include <iostream>
#include <memory>

template <class T> class NumVec{
public:
	typedef T*			iterator;
	typedef const T* 	const_iterator;
	typedef size_t	size_type;
	typedef T			value_type;

	NumVec() { create(); }
	explicit NumVec(size_type n, const T& t = T()) { create(n, t); }
	NumVec(const NumVec& v) {create(v.begin(),v.end());}
	NumVec& operator=(const NumVec&);

	~NumVec() { uncreate(); }

	T& operator[] (size_type i) { return data_[i]; }
	const T& operator[] (size_type i) const { return data_[i]; }

	void push_back(const T& t){
		if (avail_ == limit_)
			grow();
		unchecked_append(t);
	}
	void resize(size_type);

	size_type size() const {return avail_ - data_;}
	iterator begin() {return data_;}
	const_iterator begin() const { return data_; }
	iterator end() {return avail_; }
	const_iterator end() const {return avail_; }
private:
	iterator data_;	// first element in the NumVec
	iterator avail_;	// (one past) the last element in the NumVec
	iterator limit_; // (one past) the allocated memory

	// facilities for memory allocation
	std::allocator<T> alloc_;		// object to handle memory allocation

	// allocate and initialize the underlying array
	void create();
	void create(size_type, const T&);
	void create(const_iterator, const_iterator);


	// destroy the elements in the array and free memory
	void uncreate();

	// support functions for push_back
	void grow();
	void unchecked_append(const T&);

};

template <class T> void NumVec<T>::create(){
	data_ = avail_ = limit_ = 0;
}

template <class T> void NumVec<T>::create(size_type n, const T& val){
	data_ = alloc_.allocate(n);
	limit_ = avail_ = data_ + n;
	std::uninitialized_fill(data_,limit_, val);
}

template <class T>
void NumVec<T>::create(const_iterator i, const_iterator j){
	data_ = alloc_.allocate(j-i);
	limit_ = avail_ = std::uninitialized_copy(i,j,data_);
}

template <class T>
void NumVec<T>::uncreate(){
	if (data_){
		// destroy (in reverse order) the elements that were constructed
		iterator it = avail_;
		while (it != data_){
			alloc_.destroy(--it);
		}

		// return all the space that was allocated
		alloc_.deallocate(data_,limit_-data_);
	}
	// reset pointers to indicate that the NumVec is empty again
	data_ = avail_ = limit_ = 0;
}

template <class T> void NumVec<T>::resize(size_type n){
	data_ = alloc_.allocate(n);
	limit_ = avail_ = data_ + n;
}

template <class T> void NumVec<T>::grow(){
	// when growing, allocate twice as much space as currently in use
	size_type new_size = std::max(2*(limit_-data_),std::ptrdiff_t(1));

	// allocate new space and copy existing elements to the new space
	iterator new_data = alloc_.allocate(new_size);
	iterator new_avail = std::uninitialized_copy(data_,avail_,new_data);

	// return the old space
	uncreate();

	// reset pointers to point the newly allocated space
	data_ = new_data;
	avail_ = new_avail;
	limit_ = data_ + new_size;
}

// assumes avail_ points at allocated, but uninitialized space
template <class T> void NumVec<T>::unchecked_append(const T& val){
	alloc_.construct(avail_++, val);
}
#endif /* NUMVEC2_HH_ */
