/*
 * two_level_amc.hh
 *
 *  Created on: 7 Jan 2014
 *      Author: parkmh
 */

#ifndef TWO_LEVEL_AMC_HH_
#define TWO_LEVEL_AMC_HH_

#include "../mesh/point.hh"
#include "../util/config.hh"
#include <iostream>
#include <functional>
#include <string>
#include "mc_return_value.hh"
template <class Category,typename T>
class TwoLevelMC{
public:
	typedef mc_traits<Category, T> traits;
	typedef T		datatype;
	typedef typename traits::rfgenerator 	rfgenerator;
	typedef typename traits::discretization 	discretization;
	typedef typename discretization::pos		pos;
	typedef typename discretization::N			N;
	typedef typename traits::preconditioner 	preconditioner;
	typedef typename traits::lsolver			lsolver;
	typedef typename traits::qoifun				qoifun;
	typedef typename traits::secondop			secondop;
	typedef typename traits::split				split;
	int getID() const {return fine_disc_->nx(); }
	TwoLevelMC(){ rfg_ = NULL; fine_disc_ = NULL; coarse_disc_ = NULL; is_first_amg = true;}
	TwoLevelMC(const discretization&,const discretization&, const rfgenerator&);
	~TwoLevelMC(){}
	void init(const discretization&,const discretization&, const rfgenerator&);
	void resetSolver() { is_first_amg = false;}
	mc_return<T> compute() const;
	std::string getDescription() const {
		std::string s = "Monte Carlo Method with Antithetic Variates";
		return s;
	}

	friend std::ostream&
	operator<<(std::ostream& os, const TwoLevelMC& olmc){
		os << "[ Discretisation ] " << std::endl << olmc.fine_disc_ << std::endl;
		os << "[ Solver ] " << std::endl << olmc.fine_solver_ << std::endl;
		return os;
	}
private:

	const rfgenerator    			*rfg_;
	const 	discretization 	*fine_disc_;
	const 	discretization 	*coarse_disc_;
	mutable preconditioner 	fine_precond_;
	mutable lsolver		 	fine_solver_;
	mutable preconditioner 	coarse_precond_;
	mutable lsolver		 	coarse_solver_;
	qoifun				qoifun_;
	mutable CRSMatrix<datatype> 	Af_;
	mutable NumVec<datatype> 		bf_;
	mutable NumVec<datatype>     	vf_;
	mutable CRSMatrix<datatype> 	Ac_;
	mutable NumVec<datatype> 		bc_;
	mutable NumVec<datatype>     	vc_;
	mutable NumVec<datatype>     a_, anti_a_;
	mutable NumVec<datatype>     af_;
	mutable NumVec<datatype>     ac_;
	mutable bool    is_first_amg;
	secondop		secondop_;
	split			split_;
};


template <class Category, typename T>
TwoLevelMC<Category,T>::TwoLevelMC(const discretization& fine_disc,const discretization& coarse_disc, const rfgenerator& rfg)
{
	init(fine_disc, coarse_disc,rfg);

}

template <class Category, typename T>
void TwoLevelMC<Category,T>::init(const discretization& fine_disc,const discretization& coarse_disc, const rfgenerator& rfg)
{
	rfg_ = &rfg;
	fine_disc_ = &fine_disc;
	coarse_disc_= &coarse_disc;

	is_first_amg = true;
}

template <class Category, typename T>
mc_return<T> TwoLevelMC<Category,T>::compute() const {
	mc_return<T> rval;
	rfg_->generate(a_, anti_a_);

/* Original */
	split_(a_,fine_disc_->n(),af_,ac_);
	fine_disc_->discretize(Af_,bf_,af_);
	coarse_disc_->discretize(Ac_,bc_,ac_);
#ifdef AMG_RECYCLE
	if (!is_first_amg){
		fine_precond_.second_init(Af_,10,1);
		coarse_precond_.second_init(Ac_,10,1);
	} else{
		fine_precond_.init(Af_,10,1);
		coarse_precond_.init(Ac_,10,1);
		is_first_amg = false;
	}
#else
	fine_precond_.init(Af_,10,1);
	coarse_precond_.init(Ac_,10,1);
#endif

	fine_solver_.init(Af_,fine_precond_,MAX_SOLVE_CYCLE, SOLVE_TOL);
	coarse_solver_.init(Ac_,coarse_precond_,MAX_SOLVE_CYCLE, SOLVE_TOL);
	vf_.rand(bf_.size());
	vc_.rand(bc_.size());

	fine_solver_.solve(bf_,vf_);
	coarse_solver_.solve(bc_,vc_);
	rval.qh = qoifun_(*fine_disc_,vf_,af_);
	rval.q2h = qoifun_(*coarse_disc_,vc_,ac_);

/* Antithetic */
	split_(anti_a_,fine_disc_->n(),af_,ac_);
	fine_disc_->discretize(Af_,bf_,af_);
	coarse_disc_->discretize(Ac_,bc_,ac_);
#ifdef AMG_RECYCLE
	if (!is_first_amg){
		fine_precond_.second_init(Af_,10,1);
		coarse_precond_.second_init(Ac_,10,1);
	} else{
		fine_precond_.init(Af_,10,1);
		coarse_precond_.init(Ac_,10,1);
		is_first_amg = false;
	}
#else
	fine_precond_.init(Af_,10,1);
	coarse_precond_.init(Ac_,10,1);
#endif

	fine_solver_.init(Af_,fine_precond_,MAX_SOLVE_CYCLE, SOLVE_TOL);
	coarse_solver_.init(Ac_,coarse_precond_,MAX_SOLVE_CYCLE, SOLVE_TOL);
	vf_.rand(bf_.size());
	vc_.rand(bc_.size());

	fine_solver_.solve(bf_,vf_);
	coarse_solver_.solve(bc_,vc_);
	rval.qh += qoifun_(*fine_disc_,vf_,af_);
	rval.q2h += qoifun_(*coarse_disc_,vc_,ac_);
	rval.qh /= 2.0;
	rval.q2h /= 2.0;
	return rval;

}



#endif /* TWO_LEVEL_AMC_HH_ */
