/*
 * mlmc.hh
 *
 *  Created on: 2013. 8. 21.
 *      Author: parkmh
 */

#ifndef MLMC_HH_
#define MLMC_HH_
#include <cmath>
#include "../util/timer.hh"
#include <algorithm>
template <class Category, typename T>
class MLMC{
public:
	typedef T datatype;
	typedef mlmc_traits<Category,T> traits;
	typedef typename traits::mc_tag mc_tag;
	typedef typename traits::coarsest_mc_base	coarsest_mc_base;
	typedef typename traits::twolevel_mc_base twolevel_mc_base;
	typedef typename traits::coarsest_mc coarsest_mc;
	typedef typename traits::twolevel_mc twolevel_mc;

	typedef typename mc_traits<mc_tag,T>::rfgenerator rfgenerator;
	typedef typename mc_traits<mc_tag,T>::discretization discretization;
	typedef typename mc_traits<mc_tag,T>::sampling sampling;
	typedef typename discretization::pos pos;
	typedef typename discretization::N N;
	typedef typename sampling::shift_start_onelevel shift_start_onelevel;
	typedef typename sampling::shift_end_onelevel shift_end_onelevel;
	typedef typename sampling::shift_start_twolevel shift_start_twolevel;
	typedef typename sampling::shift_end_twolevel shift_end_twolevel;
	typedef typename sampling::nsamp_onelevel nsamp_onelevel;
	typedef typename sampling::nsamp_twolevel nsamp_twolevel;
	typedef typename sampling::h_onelevel h_onelevel;
	typedef typename sampling::h_twolevel h_twolevel;

	const static int dim = traits::dim;


	MLMC(){ morensamp = NULL; levelcost = NULL; tol_ = 0; level_ = 0; rfg_ = NULL; disc_ = NULL; twolevel_mc_base_ = NULL; twolevel_mc_ = NULL;}
	MLMC(const pos&, const pos&, const N&, const double &);
	~MLMC();

	void init(const pos&, const pos&, const N&, const double &);
	void set_tol(const double& tol) {tol_ = tol; std::cout << "\n\n\n@@@@@@@@@@@@ TOL\t: " << tol_ << " @@@@@@@@@@@@"<<std::endl;}
	void run();
private:
	coarsest_mc_base coarsest_mc_base_;
	twolevel_mc_base *twolevel_mc_base_;
	coarsest_mc		coarsest_mc_;
	twolevel_mc		*twolevel_mc_;
	discretization * disc_;
	rfgenerator * rfg_;
	shift_start_onelevel shift_start_onelevel_;
	shift_start_twolevel shift_start_twolevel_;
	shift_end_onelevel shift_end_onelevel_;
	shift_end_twolevel shift_end_twolevel_;
	nsamp_onelevel nsamp_onelevel_;
	nsamp_twolevel nsamp_twolevel_;
	h_onelevel h_onelevel_;
	h_twolevel h_twolevel_;

	double tol_;
	int level_;
	pos start_, end_;
	N N_cst_;
	N N_level_;
	N N_sample_pt_;
	pos h_level_;
	double * levelcost;
	int * morensamp;
	Timer runTime_;

	void addlevel(const size_t);

	double totalcost();
	void update();
	void print();
	bool check_bias();
};

template <class Category, typename T>
MLMC<Category,T>::MLMC(const pos& start, const pos& end, const N& N_cst,  const double &tol){
	rfg_ = NULL;
	disc_ = NULL;
	twolevel_mc_base_ = NULL;
	twolevel_mc_ = NULL;
	levelcost = NULL;
	morensamp = NULL;
	init(start,end,N_cst,tol);
}


template <class Category, typename T>
MLMC<Category,T>::~MLMC(){
	delete []  rfg_;
	delete []  disc_;
	delete [] twolevel_mc_;
	delete [] twolevel_mc_base_;
	delete [] levelcost;
	delete [] morensamp;
}

template <class Category, typename T>
void MLMC<Category,T>::init(const pos& start, const pos& end , const N& N_cst, const double & tol){
#ifdef RAND_SEED
	srand ( time(NULL) );
#endif
	runTime_.tic();
	level_ = 0;
	start_ = start;
	end_= end;
	N_cst_ = N_cst;
	tol_ = tol;

	delete [] rfg_;
	delete [] disc_;
	delete [] twolevel_mc_;
	delete [] twolevel_mc_base_;
	delete [] levelcost;
	delete [] morensamp;
	rfg_ = new rfgenerator[traits::maxlevel];
	disc_ = new discretization[traits::maxlevel];
	twolevel_mc_base_ = new twolevel_mc_base[traits::maxlevel-1];
	twolevel_mc_ = new twolevel_mc[traits::maxlevel-1];
	levelcost = new double[traits::maxlevel];
	morensamp = new int[traits::maxlevel];
	/*
	 * Level 0
	 */
	std::cout << "*\tDimension\t:" << dim << std::endl;
	std::cout << "*\tSTART\t:" << start_ << std::endl;
	std::cout << "*\tEND\t:" << end_ << std::endl;
	std::cout << "*\tN_cst\t:" << N_cst_ << std::endl;
	std::cout << "*\tDiscretisation\t:" << disc_[0].getDescription() << std::endl;
	std::cout << "*\tRF Generator\t:" << rfg_[0].getDescription() << std::endl;
	std::cout << "*\tOneLevelMC\t:" << coarsest_mc_base_.getDescription() << std::endl;
	std::cout << "*\tTwoLevelMC\t:" << twolevel_mc_base_[0].getDescription() << std::endl;



	std::cout << std::endl;
	std::cout << "\n\n\n@@@@@@@@@@@@ TOL\t: " << tol_ << " @@@@@@@@@@@@"<<std::endl;


	std::cout << "###############################################" << std::endl;
	std::cout << "                  Level   0                    " << std::endl;
	std::cout << "###############################################" << std::endl;

	N_level_ = N_cst_;
	for (size_t i = 0; i < dim; i++){
		h_level_[i] = (end_[i]-start_[i])/(double)N_level_[i];
	}

	Timer timer;
	timer.tic();

	rfg_[0].init(shift_start_onelevel_(start,h_level_),
				  shift_end_onelevel_(end,h_level_),
				  nsamp_onelevel_(N_level_),
				  h_onelevel_(h_level_));
	std::cout << rfg_[0] << std::endl;
	disc_[0].init(start,end,N_level_);
	coarsest_mc_base_.init(disc_[0],rfg_[0]);
	coarsest_mc_.init(coarsest_mc_base_,10);
	levelcost[0] = timer.getTime()/(double)coarsest_mc_.nsamp();
	/*
	 * Level 1
	 */
	std::cout << "###############################################" << std::endl;
	std::cout << "                  Level   1                    " << std::endl;
	std::cout << "###############################################" << std::endl;

	addlevel(10);
	/*
	 * Level 2
	 */
	std::cout << "###############################################" << std::endl;
	std::cout << "                  Level   2                    " << std::endl;
	std::cout << "###############################################" << std::endl;

	addlevel(10);
	run();
}



template <class Category, typename T>
void MLMC<Category,T>::addlevel(const size_t nsamp = 10){
	N_level_ *= 2;
	h_level_ /= 2.0;

	Timer timer;
	timer.tic();
	rfg_[level_+1].init(shift_start_twolevel_(start_,h_level_),
			  shift_end_twolevel_(end_,h_level_),
			  nsamp_twolevel_(N_level_),
			  h_twolevel_(h_level_));
	std::cout << rfg_[level_+1] << std::endl;
	disc_[level_+1].init(start_,end_,N_level_);
	twolevel_mc_base_[level_].init(disc_[level_+1], disc_[level_],rfg_[level_+1]);
	twolevel_mc_[level_].init(twolevel_mc_base_[level_],nsamp);
	levelcost[level_+1] = timer.getTime()/(double)twolevel_mc_[level_].nsamp();
	level_++;
}

template <class Category, typename T>
void MLMC<Category,T>::run(){
	print();
	while (level_ < traits::maxlevel){
	update();
	print();
	if (check_bias())
		break;
	addlevel();
	}
}

template <class Category, typename T>
double MLMC<Category,T>::totalcost(){

	double cost = sqrt(coarsest_mc_.varQ()*levelcost[0]);
	for (size_t level = 0; level < level_ ; level++){
		cost += sqrt(twolevel_mc_[level].varY()*levelcost[level+1]);
	}
	return cost;
}

template <class Category, typename T>
void MLMC<Category, T>::print(){
	int levelw = 5;
	int nsampw = 10;
	int meanw = 15;
	int varw = 15;
	double total_sample_var = 0.0;
	double qoi = 0.0;
	std::streamsize prec = 6;
	std::cout << std::endl;
	std::cout << std::setw(levelw) << "Level"
			   << std::setw(nsampw) <<"#samp"
			   << std::setw(meanw) << "E[Qh]"
			   << std::setw(varw) << "V[Qh]"
			   << std::setw(meanw)<< "E[Yh]"
			   << std::setw(varw) << "V[Yh]"
			   << std::setw(varw) << "Sample Var"<< std::endl;
	std::cout << std::setfill('-') << std::setw(levelw+2*meanw+3*varw+nsampw)<< "" <<  std::setfill(' ') <<std::endl;
	std::cout << std::setw(levelw) << "0"
			   << std::setw(nsampw) << coarsest_mc_.nsamp();
	std::cout.precision(prec);

	std::cout << std::setw(meanw) << std::scientific << coarsest_mc_.meanQ()
				<< std::setw(varw) << coarsest_mc_.varQ()
				<< std::setw(meanw) << coarsest_mc_.meanQ()
				<< std::setw(varw) << coarsest_mc_.varQ()
				<< std::setw(varw) << coarsest_mc_.varQ()/(double)coarsest_mc_.nsamp()<< std::endl;
	total_sample_var += coarsest_mc_.varQ()/(double)coarsest_mc_.nsamp();
	qoi = coarsest_mc_.meanQ();
	for (size_t level = 0; level < level_; level++){
		std::cout << std::setw(levelw) << level+1
				   << std::setw(nsampw) << twolevel_mc_[level].nsamp();
		std::cout.precision(prec);
		std::cout << std::setw(meanw) << std::scientific << twolevel_mc_[level].meanQ()
				   << std::setw(varw) << twolevel_mc_[level].varQ()
				   << std::setw(meanw) << twolevel_mc_[level].meanY()
				   << std::setw(varw) << twolevel_mc_[level].varY()
				   << std::setw(varw) << twolevel_mc_[level].varY()/(double)twolevel_mc_[level].nsamp()<< std::endl;
		total_sample_var += twolevel_mc_[level].varY()/(double)twolevel_mc_[level].nsamp();
		qoi += twolevel_mc_[level].meanY();
	}
	std::cout << std::setfill('-') << std::setw(levelw+2*meanw+3*varw+nsampw)<< "" <<  std::setfill(' ') <<std::endl;
	std::cout << std::setw(levelw) << "Total"
			   << std::setw(nsampw) << " "
			   << std::setw(meanw) << " "
			   << std::setw(varw) <<  " "
			   << std::setw(meanw) << qoi
			   << std::setw(varw) << " "
			   << std::setw(varw) << total_sample_var
			   << " [ " << tol_*tol_*.5 << " ]"<< std::endl;
	std::cout << std::endl;
}

template <class Category, typename T>
void MLMC<Category,T>::update(){
	double cost = totalcost();
	morensamp[0] = (int)ceil(2*sqrt(coarsest_mc_.varY()/levelcost[0])*cost/pow(tol_,2.0)) - (int)coarsest_mc_.nsamp();
	if (morensamp[0] < 0) morensamp[0] = 0;
	if (morensamp[0] > 0) std::cout << "#samp(0)+= " << morensamp[0] << std::endl;
	for (size_t level = 1; level < level_ + 1; level++){
		morensamp[level] = (int)ceil(2*sqrt(twolevel_mc_[level-1].varY()/levelcost[level])*cost/pow(tol_,2.0))-(int)twolevel_mc_[level-1].nsamp();
		if (morensamp[level]<0) morensamp[level] = 0;
		if (morensamp[level]>0)	std::cout << "#samp(" << level  << ")+= " <<  morensamp[level] << std::endl;
	}

	if (morensamp[0] > 0) std::cout << "Level 0" << std::endl;
	coarsest_mc_.apply(morensamp[0]);
	for (size_t level = 0; level < level_; level++){
		if (morensamp[level+1] > 0) std::cout << "Level " << level+1 << std::endl;
		twolevel_mc_[level].apply(morensamp[level+1]);
	}
}

template <class Category, typename T>
bool MLMC<Category,T>::check_bias(){
	runTime_.toc();
	std::cout << "Checking bias  " << fabs(twolevel_mc_[level_-1].meanY()) << ", " << fabs(twolevel_mc_[level_-2].meanY())/(double)traits::rfactor <<"," <<(traits::rfactor-1)*tol_/sqrt(2)<< std::endl;
	if (std::max(fabs(twolevel_mc_[level_-1].meanY()),
			fabs(twolevel_mc_[level_-2].meanY())/(double)traits::rfactor)
	        < (traits::rfactor-1)*tol_/sqrt(2))
		return true;
	return false;
}
#endif /* MLMC_HH_ */
