/*
 * cholesky.hh
 *
 *  Created on: 2013. 8. 5.
 *      Author: parkmh
 */

/*
 * Changelog
 * v.0.1 Added cholesky.hh
 *
 */
#ifndef CHOLESKY_HH_
#define CHOLESKY_HH_

#include "numvec.hh"
#include "crs_matrix.hh"
#include "graph.hh"
#include <iostream>


template <class T> class Cholesky{
public:
	Cholesky(){};
	Cholesky(const CRSMatrix<T> &);
	void init(const CRSMatrix<T> &);
	void solve(const NumVec<T> &, NumVec<T> &) const;

	friend std::ostream&
	operator<<(std::ostream& os, const Cholesky & cholesky){
		os << "Cholesky Decomposition";
		return os;
	}
private:
	CRSMatrix< T > L;
	CRSMatrix< T > LT;
	NumVec< size_t > order;
	void symmmd(const CRSMatrix<T>&);
	void factor(const CRSMatrix<T>&);
};

template <class T>
Cholesky<T>::Cholesky(const CRSMatrix<T> &A){
	init(A);
}
template <class T>
void Cholesky<T>::init(const CRSMatrix<T> &A){
	symmmd(A);
	factor(A);
}
template <class T>
void Cholesky<T>::symmmd(const CRSMatrix<T>& A){
	size_t nRow = A.row();
	size_t nCol = A.col();
	order.resize(nRow);
	order = 0;
	NumVec<int> oldOrder(nRow);
	NumVec<int> assigned(nRow);

	Graph G;
	Graph adjG;

	G.create(A);
	adjG.create(A);

	LT.ia_.resize(nRow+1);
	LT.ja_.resize(nRow*nCol);

	LT.ia_ = 0;
//	LT.ja_ = 0;
	ListNode< int > *currentPtr,*movingPtr;
	size_t i,j, ipi, ipip1, t;
	int lastIndex = nRow-1;
	size_t temp;
	size_t minVal = 0;
	size_t minPos = 0;
	size_t index;
	minVal = G.HeadNodes[ 0 ].getLength();

	// Calculate the degree of each edge
	for ( i = 1; i < nRow; i++ ){
		t = G.HeadNodes[ i ].getLength();
		if ( t > 0 && t < minVal ){
			minVal = t;
			minPos = i;
		}
	}
	temp = i;
	for ( i = 0; i< nRow; i++) {
		order[ i ] = minPos;
		oldOrder[ minPos ] = i;
		assigned[ minPos ] = 1;

		currentPtr = G.HeadNodes[ minPos ].getFirstPtr();
		while ( currentPtr != 0 ) {
			movingPtr = (*currentPtr).getNextPtr();

			while( movingPtr != 0){
				if (!G.HeadNodes[ (*currentPtr).getData() ].isIn( (*movingPtr).getData() ) ) {
					G.insertEdge((*currentPtr).getData(),(*movingPtr).getData());
					adjG.insertEdge((*currentPtr).getData(),(*movingPtr).getData());
				}
				movingPtr = (*movingPtr).getNextPtr();

			} // end while movinPtr

			currentPtr = (*currentPtr).getNextPtr();
		} // end while currentPtr

		G.removeVertex( minPos );

		// Detect minimum degree
		int isFirstT = 0;
		for ( j = 0; j< nRow; j++ ) {
			t = G.HeadNodes[ j ].getLength();
			if (isFirstT != 1)	{
				if( t > 0 ){
					minVal = t;
					minPos = j;
					isFirstT = 1;
				}
			}
			else {
				if ( t >0 && t < minVal ){
					minVal = t;
					minPos = j;
				}
			}
		}
		if ( minPos == temp ){
			lastIndex = i;
			break;
		}
		temp = minPos;
	}

	for  (i = 0; i< nRow; i++){
		if (assigned[ i ] == 0 ) {
			order[lastIndex + 1 ] = i;
			oldOrder[ i ]  = lastIndex + 1;
			lastIndex += 1;
		}
	}

	/*
	 * Symbolic factorization
	 */
	LT.ia( 0 ) = 0;

	size_t localIndex = 0;
	NumVec< int > tempArray;
	size_t tempIndex = 0;
	size_t orderlength = order.size();
	for ( i = 0; i < orderlength; i++)
	{
		LT.ja( LT.ia( i ) + localIndex ) = i;
		localIndex += 1;

		tempArray.resize(adjG.HeadNodes[order [ i ]].getLength());
		tempArray = 0;
		while( !adjG.HeadNodes[ order[ i ] ].isEmpty() ){
			index = oldOrder[adjG.HeadNodes[ order[ i ] ].getFirst()];

			if (index > i){
				tempArray[ tempIndex ] = index;
				tempIndex += 1;
			}
		}
		if (tempIndex > 0){
			tempArray.shrink(tempIndex);
			tempArray.sort();
			for ( j = 0; j < tempIndex; j ++ ){
				LT.ja( LT.ia( i ) + localIndex ) = tempArray[ j ];
				localIndex += 1;
			}
		}
		tempIndex = 0;
		LT.ia( i+ 1 ) = LT.ia( i ) + localIndex;
		localIndex = 0;
	}
	LT.ja_.shrink(LT.ia( nRow ));
	LT.aa_.resize(LT.ia( nRow ));


	int apIndex = 0;
	ipi = LT.ia(0);
	for( i = 0; i < nRow; i++ ){
		ipip1 = LT.ia( i + 1 );
		for (j = ipi; j < ipip1; j++ ){
			LT.aa( apIndex ) = A.get(order[ i ],order[ LT.ja( j ) ] );
			apIndex += 1;
		}
		ipi = ipip1;
	}

	LT.setSize(nRow,nCol, LT.ja_.size());
}


template<class T>
void Cholesky<T>::solve(const NumVec<T>& rhs, NumVec<T>& x) const{
	size_t m = L.row();
	size_t j, liai, liaip1, ltiai, ltiaip1, orderi, ljaj;
	size_t i, im1;
	T d;

	if (m!= x.size()) x.resize(m);

	NumVec<T> z;
	z.resize(m); z = T();
	T* zp = z.begin();

	const size_t * liap = L.iabegin();
	const size_t * ljap = L.jabegin();
	const T* laap = L.aabegin();

	const size_t * ltiap = LT.iabegin();
	const size_t * ltjap = LT.jabegin();
	const T* ltaap = LT.aabegin();

	const size_t * orderp = order.begin();
	const T* rhsp = rhs.begin();
	T* xp = x.begin();
	/* Forward substitution */
	liai = *liap;//L.ia( 0 );
	d = 0.0;
	for ( i = 0;i < m; i++ ){
		liaip1 = *(liap+i+1);
		*(zp + i ) = *(rhsp +  *(orderp + i) );
		for( j = liai; j < liaip1; j++ ) {
			ljaj = *(ljap+j);//L.ja(j);
			if( i == ljaj){
				d = *(laap+j);//L.aa(j);
			}
			else{
				*(zp + i ) -= *(laap+j) * *(zp + ljaj );
			}
		}
		liai = liaip1;
		*(zp + i) /= d;
	}


	ltiaip1 = *(ltiap+m);//LT.ia(m);
	/* Backward substitution */
	for ( i = m; i > 0; i--){
		im1 = i - 1;
		orderi = *(orderp+im1);
		*(xp + orderi) = *(zp  + im1 );
		ltiai = *(ltiap+im1);//LT.ia(im1);
		d = T();
		for( j = ltiai; j < ltiaip1; j++ ) {
			if( im1 == *(ltjap+j)){
				d = *(ltaap+j );
			}
			else {
				*(xp + orderi) -= *(ltaap+j) * *(xp +  *(orderp + *(ltjap+j ) ) );
			}
		}
		*(xp +  orderi )/= d;
		ltiaip1 = ltiai;
	}

}


template<class T>
void Cholesky<T>::factor(const CRSMatrix<T>& A){
	size_t i, j, k;
	size_t iRowIndex;
	size_t ltiai, ltiaip1, ltjaj;

	ltiai = LT.ia(0);
	T temp, ltaa;
	for ( i = 0;i < A.row(); i++ )
	{
		ltiaip1 = LT.ia( i + 1 );
		ltaa = LT.aa(ltiai);
		temp = sqrt(ltaa);
		LT.aa( ltiai ) = temp; //sqrt(ltaa);
#ifdef FACTOR_DEBUG
		printf(" i : %d \n", INDEX(i) );
		printf("   cdiv(%d)\n",INDEX(i));
		printf("\tL( %d, %d ) = %f\n",INDEX(i),INDEX(i),LT.aa[ LT.ia[ i ] ]);
#endif

		/* CDIV */
		for( j = ltiai + 1; j < ltiaip1; j ++ ) {
#ifdef FACTOR_DEBUG
			printf("\tL( %d, %d ) = L( %d, %d )(%f) /L( %d, %d )(%f) ",
					INDEX(i), INDEX(LT.ja[ j ]),INDEX(i), INDEX(LT.ja[ j ]),LT.aa[ j ]
					                                                               ,INDEX(i),INDEX(i), LT.aa[ LT.ia[ i ]]);
#endif
			LT.aa( j ) /=  LT.aa( ltiai );

#ifdef FACTOR_DEBUG
			printf(" = %f \n",LT.aa[ j ] );
#endif
		}

		for( j = ltiai + 1; j < ltiaip1; j ++ ) {
			/* CMOD*/
			iRowIndex = j;
			ltjaj = LT.ja(j);
			for( k = LT.ia(ltjaj); k < LT.ia(ltjaj + 1 ); k++ ) {
				if ( (iRowIndex == ltiaip1) | (LT.ja( k ) > LT.ja( iRowIndex )) ){
					break;
				}

				if (LT.ja( k ) == LT.ja( iRowIndex ) ) {
					LT.aa( k ) -=  LT.aa( j ) * LT.aa( iRowIndex );
					iRowIndex += 1;
				}
			}// end loop for k
		} // end loop for j
		ltiai = ltiaip1;
	} //end loop for i

	L = LT.transpose();
}
#endif /* CHOLESKY_HH_ */
