/*
 * numvec.hh
 *
 *  Created on: 2013. 8. 5.
 *      Author: parkmh
 */
/*
 * Changelog
 * v.0.1 Added numvec.hh
 *
 */


#ifndef NUMVEC_HH_
#define NUMVEC_HH_

#include "../util/config.hh"

#include <iostream>
#include <fstream>
#include <stdexcept>
#include <cmath>
#include <random>
#include <chrono>
#include <algorithm>
#include <functional>

template <class T> class NumVec;
template <class U> class CRSMatrix;

template <class T> void axey(const T&, const NumVec<T> &, NumVec<T> &);
template <class T> void ypaxey(NumVec<T>&, const T&, const NumVec<T>& );
template <class T> void ymaxey(NumVec<T>&, const T&, const NumVec<T>& );
template <class T> void aypxey(const T&,NumVec<T>&, const NumVec<T>&);

template <class T> class NumVec{
	template <class U> friend class CRSMatrix;

public:
	typedef T* 		iterator;
	typedef const T* 	const_iterator;
	typedef T&			reference;
	typedef const T& 	const_reference;

	NumVec(){size_ = 0; value_  = NULL;} // Default constructor
	NumVec(size_t, const T& );
	NumVec(const T*, size_t);
	NumVec(const NumVec &);		// Copy constructor
	NumVec(NumVec &&);			//	Move constructor
	~NumVec() { delete [] value_; }

	size_t size() const{return size_;}
	void resize(size_t );
	void shrink(size_t);
	void reset(const T*, size_t);
	void assign(T*, size_t);
	void assign_negative(T*, size_t);
	void assign_two_sum(T*, T*, size_t);
	void save(const char *) const;
	void load(const char *);

	T* begin() {return &value_[0];}
	const T* begin() const {return &value_[0];}
	T* end() {return &value_[size_];}
	const T* end() const {return &value_[size_];}
	NumVec<T> abs() const;
	void sort(bool);
	double l2norm() const;
	void rand(size_t);
	void randn(size_t);

	T max() const;
	T min() const;
	double mean() const;
	double var() const;
	double std() const;
	T sum() const;
	T dot(const NumVec&) const;
	void given(const size_t &, const T&, const T&);
	T* data() {return value_;}

	size_t nNeg() const;
	void sqrt();

	const NumVec & operator=(const NumVec&);
	const NumVec & operator=(const T&);
	const NumVec & operator=(NumVec && );
	const NumVec & operator+=(const NumVec&);
	const NumVec & operator-=(const NumVec&);
	const NumVec & operator*=(const NumVec&);
	const NumVec & operator/=(const NumVec&);
	const NumVec & operator+=(const T&);
	const NumVec & operator-=(const T&);
	const NumVec & operator*=(const T&);
	const NumVec & operator/=(const T&);

	T& operator[] (size_t n){ return value_[n];}
	const T& operator[] (size_t n) const{ return value_[n];}

	friend std::ostream&
	operator<<(std::ostream& os, const NumVec &rhs){
//		os << "[" << rhs.size()<< "-by-1 NUMVEC]" << std::endl;
//		for (size_t i = 0; i < rhs.size(); i++){
//			os << "[" << i << "] " << rhs[i] << std::endl;
//		}

		os << "[";
		for (size_t i = 0; i < rhs.size(); i++){
			os << rhs[i] << " ";
		}
		os << "]" << std::endl;

		return os;
	}
	friend void axey<>(const T&a, const NumVec<T> &x, NumVec<T> &y);
	friend void ypaxey<>(NumVec<T>&y, const T&a, const NumVec<T>& x);
	friend void ymaxey<>(NumVec<T>&y, const T&a, const NumVec<T>& x);
	friend void aypxey<>(const T&a,NumVec<T>&y, const NumVec<T>&x);
private:
	size_t size_;
	T *value_;
};


template <class T>
NumVec<T>::NumVec(size_t n, const T& value = 0): size_(n){
	if (size_ == 0) {
		value_ = NULL;
	} else {
		value_ = new T[size_];
		std::fill(value_,value_+size_,value);
	}
}

template <class T>
NumVec<T>::NumVec(const T* value, size_t n): size_(n){
	if (size_ == 0) {
		value_ = NULL;
	} else {
		value_ = new T[size_];

		std::copy(value,value+size_,value_);
	}
}

template <class T>
NumVec<T>::NumVec(const NumVec<T>& rhs): size_(rhs.size_){
	if (size_ == 0) {
		value_ = NULL;
	} else {
		value_ = new T[size_];
		std::copy(rhs.value_,rhs.value_+size_,value_);
	}
}

template <class T>
NumVec<T>::NumVec(NumVec<T> && rhs):size_(rhs.size_){
	value_ = rhs.value_;	// steal address
	rhs.value_ = nullptr;
	rhs.size_ = 0;

}
template <class T>
void NumVec<T>::resize(size_t size){
	if (size != size_) {
		size_ = size;
		delete [] value_;
		value_ = new T[size_];
	}
}

template <class T>
void NumVec<T>::shrink(size_t size){
	if (size < size_) {
		size_ = size;
	}
}

template <class T>
void NumVec<T>::reset(const T* val, size_t size){
	if (size != size_) {
		size_ = size;
		delete [] value_;
		value_ = new T[size_];
	}

	std::copy(val,val+size_,value_);
}

template <class T>
void NumVec<T>::assign(T* value,size_t size){
	if (size != size_) {
		size_ = size;
		delete [] value_;
		value_ = new T[size_];
	}
	std::copy(value,value+size_,value_);
}

template <class T>
void NumVec<T>::assign_negative(T* value,size_t size){
	if (size != size_) {
		size_ = size;
		delete [] value_;
		value_ = new T[size_];
	}
	for (size_t i = 0; i < size_; i++){
		value_[i] = -value[i];
	}
}

template <class T>
void NumVec<T>::assign_two_sum(T* value1, T* value2,size_t size){
	if (size != size_) {
		size_ = size;
		delete [] value_;
		value_ = new T[size_];
	}
	for (size_t i = 0; i < size_; i++){
		value_[i] = value1[i] + value2[i];
	}
}

template <class T>
void NumVec<T>::save(const char *filename) const{
	std::ofstream save;
	save.open(filename,std::ios::out);
	save << size_ << std::endl;
	save.precision(16);
	size_t size = this->size();
	for (size_t i = 0; i < size; i++)
		save << value_[i] << std::endl;
	save.close();
}

template <class T>
void NumVec<T>::load(const char *filename){
	size_t size;
	std::ifstream load;
	load.open(filename,std::ios::in);
	if (!load){
		std::cerr << "\n" ;
		std::cerr << "LOAD - Fatal error!\n";
		std::cerr << " Could not open the file: \"" << filename << "\"\n";
		abort();
	}
	load >> size_;
	load.precision(16);

	for (size_t i = 0; i < size; i++)
		load >> value_[i];
	load.close();
}

template <class T>
const NumVec<T>& NumVec<T>::operator=(const NumVec<T> &rhs){
	if (this != &rhs){
		if (rhs.size_ == 0){
			value_ = NULL;
			size_ = 0;
		} else {
			if (rhs.size_ != size_ ){
				size_ = rhs.size_;
				delete [] value_;
				value_ = new T[size_];
			}
			std::copy(rhs.value_,rhs.value_+size_,value_);
		}
	}
	return *this;
}


template <class T>
const NumVec<T>& NumVec<T>::operator=(const T& value){
	if (size_ == 0) {
		value_ = NULL;
	} else {
		std::fill(value_,value_+size_,value);
	}
	return *this;
}

template <class T>
const NumVec<T>& NumVec<T>::operator=(NumVec<T> && rhs)
{
	if (this == &rhs){
		return *this;
	}
	delete [] value_;
	size_ = rhs.size_;
	value_ = rhs.value_;
	rhs.size_ = 0;
	rhs.value_ = nullptr;
	return *this;
}

template <class T>
const NumVec<T> & NumVec<T>::operator+=(const NumVec<T> &rhs){
	if (size_ != rhs.size()){
		throw std::out_of_range("Vector size must agree (+=).");
	}

	T* vp = begin();
	const T*rvp = rhs.begin();
	size_t i;

#ifdef USE_OPENMP
#pragma omp parallel private (i) shared (vp,rvp)
{
#pragma omp for schedule(guided)
	for (size_t i = 0; i < size_; i++){
		*(vp+i) += *(rvp+i);//rhs.value_[i];
	}
}
#else
	for (i = 0; i < size_; i++){
		*(vp+i) += *(rvp+i);//rhs.value_[i];
	}
#endif
	return *this;
}

template <class T>
const NumVec<T> & NumVec<T>::operator-=(const NumVec<T> &rhs){
	if (size_ != rhs.size()){
		throw std::out_of_range("Vector size must agree (-=).");
	}

	T* vp = begin();
	const T*rvp = rhs.begin();
	size_t i;

#ifdef USE_OPENMP
#pragma omp parallel private (i) shared (vp,rvp)
{
#pragma omp for schedule(guided)
	for (i = 0; i < size_; i++){
		*(vp+i) -= *(rvp+i);//rhs.value_[i];
	}
}
#else
	for (i = 0; i < size_; i++){
		*(vp+i) -= *(rvp+i);//rhs.value_[i];
	}
#endif
	return *this;
}

template <class T>
const NumVec<T> & NumVec<T>::operator*=(const NumVec<T> &rhs){
	if (size_ != rhs.size()){
		throw std::out_of_range("Vector size must agree (*=).");
	}
	T* vp = begin();
	const T*rvp = rhs.begin();
	size_t i;

#ifdef USE_OPENMP
#pragma omp parallel private (i) shared (vp,rvp)
{
#pragma omp for schedule(guided)
	for (i = 0; i < size_; i++){
		*(vp+i) *= *(rvp+i);//rhs.value_[i];
	}
}
#else
	for (i = 0; i < size_; i++){
		*(vp+i) *= *(rvp+i);//rhs.value_[i];
	}
#endif

	return *this;

}

template <class T>
const NumVec<T> & NumVec<T>::operator/=(const NumVec<T> &rhs){
	if (this->size() != rhs.size()){
		throw std::out_of_range("Vector size must agree (/=).");
	}
	for (size_t i = 0; i < size_; i++){
		if (rhs.value_[i] == 0)
			throw std::out_of_range("Can't divide by 0");
		value_[i] /= rhs.value_[i];
	}
	return *this;
}

template <class T>
const NumVec<T> & NumVec<T>::operator+=(const T &val){

	for (size_t i = 0; i < size_; i++){
		value_[i] += val;
	}
	return *this;
}

template <class T>
const NumVec<T> & NumVec<T>::operator-=(const T &val){

	for (size_t i = 0; i < size_; i++){
		value_[i] -= val;
	}
	return *this;
}

template <class T>
const NumVec<T> & NumVec<T>::operator*=(const T &val){
	for (size_t i = 0; i < size_; i++){
		value_[i] *= val;
	}
	return *this;
}

template <class T>
const NumVec<T> & NumVec<T>::operator/=(const T &val){
	if (val == 0)
		throw std::out_of_range("Can't divide by 0");

	for (size_t i = 0; i < size_; i++){
		value_[i] /= val;
	}
	return *this;
}



template<class T>
const NumVec<T> operator+(const NumVec<T> &v1, const NumVec<T> &v2){
	return NumVec<T>(v1)+=v2;
}

template<class T>
const NumVec<T> operator-(const NumVec<T> &v1, const NumVec<T> &v2){
	return NumVec<T>(v1)-=v2;
}

template<class T>
const NumVec<T> operator*(const NumVec<T> &v1, const NumVec<T> &v2){
	return NumVec<T>(v1)*=v2;
}

template<class T>
const NumVec<T> operator/(const NumVec<T> &v1, const NumVec<T> &v2){
	return NumVec<T>(v1)/=v2;
}

template<class T>
const NumVec<T> operator+(const NumVec<T> &v1, const T &val){
	return NumVec<T>(v1)+=val;
}

template<class T>
const NumVec<T> operator+(const T &val, const NumVec<T> &v1){
	return NumVec<T>(v1)+=val;
}

template<class T>
const NumVec<T> operator-(const NumVec<T> &v1, const T &val){
	return NumVec<T>(v1)-=val;
}

template<class T>
const NumVec<T> operator-(const T &val, const NumVec<T> &v1){
	return -(NumVec<T>(v1)-=val);
}

template<class T>
const NumVec<T> operator*(const NumVec<T> &v1, const T &val){
	return NumVec<T>(v1)*=val;
}

template<class T>
const NumVec<T> operator*(const T &val, const NumVec<T> &v1){
	return NumVec<T>(v1)*=val;
}

template<class T>
const NumVec<T> operator/(const NumVec<T> &v1, const T &val){
	return NumVec<T>(v1)/=val;
}

template<class T>
const NumVec<T> operator/(const T &val,const NumVec<T> &v1){
	return reciprocal(v1)*=val;
}

template<class T>
const NumVec<T> reciprocal(const NumVec<T> &v1){
	NumVec<T> recp(v1);
	for (size_t i = 0; i < recp.size(); i++){
		recp[i] = (T)1/recp[i];
	}
	return recp;
}

template <class T>
T NumVec<T>::sum() const{
	T sum = T();
	for (size_t i = 0; i < size_; i++) sum+=value_[i];
	return sum;
}

template <class T>
NumVec<T> NumVec<T>::abs() const{
	NumVec<T> temp(*this);
	for (size_t i = 0; i < size_; i++){
		if (value_[i] < 0)
			temp[i] *= (T)-1;
	}
	return temp;
}
template <class T>
void NumVec<T>::sort(bool isDescending= false){
	std::vector<T> tmp(value_,value_+size_);
	std::sort(tmp.begin(),tmp.end());
	if (isDescending){
		for (size_t i = 0; i < size_; i++){
			value_[size_ - i - 1] = tmp[i];
		}

	} else {
		for (size_t i = 0; i < size_; i++){
			value_[i] = tmp[i];
		}
	}
}

template <class T>
double NumVec<T>::l2norm() const{
	NumVec<T> tmp(*this);
	tmp *= *this;
	return std::sqrt((double)tmp.sum());
}

template <class T>
void NumVec<T>::rand(size_t n = 0){
	if (n > 0 && n!= size_){
		delete [] value_;
		size_ = n;
		value_ = new T[size_];
	}
#ifdef RAND_SEED
	unsigned seed = std::chrono::system_clock::now().time_since_epoch().count();
	std::default_random_engine generator (seed);
#else
	std::default_random_engine generator;
#endif
	std::uniform_real_distribution<T> distribution(0.0,1.0);
	for (size_t i = 0; i < size_; i++){
		value_[i] = (T)distribution(generator);
	}
}

template <class T>
void NumVec<T>::randn(size_t n = 0){
	if (n > 0 && n!= size_){
		delete [] value_;
		size_ = n;
		value_ = new T[size_];
	}
#ifdef RAND_SEED
	unsigned seed = std::chrono::system_clock::now().time_since_epoch().count();
	std::default_random_engine generator (seed);
#else
	std::default_random_engine generator;
#endif
	std::normal_distribution<T> distribution(0.0,1.0);
	size_t size = this->size();
	for (size_t i = 0; i < size; i++){
		this->operator [](i) = (T)distribution(generator);
	}
}

template <class T>
T NumVec<T>::max() const{
	return *std::max_element(value_,value_+size_);
}



template <class T>
T NumVec<T>::min() const{
	return *std::min_element(value_,value_+size_);
}

template <class T>
double NumVec<T>::mean() const{
	return (double)this->sum()/(double)this->size();
}

template <class T>
double NumVec<T>::var() const{
	NumVec<T> tmp(*this);
	tmp*= *this;
	double mval = this->mean();
	return double(size_)/double(size_-1)*(tmp.mean() - mval * mval);
}

template <class T>
double NumVec<T>::std() const{
	return std::sqrt(this->var());
}

template<class T>
T NumVec<T>::dot(const NumVec<T> &v) const{
	T dproduct = T();
	for (size_t i = 0; i < size_; i++)
		dproduct += value_[i] * v.value_[i];
	return dproduct;
}

template<class T>
T operator,(const NumVec<T> &v1, const NumVec<T> &v2){
	return v1.dot(v2);
}

template<class T>
void NumVec<T>::given(const size_t &i, const T& c, const T& s){
	T g1, g2;
	g1 = c*value_[i] - s*value_[i+1];
	g2 = s*value_[i] + c*value_[i+1];

	value_[i] = g1;
	value_[i+1] = g2;
}

template<class T>
size_t NumVec<T>::nNeg() const{
	size_t nneg = 0;
	for (size_t i = 0; i < size_; i++){
		if (value_[i] < (T)0){
			nneg++;
		}
	}
	return nneg;
}

template <class T>
void NumVec<T>::sqrt() {
	for (size_t i = 0; i < size_; i++){
		value_[i] = std::sqrt(value_[i]);
	}
}
template<class T>
const NumVec<T> exp(const NumVec<T> & v){
	NumVec<T> rv(v);
	for(size_t i = 0; i < rv.size(); i++){
		rv[i] = (T)exp(rv[i]);
	}
	return rv;
}

template<class T>
const NumVec<T> pow(const T& base, const NumVec<T> & v){
	NumVec<T> rv(v);
	for(size_t i = 0; i < rv.size(); i++){
		rv[i] = (T)pow(base, rv[i]);
	}
	return rv;
}
template <class T>
void axey(const T&a, const NumVec<T> &x, NumVec<T> &y){
	size_t n = x.size();
	for (size_t i = 0; i < n; i++){
		*(y.value_+i) = a* *(x.value_+i);
	}
}

template <class T>
void ypaxey(NumVec<T>&y, const T&a, const NumVec<T>& x){
	size_t n = x.size();
	for (size_t i = 0; i < n; i++){
		*(y.value_+i) += a* *(x.value_+i);
	}
}

template <class T>
void ymaxey(NumVec<T>&y, const T&a, const NumVec<T>& x){
	size_t n = x.size();
	for (size_t i = 0; i < n; i++){
		*(y.value_+i) -= a* *(x.value_+i);
	}
}

template <class T>
void aypxey(const T&a,NumVec<T>&y, const NumVec<T>&x){
	size_t n = x.size();
	for (size_t i = 0; i < n; i++){
		*(y.value_+i) = a* *(y.value_+i) +*(x.value_+i);
	}
}
#endif /* NUMVEC_HH_ */
