/*
 * amg.hh
 *
 *  Created on: 2013. 8. 8.
 *      Author: parkmh
 */

#ifndef AMG_HH_
#define AMG_HH_

#include <cmath>
#include <iomanip>
#include "../util/config.hh"

template <class Category, typename T>
class AMG{
public:
	typedef amg_traits<Category,T> 			traits;
	typedef typename traits::datatype 		datatype;
	typedef typename traits::matrix 		matrix;
	typedef typename traits::coarsen 		coarsen;
	typedef typename traits::fcoarsen 		fcoarsen;
	typedef typename traits::prolong 		prolong;
	typedef typename traits::fprolong 		fprolong;
	typedef typename traits::shorten 		shorten;
	typedef typename traits::smooth1 		smooth1;
	typedef typename traits::smooth2   	smooth2;
	typedef typename traits::cstsolver	 	cstsolver;
	AMG();
	AMG(const matrix& , size_t , size_t, double  );
	~AMG();
	void init(const matrix& , size_t , size_t, double );
	void second_init(const matrix& , size_t , size_t, double );
	void solve(const NumVec<datatype>&, NumVec<datatype> &, const bool) const;
	void vcycle(const NumVec<datatype>&, NumVec<datatype> &, const size_t) const;
	double residual( const NumVec<datatype> &rhs, const NumVec<datatype> &x) const;
	void overview(std::ostream&) const;
	void convergence(std::ostream&) const;
	friend std::ostream&
	operator<<(std::ostream& os, const AMG &amg){
		amg.overview(os);
		amg.convergence(os);
		return os;
	}
private:
	const matrix 			*A_;
	size_t 				coarsest_;
	cstsolver 				cstsolver_;
	size_t 				mu_;
	double 				tol_;
	size_t 				ncycle_;
	double* 				Residual_;
	mutable size_t 		lastcycle_;
	mutable bool 			solved_;
	std::vector<Relaxation<matrix>* > 		s1vec;
	std::vector<Relaxation<matrix>* > 		s2vec;
	std::vector<CSN<datatype>* > 			csnvec;
	std::vector<INTP<matrix> *>    			intpvec;
	std::vector<Restriction<matrix> *> 	rstrvec;
	matrix * 									Ac;

	void delete_pointer_vector();
};
template <class Category, typename T>
AMG<Category, T>::AMG(){
	A_ = NULL;
	mu_ = 0;
	ncycle_ = 0;
	lastcycle_ = ncycle_;

	Residual_ = NULL;
	tol_ = 0;
	coarsest_= 0;
	solved_ = false;
	Ac 		= NULL;
}

template <class Category, typename T>
AMG<Category, T>::AMG(const matrix& A,size_t coarsest=10, size_t ncycle = 20,double tol = 1e-10){
	Residual_ = NULL;
	Ac = NULL;
	init(A,coarsest,ncycle,tol);
}

template <class Category, typename T>
AMG<Category, T>::~AMG(){
	//	delete [] amgbase_;
	delete [] Residual_;
	delete [] Ac;
	for(auto p = s1vec.begin(); p!= s1vec.end(); p++){
		delete *p;
		*p = NULL;
	}
	for(auto p = s2vec.begin(); p!= s2vec.end(); p++){
		delete *p;
		*p = NULL;
	}

	for(auto p = csnvec.begin(); p!= csnvec.end(); p++){
		delete *p;
		*p = NULL;
	}
	for(auto p = intpvec.begin(); p!= intpvec.end(); p++){
		delete *p;
		*p = NULL;
	}
	for(auto p = rstrvec.begin(); p!= rstrvec.end(); p++){
		delete *p;
		*p = NULL;
	}

}

template <class Category, typename T>
void AMG<Category,T>::init(const matrix& A,size_t coarsest=10, size_t ncycle = 20,double tol = 1e-10){
	A_ = &A;
	mu_ = traits::mu;
	ncycle_ = ncycle;
	lastcycle_ = ncycle_;

	delete [] Residual_;
	Residual_ = new double [ncycle+1];
	for (size_t i = 0; i < ncycle+1; i++){
		Residual_[i] = 0.0;
	}

	tol_ = tol;
	// Define interpolation, restriction and coarse grid operators on each level
	coarsest_=coarsest;

	delete []  Ac;
	Ac = new matrix[coarsest_];
	solved_ = false;

	csnvec.push_back(new fcoarsen(A));
	intpvec.push_back(new fprolong(A,*csnvec[0]));
	rstrvec.push_back(new shorten(A,*intpvec[0]));
	Ac[0] = *rstrvec[0]->operator()();
	Ac[0] *=(*A_);
	Ac[0] *= (*intpvec[0]->operator()());

	if (Ac[0].row() < CSTMAXROW ){
		coarsest_ = +2;
	}
	for (size_t level = 1; level < coarsest_-1; level++){
		csnvec.push_back(new coarsen(Ac[level-1]));
		intpvec.push_back(new prolong(Ac[level-1],*csnvec[level]));
		rstrvec.push_back(new shorten(Ac[level-1],*intpvec[level]));
		Ac[level] = *rstrvec[level]->operator()();
		Ac[level] *= Ac[level-1];
		Ac[level] *= (*intpvec[level]->operator()());

		if (Ac[level].row() < CSTMAXROW ){
			coarsest_ = level+2;
			break;
		}
	}

	// Define pre and post smoothers on each level but the coarsest level.
	s1vec.push_back(new smooth1());
	s2vec.push_back(new smooth2());
	s1vec[0]->init(A,traits::r1);
	s2vec[0]->init(A,traits::r2);

	for (size_t level = 1; level< coarsest_-1; level++){
		s1vec.push_back(new smooth1());
		s2vec.push_back(new smooth2());
		s1vec[level]->init(Ac[level-1],traits::r1);
		s2vec[level]->init(Ac[level-1],traits::r2);
	}

	// Define the solver for the coarsest level
	cstsolver_.init(Ac[coarsest_-2]);
}

template <class Category, typename T>
void AMG<Category,T>::second_init(const matrix& A,size_t coarsest=10, size_t ncycle = 20,double tol = 1e-10){
	A_ = &A;
	lastcycle_ = ncycle_;

	delete [] Residual_;
	Residual_ = new double [ncycle+1];
	for (int i = 0; i < ncycle+1; i++){
		Residual_[i] = 0.0;
	}

	tol_ = tol;
	// Define interpolation, restriction and coarse grid operators on each level

	delete[] Ac;
	Ac = new matrix[coarsest_];
	solved_ = false;

	//TODO FIX second_init
	intpvec[0]->init(A,*csnvec[0]);
	rstrvec[0]->init(A,*intpvec[0]);

	Ac[0] = *rstrvec[0]->operator()();
	Ac[0] *=(*A_);
	Ac[0] *= (*intpvec[0]->operator()());

	for (size_t level = 1; level < coarsest_-1; level++){

		intpvec[level]->init(Ac[level-1],*csnvec[level]);
		rstrvec[level]->init(Ac[level-1],*intpvec[level]);
		Ac[level] = *rstrvec[level]->operator()();
		Ac[level] *= Ac[level-1];
		Ac[level] *= (*intpvec[level]->operator()());
	}

	// Define pre and post smoothers on each level but the coarsest level.
	s1vec[0]->init(A,traits::r1);
	s2vec[0]->init(A,traits::r2);

	for (size_t level = 1; level< coarsest_-1; level++){
		s1vec[level]->init(Ac[level-1],traits::r1);
		s2vec[level]->init(Ac[level-1],traits::r2);
	}

	// Define the solver for the coarsest level
	cstsolver_.init(Ac[coarsest_-2]);
}

template <class Category, typename T>
void AMG<Category,T>::solve(const NumVec<datatype>& f, NumVec<datatype> &v, const bool isPCOND = true) const{
	if (isPCOND){
		for (size_t i = 0; i <= ncycle_; i++){
			vcycle(f,v,1);
		}
		lastcycle_ = ncycle_;
	} else{
		lastcycle_ = ncycle_;
		NumVec<T> r; r.resize(f.size());
		bmaxey<T>(f,*A_,v,r);
		Residual_[0] = r.l2norm();
		for (size_t i = 0; i <= ncycle_; i++){
			vcycle(f,v,1);
			bmaxey<T>(f,*A_,v,r);
			Residual_[i] = r.l2norm();
			if (Residual_[ i ] / Residual_[ 0 ] < tol_) {
				lastcycle_ = i;
				break;
			}
		}
	}
	solved_ = true;
}

template <class Category, typename T>
void AMG<Category,T>::vcycle(const NumVec<datatype>& f, NumVec<datatype> &v, const size_t level = 1) const{
	if (level == coarsest_){
		cstsolver_.solve(f,v);
	}
	else {
		s1vec[level-1]->solve(f,v);
		NumVec<datatype> f2h;

		NumVec<datatype> r; r.resize(v.size());
		if (level == 1){
			bmaxey<datatype>(f,*A_,v,r);
		}
		else {
			bmaxey<datatype>(f,Ac[level-2],v,r);
		}
		rstrvec[level-1]->shorten(f2h,r);

//		if (level == 1){
//			rstrvec[level-1]->shorten(f2h,f-(*A_)*v);
//		}
//		else {
//			rstrvec[level-1]->shorten(f2h,f-Ac[level-2]*v);
//		}

		NumVec<datatype> v2h(f2h.size());

		if (level < coarsest_ - 1){
			for (size_t i = 0; i < mu_; i++){
				vcycle(f2h,v2h,level+1);
			}
		} else {
			vcycle(f2h,v2h,level+1);
		}
		intpvec[level-1]->correct(v2h,v);
		s2vec[level-1]->solve(f,v);
	}
}

template <class Category,class T>
double AMG<Category,T>::residual(const NumVec<datatype> & rhs, const NumVec<datatype> & x) const{
	NumVec<datatype> r(rhs);
	r -= (*A_)*x;
	return (rhs-*(A_)*x).l2norm();
}

template <class Category, class T>
void AMG<Category,T>::overview(std::ostream & os) const{
	int textw = 20;
	for (size_t level = 1; level < coarsest_; level++){
		os << "Level[" << level << "]" << std::endl;
		os << std::setw(textw) << std::left<<"Coarsening"<< std::right<< ": " << *csnvec[level-1] << std::endl;
		os << std::setw(textw) << std::left<<"Interpolation"<< std::right<< ": " << *intpvec[level-1] <<std::endl;
		os << std::setw(textw) << std::left<<"Restriction"<< std::right<< ": " << *rstrvec[level-1] << std::endl;
		os << std::setw(textw) << std::left<<"Pre-Smoother"<< std::right<< ": " << *s1vec[level-1]<< std::endl;
		os << std::setw(textw) << std::left<<"Post-Smoother"<< std::right<< ": " << *s2vec[level-1] << std::endl;
		os << std::endl;
	}
	os << "Level[" << coarsest_ << "]" << std::endl;
	os << std::setw(textw) << std::left<<"Solver"<< std::right<< ": " << cstsolver_ <<std::endl;

	os << std::endl;
	os << std::setw(textw) << std::left<<"cycle_index (mu)" << std::right<< ": " << mu_ << std::endl;
	os << std::endl;

	int levelw = 5;
	int nroww = std::max((int)ceil(log10((double)A_->row()))+1,10);
	int nnzw = std::max((int)ceil(log10((double)A_->nnz()))+1,10);
	int densityw = 12;

	os << std::setw(levelw) << "Level" << std::setw(nroww) << "nrow"
			<< std::setw(nnzw) << "nnz" << std::setw(densityw) << "density" << std::endl;
	os << std::setfill('-') << std::setw(levelw+nroww+nnzw+densityw+1)<< "" << std::endl;
	os << std::setfill(' ');
	os << std::setw(levelw) << "1" << std::setw(nroww) << A_->row()
						<< std::setw(nnzw) << A_->nnz()
						<< std::setw(densityw)
	<< (double)A_->nnz()/(double)A_->row()/(double)A_->row()<<std::endl;
	int totalnrow = A_->row();
	int totalnnz = A_->nnz();

	for (size_t level = 0; level < coarsest_-1; level++){
		os << std::setw(levelw) << level + 2 << std::setw(nroww) << Ac[level].row()
								<< std::setw(nnzw) << Ac[level].nnz()
								<< std::setw(densityw)
		<< (double)Ac[level].nnz()/(double)Ac[level].row()/(double)Ac[level].row()<<std::endl;
		totalnrow += (double)Ac[level].row();
		totalnnz += (double)Ac[level].nnz();
	}
	os.precision(3);
	os << "Operator complexity : " << (double)totalnnz/(double)A_->nnz() <<std::endl;
	os << "Grid Complexity     : " << (double)totalnrow/(double)A_->row() << std::endl;
	os.precision();
	os << std::endl;
}

template <class Category, class T>
void AMG<Category,T>::convergence(std::ostream & os) const{
	if (solved_){
		int cyclew = 5;
		int residualw = 12;
		int ratiow   = 10;
		os << std::setw(cyclew) << "cycle" << std::setw(residualw) << "residual"
				<< std::setw(ratiow) << "rate" << std::endl;
		os << std::setw(cyclew) << 0 << std::setw(residualw) << Residual_[0]<< std::endl;


		for (size_t i = 1; i <= lastcycle_; i++){
			os << std::setw(cyclew) << i;
			os.precision(2);
			os	<< std::setw(residualw) << std::scientific << Residual_[i];
			os.precision(2);
			os	<< std::setw(ratiow) << std::fixed << Residual_[i]/Residual_[i-1] << std::endl;
			os.precision();

		}
	}
}

#endif /* AMG_HH_ */
