# Script for reproducing orientation data example

require(orientlib)
require(scatterplot3d)
source("funcs.R")

# Load the data
load("handquat.rda")

## mean of data
qm <- quaternion(mean(mquat))
## center data
cmquat <- quatmult(quatinv(qm),mquat)

## form the distance matrix
n <- nrow(mquat@x)
distm <- matrix(0,n,n)
for(i in 2:n){
  for(j in 1:(i-1)){
    dd <- rotation.distance(mquat[i,],mquat[j,])
    distm[i,j] <- dd
    distm[j,i] <- dd
  }
}
diag(distm) <- 0

# The maximum distance between two points (in degrees)
 max(distm)*180/pi

## PCA on tangent space

lquat <- logmap(cmquat)
lpca <- prcomp(lquat)
## proportion of variance explained
vv <- lpca$sdev^2
vv/sum(vv)

## Classical MDS

qcmds <- cmdscale(distm,k=2,eig=TRUE,x.ret=TRUE)

               ## very little difference between PCA and PCoA
max(abs(qcmds$points[,2]-lpca$x[,2]))
max(abs(qcmds$points[,1]-lpca$x[,1]))

# Compute the quaternions corresponding to +/- 1 PC1
set.seed(123)
ssu <- backscoreq(mquat,qm,c(1,0),qcmds,100)
ssd <- backscoreq(mquat,qm,c(-1,0),qcmds,100)

# distances between the randomly generated scores and the desired score
ddu <- sqrt((ssu$actscore[,1]-1)^2+ssu$actscore[,2]^2)
ddd <- sqrt((ssd$actscore[,1]+1)^2+ssd$actscore[,2]^2)

# consider only close solutions for plotting purposes
comres <- rbind(ssu$pcorb[ddu < 0.01,],ssd$pcorb[ddd < 0.01,])

# produce Figure 1
pdf(file="orient3d.pdf",height=4,width=6)
par(mfrow=c(1,2))
p3d <- scatterplot3d(lquat,angle=135,box=FALSE,xlab="",ylab="",zlab="",
                     xlim=c(-0.4,0.4),ylim=c(-0.4,0.4),zlim=c(-0.4,0.4),mar=rep(0.5,4))

p3d <- scatterplot3d(comres,angle=135,box=FALSE,xlab="",ylab="",zlab="",pch=".",
                     xlim=c(-0.4,0.4),ylim=c(-0.4,0.4),zlim=c(-0.4,0.4),mar=rep(0.5,4))
p3d$points3d(0,0,0,pch=16)

# pick out the best solution
x <- ssu$pcorb[ddu < 0.01,]
y <- ssu$rotang[ddu < 0.01]
bs <- x[which.min(y),]
p3d$points3d(bs[1],bs[2],bs[3],pch=1,angle=50)

# pick out the best solution
x <- ssd$pcorb[ddd < 0.01,]
y <- ssd$rotang[ddd < 0.01]
bs <- x[which.min(y),]
p3d$points3d(bs[1],bs[2],bs[3],pch=1,angle=50)
dev.off()
