%% Numerical Simulations of Stochastic ODEs for the model GRN problem.
%% Copyright Huy Vo, Jonathan Dawes, and Robert Kelsh, 2022 - 2024.
%% Code supplied 'as is'.
%% Makes use of the sde_euler() routine written by Andrew D. Horchler.
%% Available from: https://github.com/horchler/SDETools
%% Version 1.2 used here (9-21-13).

%% This code:
% - generates a sample trajectory of the stochastic ODE
% - permutes the coordinates to generate symmetric copies of that
%     trajectory (starting from the initial condition that is a 
%     symmetric copy)
% - samples 400 points uniformly at random along each trajectory
% - saves the set of 1200 points in a data file.

% Note that this code splits the generation of simulation data into two
% halves: trials 1-500 and trials 501-1000. These use different seeds.
% Lines 29/30, 122/123 and 133 need to be adjusted between these two sets
% so that output filenames and screen outputs are correct (nn, or nn+500).

% Workflow:
% After running this code, run count_lineages.R to use the output data
% as synthetic data for slingshot routines.

% Then, analyse those results using analyse_lineages.m .

close all; clear all; % clear Matlab workspace and plot windows.

%opts = sdeset('RandSeed',1); % Set random seed - for reproducibility for trials 1 to 500
opts = sdeset('RandSeed',5); % Set random seed - for reproducibility for trials 501 to 1000.

N=400; % number of sample points.

%% Parameters in the differential equations
%sigma_arr=10.^[-4:-3];     % array of sigma values. 
sigma_arr=10.^[-4];         % just do one value of sigma to start with. 
alpha_arr=10.^[-1:0.25:3];  % array of alpha values.

for kk=1:length(sigma_arr)  % repeat over values of sigma - noise level
for jj=1:length(alpha_arr)  % repeat over values of alpha - ODE parameter (amount of spiralling)
    for nn=1:500          % repeat over 500 trials for each pair (alpha, sigma)
        
alpha1=alpha_arr(jj); alpha2=alpha1; alpha3=alpha1; % parameters in SDEs
beta1=0.1; beta2=beta1;  beta3=beta1;               % parameters in SDEs

b1=1e-3; b2=b1; b3=b1;  % taking b_i>0 is important for the SDE simulations
                        % but has very little impact on the ODE simulations.
d1=0.01; d2=d1; d3=d1;  % degradation rates for TFs.
h1=3; h2=3; h3=3;       % Hill function exponents, called h in the paper.

sigma=sigma_arr(kk);     % diffusion coefficient.

%% Numerical parameters
dt = 1e-0;                  % time step
tmax=2000;                  % max time to integrate for.
t = 0:dt:tmax;              % time vector
x0=[0.097; 0.09; 0.1];      % initial condition
%x0=circshift(x0,1);        % to produce circularly-shifted trajectories.
%x0=circshift(x0,2);        % to produce circularly-shifted trajectories.



% Drift and diffusion terms

% drift function
f = @(t,x)[b1 + (4*t/(tmax+t))/((1+alpha1*x(2)^h1)*(1+beta1*x(3)^h1)) - d1*x(1);
b2 + (4*t/(tmax+t))/((1+alpha2*x(3)^h2)*(1+beta2*x(1)^h2)) - d2*x(2);
b3 + (4*t/(tmax+t))/((1+alpha3*x(1)^h3)*(1+beta3*x(2)^h3)) - d3*x(3)];

diffusion = @(t,x)sigma*sqrt(x.*(x>0)); % state-dependent diffusion function.
                                        % note the sqrt(x) dependence. The
                                        % x>0 condition is there to avoid
                                        % complex numbers.
% Time integration
y = sde_euler(f,diffusion,t,x0,opts);   % Carry out the integration

% Display results (if required), e.g. to generate Figures 2, 3 or 9 in the paper. 
%ymax=max(max(log(y)));
% 
%figure(1);
% f=figure(1);
% f.Position=[100 100 800 400]; % [left bottom width height] % for 2 plots
% %f.Position=[100 100 800 200]; % [left bottom width height]  % for 1 plot
% subplot(2,1,1);
% plot(t,log(y),'-','LineWidth',2);    % Plot y-axis on a log scale.
% set(gca,'FontSize',14);             % make axis labels larger.
% legend('$x_1$','$x_2$','$x_3$','Location','NorthWest','Interpreter','latex','NumColumns',3);
% xlabel('time (a. u.)','fontsize',14,'Interpreter','latex');
% ylabel('log (TF expr. level)','fontsize',12,'Interpreter','latex');
% axis([0 tmax -1 ymax])
% 
% subplot(2,1,2);
% g=4*t./(tmax+t); gmax=max(g);    % gmax should therefore be the value 2.
% plot(t,g,'-','LineWidth',2);
% set(gca,'FontSize',14);            % make axis labels larger.
% xlabel('time (a. u.)','fontsize',14,'Interpreter','latex');
% ylabel('$g(t)$ (a. u.)','fontsize',14,'Interpreter','latex');
% axis([0 tmax 0 gmax])



% Generate 2 symmetrically-related copies of the SDE data.
y2=circshift(y,[0,2]);
y3=circshift(y,[0,1]);

y=[t' y];  % add time points to the y_i data.
y2=[t' y2];
y3=[t' y3];

% Sample N=400 points from each of {y,y2,y3} to simulate a set of 3N cells, 
% with N of each choosing one of the three cell fates.

random_times=randsample(1:length(t),N);  %Selects N times at random from the time value vector.
y1_rand=y(random_times,1:4);   % choose these y values.
random_times=randsample(1:length(t),N);  %Selects N times at random from the time value vector.
y2_rand=y2(random_times,1:4);   % choose these y values.
random_times=randsample(1:length(t),N);  %Selects N times at random from the time value vector.
y3_rand=y3(random_times,1:4);   % choose these y values.

A=[y1_rand; y2_rand; y3_rand]; % combine into one matrix that is 3N x 4.
% Write this matrix to a csv file.
filename=strcat(".\sde_data\data_a", ...
    num2str(alpha1),"s",num2str(sigma),"n",num2str(nn+500),".csv");
writematrix(A,filename)

% Now compute barycentric coordinates. This version has one eqm pt at (1,0).
%x1=(-y(:,2)-y(:,3)+2*y(:,1))/sqrt(6);
%x2=(y(:,2)-y(:,3))/sqrt(2);
%x_data=[x1 x2];  % data to fit to.

end
end
fprintf('Simulation: sigma=%g; alpha=%g; n=%g \n',[sigma,alpha1,nn+500]);
end


